﻿/*
	Name:			XenoN Core Header
	Version:		1.0.1.0 Dark Issue
	Update:			2011-04-24
	Copyright:		Copyright © 2007-2011 XenoN Core by PsichiX. All rights reserved.
	Author:			PsichiX
	Website:		http://www.xenon.psichix.com
	Description:	XenoN Core 4D World Engine Header
*/

/*
	== EN ==
	This file is part of Xenon Core Engine.
	You may distribute it further, but you can not modify it.
	Please do not use in modified form.
	The principles of XenoN Core License available in the file LICENSE_CORE_EN.TXT or visit: http://www.xenon.psichix.com.

	== PL ==
	Ten plik jest czescia XenoN Core Engine.
	Mozesz go rozpowszechniac dalej, jednak nie mozesz go modyfikowac.
	Nalezy uzytkowac w nie zmodyfikowanej formie.
	Nalezy przestrzegac zasad Licencji XenoN Core dostepnej w pliku LICENSE_CORE_PL.TXT oraz na stronie: http://www.xenon.psichix.com.
*/

/*
	MODULES:
		Photon		- Graphics
		Chaos		- Physics
		Psyche		- Artificial Intelligence
		TimeBridge	- Time Control
		Echo		- Audio
		Ether		- Network
*/

#ifndef _XENON_CORE_H_
#define _XENON_CORE_H_

#include "XenonCoreOptions.h"
#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

/*!
\namespace XeCore				Przestrzen nazw XenoN Core.
\namespace XeCore::Photon		Przestrzen nazw modulu Photon
\namespace XeCore::Chaos		Przestrzen nazw modulu Chaos
\namespace XeCore::Psyche		Przestrzen nazw modulu Psyche
\namespace XeCore::TimeBridge	Przestrzen nazw modulu TimeBridge
\namespace XeCore::Ether		Przestrzen nazw modulu Ether
\namespace XeCore::Echo			Przestrzen nazw modulu Echo
*/

namespace XeCore
{

/*--- Mode ---*/
#ifdef XE_USE_STATIC_DEV
#define XE_COMPILE_STATIC
#endif /* XE_USE_STATIC_DEV */
#ifndef XE_COMPILE_EXPORT
#ifndef XE_NOT_USE_LIBRARY
#ifndef XE_COMPILE_STATIC
#pragma comment(lib,"XeCore/XenonCore.lib")
#else /* XE_COMPILE_STATIC */
#ifndef _DLL
#ifndef _DEBUG
#pragma comment(lib,"XeCore/XenonCoreMT.lib")
#else /* _DEBUG */
#pragma comment(lib,"XeCore/XenonCoreMTd.lib")
#endif /* _DEBUG */
#else /* _DLL */
#ifndef _DEBUG
#pragma comment(lib,"XeCore/XenonCoreMD.lib")
#else /* _DEBUG */
#pragma comment(lib,"XeCore/XenonCoreMDd.lib")
#endif /* _DEBUG */
#endif /* _DLL */
#endif /* XE_COMPILE_STATIC */
#endif /* XE_NOT_USE_LIBRARY */
#endif /* XE_COMPILE_EXPORT */
#ifdef XE_COMPILE_STATIC
#define XE_CALLTYPE _cdecl
#define XE_IMPEX
#else /* XE_COMPILE_STATIC */
#define XE_CALLTYPE __stdcall
#ifdef XE_COMPILE_EXPORT
#define XE_IMPEX extern "C" __declspec(dllexport)
#else
#define XE_IMPEX extern "C" __declspec(dllimport)
#endif /* XE_COMPILE_EXPORT */
#endif /* XE_COMPILE_STATIC */
#if (defined(XE_COMPILE_EXPORT)||(!defined(XE_COMPILE_EXPORT)&&!defined(XE_COMPILE_STATIC)))&&!defined(XE_ALREADY_IMPLEMENTED)
#define XE_CANIMP
#endif

/*--- Misc Definitions ---*/
#define PI 3.1415926535897932384626433832795
#define cstring char*
#define MEMRANGESIZE(firstvar,lastvar)	(((unsigned int)&(lastvar))-((unsigned int)&(firstvar))+sizeof(lastvar))

/*--- XE_RES_* ---*/
#define XE_RES_XENON	1024

#define XE_RES_SPLASH	1
#define XE_RES_INTRO	2

/*--- XE_BASE_* ---*/
#define XE_BASE_BIN		2
#define XE_BASE_OCT		8
#define XE_BASE_DEC		10
#define XE_BASE_HEX		16

/*--- XE_FLAG_* ---*/
#define	XE_FLAG_EMPTY					0x00000000
#define	XE_FLAG_ALL						0xFFFFFFFF
#define	XE_FLAG_SWAP					0x00000001
#define	XE_FLAG_COLORBUFF				0x00000002
#define	XE_FLAG_DEPTHBUFF				0x00000004
#define	XE_FLAG_STENCILBUFF				0x00000008
#define	XE_FLAG_ACCUMBUFF				0x00000010
#define	XE_FLAG_TRANSFORMIDENT			0x00000020
#define	XE_FLAG_PREMULTALPHA			0x00000040
#define XE_FLAG_SUBSTANCE_SOLID_RANGE	0x00000001
#define XE_FLAG_SUBSTANCE_SOLID_SURFACE	0x00000002
#define XE_FLAG_X						0x00000001
#define XE_FLAG_Y						0x00000002
#define XE_FLAG_Z						0x00000004
#define XE_FLAG_W						0x00000008

/*--- XE_ESTATE ---*/
//! Zmienne stanu silnika
#define XE_NOONE (XE_ESTATE)-1
#define XE_NULL (XE_ESTATE)0
enum XE_ESTATE
{
XE_FALSE,
XE_TRUE,
XE_SINGLE,
XE_DOUBLE,
XE_POINTS,
XE_LINES,
XE_LINELOOP,
XE_LINESTRIP,
XE_TRIANGLES,
XE_TRIANGLESTRIP,
XE_TRIANGLEFAN,
XE_QUADS,
XE_QUADSTRIP,
XE_POLYGON,
XE_FLUSH,
XE_FINISH,
XE_RENDERTARGET,
XE_FRAMEBUFFER,
XE_VERTEXBUFFER,
XE_SHADER,
XE_TEXTURE,
XE_DISPLAYLIST,
XE_SUBSTANCE,
XE_RAY,
XE_COVALENCE_FORCE,
XE_FORCE,
XE_SURFACE,
XE_PARTICLE,
XE_GLOBAL_FORCE,
XE_BLACK_HOLE,
XE_WAVE_FORCE,
XE_PSYCHE,
XE_DNA,
XE_DNA_BRAIN,
XE_DNA_CELLS,
XE_DNA_MUSCLES,
XE_DNA_SKIN,
XE_DNA_BODY_TEXTURE,
XE_SOCKET,
XE_RENDERTARGET_COUNT,
XE_FRAMEBUFFER_COUNT,
XE_VERTEXBUFFER_COUNT,
XE_SHADER_COUNT,
XE_TEXTURE_COUNT,
XE_DISPLAYLIST_COUNT,
XE_SUBSTANCE_COUNT,
XE_RAY_COUNT,
XE_COVALENCE_FORCE_COUNT,
XE_FORCE_COUNT,
XE_SURFACE_COUNT,
XE_PARTICLE_COUNT,
XE_GLOBAL_FORCE_COUNT,
XE_BLACK_HOLE_COUNT,
XE_WAVE_FORCE_COUNT,
XE_PSYCHE_COUNT,
XE_DNA_COUNT,
XE_SOCKET_COUNT,
XE_1D,
XE_2D,
XE_3D,
XE_4D,
XE_CUBE,
XE_CUBE_POS_X,
XE_CUBE_NEG_X,
XE_CUBE_POS_Y,
XE_CUBE_NEG_Y,
XE_CUBE_POS_Z,
XE_CUBE_NEG_Z,
XE_DATA,
XE_FRONT_LEFT,
XE_FRONT_RIGHT,
XE_BACK_LEFT,
XE_BACK_RIGHT,
XE_ENGINE_NAME,
XE_ENGINE_CODENAME,
XE_ENGINE_CREATOR,
XE_ENGINE_VERSION,
XE_ENGINE_DESCRIPTION,
XE_ENGINE_COPYRIGHT,
XE_ENGINE_OWNER,
XE_ENGINE_COMPILE_OPTIONS,
XE_DRAW_TO_WINDOW,
XE_DRAW_TO_LAYER,
XE_RENDER_VENDOR,
XE_RENDER_RENDERER,
XE_RENDER_VERSION,
XE_RENDER_EXTENSIONS,
XE_RENDER_MODE,
XE_RENDER_FORMAT_RGBA,
XE_RENDER_FORMAT_DEPTH,
XE_RENDER_FORMAT_STENCIL,
XE_RENDER_VSYNC_MODE,
XE_RENDER_ORIENTATION,
XE_RENDER_CLOCKWISE,
XE_RENDER_COUNTERCLOCKWISE,
XE_RENDER_TEXT_SETTINGS,
XE_RENDER_TEXT_SIZE_X,
XE_RENDER_TEXT_SIZE_Y,
XE_RENDER_TEXT_SEP_X,
XE_RENDER_TEXT_SEP_Y,
XE_RENDER_TEXT_NEWLINE,
XE_RENDER_MASK,
XE_RENDER_MASK_DEPTH,
XE_RENDER_MASK_RED,
XE_RENDER_MASK_GREEN,
XE_RENDER_MASK_BLUE,
XE_RENDER_MASK_ALPHA,
XE_RENDER_DITHER,
XE_RENDER_TEXTURING_2D,
XE_RENDER_NORMALIZE,
XE_RENDER_FACE_FRONT,
XE_RENDER_FACE_BACK,
XE_RENDER_FACE_FRONT_AND_BACK,
XE_RENDER_LIGHTING,
XE_RENDER_LIGHT_SOURCE,
XE_RENDER_LIGHT_AMBIENT,
XE_RENDER_LIGHT_DIFFUSE,
XE_RENDER_LIGHT_SPECULAR,
XE_RENDER_LIGHT_POSITION,
XE_RENDER_LIGHT_SPOT_DIRECTION,
XE_RENDER_LIGHT_SPOT_EXPONENT,
XE_RENDER_LIGHT_SPOT_CUTOFF,
XE_RENDER_LIGHT_MODEL_AMBIENT,
XE_RENDER_MATERIAL_AMBIENT,
XE_RENDER_MATERIAL_DIFFUSE,
XE_RENDER_MATERIAL_SPECULAR,
XE_RENDER_MATERIAL_EMISSION,
XE_RENDER_MATERIAL_SHININESS,
XE_RENDER_DEPTH_TEST,
XE_RENDER_DEPTH_TYPE,
XE_RENDER_DEPTH_NEVER,
XE_RENDER_DEPTH_LESS,
XE_RENDER_DEPTH_EQUAL,
XE_RENDER_DEPTH_LEQUAL,
XE_RENDER_DEPTH_GREATER,
XE_RENDER_DEPTH_NOTEQUAL,
XE_RENDER_DEPTH_GEQUAL,
XE_RENDER_DEPTH_ALWAYS,
XE_RENDER_ALPHA_TEST,
XE_RENDER_ALPHA_TYPE,
XE_RENDER_ALPHA_REF,
XE_RENDER_ALPHA_NEVER,
XE_RENDER_ALPHA_LESS,
XE_RENDER_ALPHA_EQUAL,
XE_RENDER_ALPHA_LEQUAL,
XE_RENDER_ALPHA_GREATER,
XE_RENDER_ALPHA_NOTEQUAL,
XE_RENDER_ALPHA_GEQUAL,
XE_RENDER_ALPHA_ALWAYS,
XE_RENDER_CULL_FACE,
XE_RENDER_CULL_TYPE,
XE_RENDER_CULL_FRONT,
XE_RENDER_CULL_BACK,
XE_RENDER_CULL_FRONT_AND_BACK,
XE_RENDER_BLEND,
XE_RENDER_BLEND_TYPE,
XE_RENDER_BLEND_SRC,
XE_RENDER_BLEND_DST,
XE_RENDER_BLEND_ZERO,
XE_RENDER_BLEND_ONE,
XE_RENDER_BLEND_SRC_COLOR,
XE_RENDER_BLEND_ONE_MINUS_SRC_COLOR,
XE_RENDER_BLEND_DST_COLOR,
XE_RENDER_BLEND_ONE_MINUS_DST_COLOR,
XE_RENDER_BLEND_SRC_ALPHA,
XE_RENDER_BLEND_ONE_MINUS_SRC_ALPHA,
XE_RENDER_BLEND_DST_ALPHA,
XE_RENDER_BLEND_ONE_MINUS_DST_ALPHA,
XE_RENDER_BLEND_SRC_ALPHA_SATURATE,
XE_RENDER_BLEND_CONSTANT_COLOR,
XE_RENDER_BLEND_ONE_MINUS_CONSTANT_COLOR,
XE_RENDER_BLEND_CONSTANT_ALPHA,
XE_RENDER_BLEND_ONE_MINUS_CONSTANT_ALPHA,
XE_RENDER_BLEND_EQUATION,
XE_RENDER_BLEND_ADD,
XE_RENDER_BLEND_SUBTRACT,
XE_RENDER_BLEND_REVERSE_SUBTRACT,
XE_RENDER_BLEND_MIN,
XE_RENDER_BLEND_MAX,
XE_RENDER_FOG,
XE_RENDER_FOG_COLOR,
XE_RENDER_FOG_START,
XE_RENDER_FOG_END,
XE_RENDER_FOG_DENSITY,
XE_RENDER_FOG_MODE,
XE_RENDER_FOG_TYPE,
XE_RENDER_FOG_DISTANCE,
XE_RENDER_FOG_COORD,
XE_RENDER_FOG_LINEAR,
XE_RENDER_FOG_EXP,
XE_RENDER_FOG_EXP2,
XE_RENDER_STENCIL_TEST,
XE_RENDER_STENCIL_CLEAR,
XE_RENDER_STENCIL_TYPE,
XE_RENDER_STENCIL_FUNC,
XE_RENDER_STENCIL_FUNC_NEVER,
XE_RENDER_STENCIL_FUNC_LESS,
XE_RENDER_STENCIL_FUNC_EQUAL,
XE_RENDER_STENCIL_FUNC_LEQUAL,
XE_RENDER_STENCIL_FUNC_GREATER,
XE_RENDER_STENCIL_FUNC_NOTEQUAL,
XE_RENDER_STENCIL_FUNC_GEQUAL,
XE_RENDER_STENCIL_FUNC_ALWAYS,
XE_RENDER_STENCIL_REF,
XE_RENDER_STENCIL_MASK,
XE_RENDER_STENCIL_OPERATION,
XE_RENDER_STENCIL_KEEP,
XE_RENDER_STENCIL_ZERO,
XE_RENDER_STENCIL_REPLACE,
XE_RENDER_STENCIL_INCR,
XE_RENDER_STENCIL_DECR,
XE_RENDER_STENCIL_INVERT,
XE_RENDER_ACCUM,
XE_RENDER_ACCUM_COLOR,
XE_RENDER_ACCUM_ACCUM,
XE_RENDER_ACCUM_LOAD,
XE_RENDER_ACCUM_RETURN,
XE_RENDER_ACCUM_MULT,
XE_RENDER_ACCUM_ADD,
XE_RENDER_SCISSOR,
XE_RENDER_SCISSOR_MODE,
XE_RENDER_POINT_SIZE,
XE_RENDER_POINT_SIZE_MIN,
XE_RENDER_POINT_SIZE_MAX,
XE_RENDER_POINT_SPRITES_MODE,
XE_RENDER_POINT_SPRITES_COORD_REPLACE,
XE_RENDER_POINT_SMOOTH,
XE_RENDER_LINE_SMOOTH,
XE_RENDER_POLYGON_SMOOTH,
XE_HINT_FOG,
XE_HINT_PERSPECTIVE_CORRECTION,
XE_HINT_POINT_SMOOTH,
XE_HINT_LINE_SMOOTH,
XE_HINT_POLYGON_SMOOTH,
XE_HINT_NICEST,
XE_HINT_FASTEST,
XE_HINT_DONT_CARE,
XE_TEXTURE_PARAM_MIN_FILTER,
XE_TEXTURE_PARAM_MAG_FILTER,
XE_TEXTURE_PARAM_WRAP_S,
XE_TEXTURE_PARAM_WRAP_T,
XE_TEXTURE_PARAM_PRIORITY,
XE_TEXTURE_PARAM_NEAREST,
XE_TEXTURE_PARAM_LINEAR,
XE_TEXTURE_PARAM_NEAREST_MIPMAP_NEAREST,
XE_TEXTURE_PARAM_LINEAR_MIPMAP_NEAREST,
XE_TEXTURE_PARAM_NEAREST_MIPMAP_LINEAR,
XE_TEXTURE_PARAM_LINEAR_MIPMAP_LINEAR,
XE_TEXTURE_PARAM_REPEAT,
XE_TEXTURE_PARAM_CLAMP,
XE_TEXTURE_MODE_ADD,
XE_TEXTURE_MODE_MODULATE,
XE_TEXTURE_MODE_DECAL,
XE_TEXTURE_MODE_BLEND,
XE_TEXTURE_MODE_REPLACE,
XE_TEXTURE_MODE_COMBINE,
XE_TEXTURE_MULTI_MODE,
XE_TEXTURE_UNIT_COUNT,
XE_TEXTURE_WIDTH,
XE_TEXTURE_HEIGHT,
XE_TEXTURE_PIXELS_BUFFER,
XE_TEXTURE_PIXELS_SIZE,
XE_SCENE_COLOR,
XE_COLOR,
XE_MATRIX_PROJECTION,
XE_MATRIX_MODELVIEW,
XE_MATRIX_TEXTURE,
XE_PHYSICAL_SUBSTANCE_ENERGY_LEVEL,
XE_PHYSICAL_SUBSTANCE_ENERGY_LEVEL_MODE,
XE_PHYSICAL_GRAVITY_MODE,
XE_PHYSICAL_GRAVITY_FACTOR,
XE_PHYSICAL_COVALENCE_FORCE_MODE,
XE_PHYSICAL_COVALENCE_FORCE_FIRST_STREAM,
XE_PHYSICAL_COVALENCE_FORCE_LAST_STREAM,
XE_PHYSICAL_COVALENCE_FORCE_ITERATIVE_STREAM,
XE_PHYSICAL_COVALENCE_FORCE_BREAK_MODE,
XE_PHYSICAL_COVALENCE_FORCE_BREAK_TESSELLATE,
XE_PHYSICAL_FORCE_MODE,
XE_PHYSICAL_GLOBAL_FORCE_MODE,
XE_PHYSICAL_BLACK_HOLE_MODE,
XE_PHYSICAL_RANGE_COLLISION_MODE,
XE_PHYSICAL_RANGE_COLLISION_TYPE,
XE_PHYSICAL_RANGE_COLLISION_STANDARD_MOVE,
XE_PHYSICAL_RANGE_COLLISION_TEMPORARY_MOVE,
XE_PHYSICAL_SURFACE_COLLISION_MODE,
XE_PHYSICAL_SURFACE_COLLISION_MODE_2D,
XE_PHYSICAL_SURFACE_COLLISION_2D_AXIS_XY,
XE_PHYSICAL_SURFACE_COLLISION_2D_AXIS_YZ,
XE_PHYSICAL_SURFACE_COLLISION_2D_AXIS_ZX,
XE_PHYSICAL_SURFACE_COLLISION_TYPE,
XE_PHYSICAL_SURFACE_COLLISION_STANDARD_MOVE,
XE_PHYSICAL_SURFACE_COLLISION_TEMPORARY_MOVE,
XE_PHYSICAL_SURFACE_COLLISION_SAMPLE,
XE_PHYSICAL_INERTION_MODE,
XE_PHYSICAL_INERTION_FRICTION,
XE_PHYSICAL_INERTION_RANGE,
XE_PHYSICAL_INERTION_SURFACE,
XE_PHYSICAL_INERTION_FORCE,
XE_PHYSICAL_INERTION_GLOBAL_FORCE,
XE_PHYSICAL_INERTION_WAVE_FORCE,
XE_PHYSICAL_INERTION_MOVE,
XE_PHYSICAL_INERTION_FACTOR,
XE_PHYSICAL_INERTION_LIMITATION,
XE_PHYSICAL_WAVE_FORCE_MODE,
XE_PHYSICAL_GENERATE_TESSELLATION,
XE_PHYSICAL_GENERATE_SUBSTANCE_PROC,
XE_PHYSICAL_GENERATE_RAY_PROC,
XE_PHYSICAL_GENERATE_COVALENCE_FORCE_PROC,
XE_PHYSICAL_GENERATE_FORCE_PROC,
XE_PHYSICAL_GENERATE_SURFACE_PROC,
XE_PHYSICAL_GENERATE_PARTICLE_PROC,
XE_PHYSICAL_GENERATE_GLOBAL_FORCE_PROC,
XE_PHYSICAL_GENERATE_BLACK_HOLE_PROC,
XE_PHYSICAL_GENERATE_WAVE_FORCE_PROC,
XE_SUBSTANCE_OBJECT,
XE_SUBSTANCE_POSITION,
XE_SUBSTANCE_LAST_POSITION,
XE_SUBSTANCE_RANGE,
XE_SUBSTANCE_MASS,
XE_SUBSTANCE_ENERGY,
XE_SUBSTANCE_SOLID,
XE_SUBSTANCE_INERTION,
XE_RAY_POSITION,
XE_RAY_DIRECTION,
XE_RAY_ANGLE,
XE_RAY_ENERGY,
XE_RAY_FREQUENCY,
XE_COVALENCE_FORCE_SUBSTANCE_1,
XE_COVALENCE_FORCE_SUBSTANCE_2,
XE_COVALENCE_FORCE_FORCE,
XE_COVALENCE_FORCE_RESILIENCE,
XE_COVALENCE_FORCE_RANGE,
XE_COVALENCE_FORCE_SOLID,
XE_COVALENCE_FORCE_SOLID_ALWAYS,
XE_COVALENCE_FORCE_SOLID_TEMPORARY,
XE_COVALENCE_FORCE_SOLID_NEVER,
XE_FORCE_POSITION,
XE_FORCE_DIRECTION,
XE_FORCE_FORCE,
XE_FORCE_SPEED,
XE_FORCE_ANCHOR,
XE_FORCE_ANCHOR_DIRECTION,
XE_FORCE_RICOCHET,
XE_FORCE_RANGE,
XE_SURFACE_SUBSTANCE_1,
XE_SURFACE_SUBSTANCE_2,
XE_SURFACE_SUBSTANCE_3,
XE_SURFACE_TEXTURE,
XE_SURFACE_TEXTURE_TYPE,
XE_SURFACE_ORDER,
XE_SURFACE_COORD_SUBSTANCE_1,
XE_SURFACE_COORD_SUBSTANCE_2,
XE_SURFACE_COORD_SUBSTANCE_3,
XE_SURFACE_DENSITY,
XE_SURFACE_ORIENTATION,
XE_SURFACE_ORIENTATION_SINGLE,
XE_SURFACE_ORIENTATION_DOUBLE,
XE_PARTICLE_SUBSTANCE,
XE_PARTICLE_SIZE,
XE_PARTICLE_TEXTURE,
XE_PARTICLE_TEXTURE_TYPE,
XE_GLOBAL_FORCE_POSITION,
XE_GLOBAL_FORCE_DIRECTION,
XE_GLOBAL_FORCE_FORCE,
XE_BLACK_HOLE_POSITION,
XE_BLACK_HOLE_FORCE,
XE_BLACK_HOLE_RANGE,
XE_BLACK_HOLE_ANCHOR,
XE_WAVE_FORCE_POSITION,
XE_WAVE_FORCE_FORCE,
XE_WAVE_FORCE_RANGE,
XE_WAVE_FORCE_LENGTH,
XE_WAVE_FORCE_SPEED,
XE_WAVE_FORCE_ORIENTATION,
XE_WAVE_FORCE_ORIENTATION_IN,
XE_WAVE_FORCE_ORIENTATION_OUT,
XE_WAVE_FORCE_ORIENTATION_INNER,
XE_WAVE_FORCE_ORIENTATION_OUTER,
XE_PSYCHE_NEURON,
XE_PSYCHE_NEURON_LENGTH,
XE_PSYCHE_NEURON_GATE,
XE_PSYCHE_NEURON_POSITION,
XE_PSYCHE_NEURON_FIRST,
XE_PSYCHE_NEURON_LAST,
XE_PSYCHE_NEURON_NEXT,
XE_PSYCHE_NEURON_PREV,
XE_PSYCHE_DETECTOR,
XE_PSYCHE_DETECTOR_SIGNAL,
XE_PSYCHE_DETECTOR_POSITION,
XE_PSYCHE_DETECTOR_FIRST,
XE_PSYCHE_DETECTOR_LAST,
XE_PSYCHE_DETECTOR_NEXT,
XE_PSYCHE_DETECTOR_PREV,
XE_PSYCHE_EFFECTOR,
XE_PSYCHE_EFFECTOR_LENGTH,
XE_PSYCHE_EFFECTOR_SIGNAL,
XE_PSYCHE_EFFECTOR_POSITION,
XE_PSYCHE_EFFECTOR_FIRST,
XE_PSYCHE_EFFECTOR_LAST,
XE_PSYCHE_EFFECTOR_NEXT,
XE_PSYCHE_EFFECTOR_PREV,
XE_PSYCHE_SIGNAL,
XE_PSYCHE_SIGNAL_SIGNAL,
XE_PSYCHE_SIGNAL_POSITION,
XE_PSYCHE_SIGNAL_FIRST,
XE_PSYCHE_SIGNAL_LAST,
XE_PSYCHE_SIGNAL_NEXT,
XE_PSYCHE_SIGNAL_PREV,
XE_PSYCHE_SYNAPSIS,
XE_PSYCHE_SYNAPSIS_GATE,
XE_PSYCHE_SYNAPSIS_POSITION,
XE_PSYCHE_SYNAPSIS_FIRST,
XE_PSYCHE_SYNAPSIS_LAST,
XE_PSYCHE_SYNAPSIS_NEXT,
XE_PSYCHE_SYNAPSIS_PREV,
XE_LOGGER_FILE,
XE_AREA_MODE,
XE_AREA_X1,
XE_AREA_Y1,
XE_AREA_Z1,
XE_AREA_X2,
XE_AREA_Y2,
XE_AREA_Z2,
XE_AREA_PROPERTIES,
XE_PHYSICAL_PROCEDURE,
XE_V3,
XE_C3_V3,
XE_C4_N3_V3,
XE_T4_C4_N3_V4,
XE_SHADER_VP_ARB,
XE_SHADER_FP_ARB,
XE_SHADER_VP_SL,
XE_SHADER_FP_SL,
XE_SHADER_PROGRAM_SL,
XE_SHADER_VP_ARB_PROFILE,
XE_SHADER_FP_ARB_PROFILE,
XE_SHADER_SL_PROFILE,
XE_RENDERTARGET_WINDOW_HANDLE,
XE_RENDERTARGET_DISPLAY_CONTEXT,
XE_RENDERTARGET_RENDER_CONTEXT,
XE_RENDERTARGET_X,
XE_RENDERTARGET_Y,
XE_RENDERTARGET_WIDTH,
XE_RENDERTARGET_HEIGHT,
XE_RENDERTARGET_ANGLE,
XE_RENDERTARGET_ZNEAR,
XE_RENDERTARGET_ZFAR,
XE_RENDERTARGET_ASPECT,
XE_RENDERTARGET_DRAW_MODE,
XE_FRAMEBUFFER_WIDTH,
XE_FRAMEBUFFER_HEIGHT,
XE_FRAMEBUFFER_TEXTURE,
XE_VERTEXBUFFER_ARRAY,
XE_VERTEXBUFFER_ELEMENT,
XE_VERTEXBUFFER_STATIC_DRAW,
XE_VERTEXBUFFER_STATIC_READ,
XE_VERTEXBUFFER_STATIC_COPY,
XE_VERTEXBUFFER_DYNAMIC_DRAW,
XE_VERTEXBUFFER_DYNAMIC_READ,
XE_VERTEXBUFFER_DYNAMIC_COPY,
XE_VERTEXBUFFER_STREAM_DRAW,
XE_VERTEXBUFFER_STREAM_READ,
XE_VERTEXBUFFER_STREAM_COPY,
XE_VERTEXBUFFER_READ,
XE_VERTEXBUFFER_WRITE,
XE_VERTEXBUFFER_READWRITE,
XE_REGISTER_ON_TIME_AXIS,
XE_PHYSICAL_TIME,
XE_PHYSICAL_TIME_FACTOR,
XE_ESTATE_END
};

/*--- XE_TIME ---*/
/*! Klasa operacji na czasie. Singleton. */
class XE_TIME
{
private:
XE_TIME();
XE_TIME(const XE_TIME&);
LARGE_INTEGER Freq;
bool UseQPC;
public:
//! Zwraca referencje na instancje.
static XE_TIME& Use();
/*! Zwraca aktualny czas systemu (sekundy od uruchomienia systemu).
\return Sekundy.
*/
static float GetSecs();
/*! Zwraca aktualny czas systemu o podwojnej czulosci (sekundy od uruchomienia systemu).
\return Sekundy.
*/
static double GetSecsSens();
/*! Zwraca aktualny czas systemu (milisekundy od uruchomienia systemu).
\return Milisekundy.
*/
static LONGLONG GetMilisecs();
};

#ifdef XE_CANIMP

XE_TIME::XE_TIME()
{
UseQPC=(QueryPerformanceFrequency(&Freq)!=0);
}

XE_TIME& XE_TIME::Use()
{
static XE_TIME Instance;
return(Instance);
}

float XE_TIME::GetSecs()
{
if(!XE_TIME::Use().UseQPC)return((float)GetTickCount()/1000);
else
{
LARGE_INTEGER Ticks;
QueryPerformanceCounter(&Ticks);
return((float)(Ticks.QuadPart/(double)XE_TIME::Use().Freq.QuadPart));
}
}

double XE_TIME::GetSecsSens()
{
if(!XE_TIME::Use().UseQPC)return((double)GetTickCount()/1000);
else
{
LARGE_INTEGER Ticks;
QueryPerformanceCounter(&Ticks);
return((double)Ticks.QuadPart/(double)XE_TIME::Use().Freq.QuadPart);
}
}

LONGLONG XE_TIME::GetMilisecs()
{
if(!XE_TIME::Use().UseQPC)return((LONGLONG)GetTickCount());
else
{
LARGE_INTEGER Ticks;
QueryPerformanceCounter(&Ticks);
return((LONGLONG)((Ticks.QuadPart/(double)XE_TIME::Use().Freq.QuadPart)*1000));
}
}

#endif /* XE_CANIMP */

/*--- XE_MATH ---*/
/*! Klasa podstawowych operacji matematycznych. */
class XE_MATH
{
public:
/*! Zamienia stopnie na radiany.
\param deg Stopnie.
\return Radiany.
*/
static double DegToRad(double deg);
static float DegToRad(float deg);
/*! Zamienia radiany na stopnie.
\param rad Radiany.
\return Stopnie.
*/
static double RadToDeg(double rad);
static float RadToDeg(float rad);
/*! Wylicza pseudolosowa wartosc.
\param range Zakres wartosci liczony od 0.
\return Wartosc pseudolosowa.
*/
static double Random(double range=0);
/*! Resetuje ziarno losowania pseudolosowych liczb. */
static void RandomReset();
/*! Oblicza wartosc pomiedzy dwoma podanymi wartosciami, wzgledem wagi liczb (np. 0: pierwsza; 1: druga; 0.5: pomiedzy).
\param v1 Pierwsza wartosc.
\param v2 Druga wartosc.
\param amount Waga liczb.
\return Wyliczona wartosc.
*/
static double Amount(double v1,double v2,double amount);
/*! Umieszcza bit na danej pozycji.
\param data Wskaznik na dane.
\param bit Pozycja bitu.
*/
static void BitInclude(void* data,unsigned int bit);
/*! Zabiera bit z danej pozycji.
\param data Wskaznik na dane.
\param bit Pozycja bitu.
*/
static void BitExclude(void* data,unsigned int bit);
/*! Sprawdza bit na danej pozycji.
\param data Wskaznik na dane.
\param bit Pozycja bitu.
\return Wartosc bitu.
*/
static bool BitCheck(void* data,unsigned int bit);
};

#ifdef XE_CANIMP

double XE_MATH::DegToRad(double deg)
{
return(deg*PI/180);
}

float XE_MATH::DegToRad(float deg)
{
return(deg*(float)PI/180);
}

double XE_MATH::RadToDeg(double rad)
{
return(rad*180/PI);
}

float XE_MATH::RadToDeg(float rad)
{
return(rad*180/(float)PI);
}

double XE_MATH::Random(double range)
{
return((((double)(rand()%1000))/1000.0f)*range);
}

void XE_MATH::RandomReset()
{
srand((unsigned int)XE_TIME::GetSecsSens());
}

double XE_MATH::Amount(double v1,double v2,double amount)
{
return((v2-v1)*min(1,max(0,amount))+v1);
}

void XE_MATH::BitInclude(void* data,unsigned int bit)
{
unsigned int _field=bit/8;
unsigned int _bit=bit%8;
unsigned char* _buff=(unsigned char*)data;
_buff[_field]=_buff[_field] | ((unsigned char)1 << _bit);
}

void XE_MATH::BitExclude(void* data,unsigned int bit)
{
unsigned int _field=bit/8;
unsigned int _bit=bit%8;
unsigned char* _buff=(unsigned char*)data;
_buff[_field]=_buff[_field] & ~((unsigned char)1 << _bit);
}

bool XE_MATH::BitCheck(void* data,unsigned int bit)
{
unsigned int _field=bit/8;
unsigned int _bit=bit%8;
unsigned char* _buff=(unsigned char*)data;
if(_buff[_field] & ((unsigned char)1 << _bit))
return(true);
return(false);
}

#endif /* XE_CANIMP */

/*--- XE_SMART ---*/
template<typename T> class XE_SMART;

/*! Klasa inteligentnego wskaznika.
Obiekt tego typu przechowuje wskaznik na dane tworzone dynamicznie i zwalnia zajmowana pamiec gdy zaden inny obiekt z niego nie korzysta.
*/
template<typename T>
class XE_SMART
{
private:
bool Empty;
unsigned int* Counter;
T* Data;
public:
/*! Ustala czy pamiec obiektu jest statyczna (jesli true to podczas niszczenia wskaznika nie zostaje zwalniana pamiec obiektu) */
bool Static;
/*! Konstruktor domyslny. */
XE_SMART();
/*! Konstruktor powiazujacy.
Powiazuje ze soba dany obiekt wskazywany przez wskaznik.
\param _pointer Wskaznik na obiekt.
*/
XE_SMART(T* _pointer);
/*! Konstruktor klonujacy.
Klonuje dany obiekt i powiazuje go ze soba.
\param _clone Obiekt.
*/
XE_SMART(const T _clone);
/*! Konstruktor powiazujacy.
Powiazuje ze soba obiekt innego inteligentnego wskaznika.
\param _smart Referencja na wskaznik inteligentny.
*/
XE_SMART(const XE_SMART<T>& _smart);
/*! Konstruktor powiazujacy.
Powiazuje ze soba obiekt innego inteligentnego wskaznika, poprzez wskaznik na niego.
\param _smartp Wskaznik na wskaznik inteligentny.
*/
XE_SMART(XE_SMART<T>* _smartp);
/*! Destruktor. */
~XE_SMART();
/*! Powiazuje ze soba dany obiekt wskazywany przez wskaznik.
\param _pointer Wskaznik na obiekt.
\return true jesli operacja powiodla sie.
*/
bool RefPointer(T* _pointer);
/*! Klonuje dany obiekt i powiazuje go ze soba.
\param _clone Obiekt.
\return true jesli operacja powiodla sie.
*/
bool RefClone(const T _clone);
/*! Powiazuje ze soba obiekt innego inteligentnego wskaznika.
\param _smart Referencja na wskaznik inteligentny.
\return true jesli operacja powiodla sie.
*/
bool RefSmart(const XE_SMART<T>& _smart);
/*! Powiazuje ze soba obiekt innego inteligentnego wskaznika, poprzez wskaznik na niego.
\param _smartp Wskaznik na wskaznik inteligentny.
\return true jesli operacja powiodla sie.
*/
bool RefSmartP(XE_SMART<T>* _smartp);
/*! Odwiazuje siebie od obiektu. */
void Unref();
/*! Abstrakcyjne powiazanie (nie powiazuje z zadnym fizycznym obiektem, jednak zwieksza licznik powiazan).
\return true jesli operacja powiodla sie.
*/
bool AbstractRef();
/*! Abstrakcyjne odwiazanie (nie odwiazuje od siebie fizycznego obiektu, jednak zmniejsza licznik powiazan). */
void AbstractUnref();
/*! Zwraca wskaznik na obiekt powiazany. */
T* Pointer();
/*! Zwraca klona obiektu powiazanego. */
T Clone();
/*! Sprawdza czy inteligentny wskaznik jest pusty (czyli nie jest powiazany z zadnym obiektem).
\return true jesli wskaznik jest pusty.
*/
bool IsEmpty();
/*! Zwraca liczbe powiazan.
\return Liczba powiazan inteligentnego wskaznika.
*/
unsigned int Count();
/*! Operator powiazujacy.
Powiazuje ze soba dany obiekt wskazywany przez wskaznik.
\param _pointer Wskaznik na obiekt.
*/
void operator= (T* _pointer);
/*! Operator klonujacy.
Klonuje dany obiekt i powiazuje go ze soba.
\param _clone Obiekt.
*/
void operator= (const T _clone);
/*! Operator powiazujacy.
Powiazuje ze soba obiekt innego inteligentnego wskaznika.
\param _smart Referencja na wskaznik inteligentny.
*/
void operator= (const XE_SMART<T>& _smart);
/*! Operator powiazujacy.
Powiazuje ze soba obiekt innego inteligentnego wskaznika, poprzez wskaznik na niego.
\param _smartp Wskaznik na wskaznik inteligentny.
*/
void operator= (XE_SMART<T>* _smartp);
/*! Operator wyluskania.
Zwraca wskaznik na obiekt powiazany.
\return Wskaznik na obiekt powiazany.
*/
T* operator-> ();
};

template<typename T>
XE_SMART<T>::XE_SMART()
{
Empty=true;
Static=false;
Counter=NULL;
Data=NULL;
RefClone(T());
}

template<typename T>
XE_SMART<T>::XE_SMART(T* _pointer)
{
Empty=true;
Static=false;
Counter=NULL;
Data=NULL;
RefPointer(_pointer);
}

template<typename T>
XE_SMART<T>::XE_SMART(const T _clone)
{
Empty=true;
Static=false;
Counter=NULL;
Data=NULL;
RefClone(_clone);
}

template<typename T>
XE_SMART<T>::XE_SMART(const XE_SMART<T>& _smart)
{
Empty=true;
Static=false;
Counter=NULL;
Data=NULL;
RefSmart(_smart);
}

template<typename T>
XE_SMART<T>::XE_SMART(XE_SMART<T>* _smartp)
{
Empty=true;
Static=false;
Counter=NULL;
Data=NULL;
RefSmartP(_smartp);
}

template<typename T>
XE_SMART<T>::~XE_SMART()
{
Unref();
}

template<typename T>
bool XE_SMART<T>::RefPointer(T* _pointer)
{
if(Empty&&Counter==NULL)
{
Empty=false;
#ifdef XE_ALLOC_C
Counter=(unsigned int*)malloc(sizeof(unsigned int));
#else
Counter=new unsigned int[1];
#endif /* XE_ALLOC_C */
*Counter=0;
Data=_pointer;
*Counter+=1;
return(true);
}
if(!Empty&&Counter!=NULL)
if(_pointer!=Data)
{
Unref();
Empty=false;
#ifdef XE_ALLOC_C
Counter=(unsigned int*)malloc(sizeof(unsigned int));
#else
Counter=new unsigned int[1];
#endif /* XE_ALLOC_C */
*Counter=0;
Data=_pointer;
*Counter+=1;
return(true);
}
return(false);
}

template<typename T>
bool XE_SMART<T>::RefClone(const T _clone)
{
if(Empty&&Counter==NULL)
{
Empty=false;
#ifdef XE_ALLOC_C
Counter=(unsigned int*)malloc(sizeof(unsigned int));
#else
Counter=new unsigned int[1];
#endif /* XE_ALLOC_C */
*Counter=0;
#ifdef XE_ALLOC_C
Data=(T*)malloc(sizeof(T));
#else
Data=new T[1];
#endif /* XE_ALLOC_C */
*Data=_clone;
*Counter+=1;
return(true);
}
if(!Empty&&Counter!=NULL)
{
Unref();
Empty=false;
#ifdef XE_ALLOC_C
Counter=(unsigned int*)malloc(sizeof(unsigned int));
#else
Counter=new unsigned int[1];
#endif /* XE_ALLOC_C */
*Counter=0;
#ifdef XE_ALLOC_C
Data=(T*)malloc(sizeof(T));
#else
Data=new T[1];
#endif /* XE_ALLOC_C */
*Data=_clone;
*Counter+=1;
return(true);
}
return(false);
}

template<typename T>
bool XE_SMART<T>::RefSmart(const XE_SMART<T>& _smart)
{
if(Empty&&Counter==NULL)
{
Empty=false;
Counter=_smart.Counter;
Data=_smart.Data;
*Counter+=1;
return(true);
}
if(!Empty&&Counter!=NULL)
if(_smart.Data!=Data)
{
Unref();
Empty=false;
Counter=_smart.Counter;
Data=_smart.Data;
*Counter+=1;
return(true);
}
return(false);
}

template<typename T>
bool XE_SMART<T>::RefSmartP(XE_SMART<T>* _smartp)
{
if(Empty&&Counter==NULL)
{
Empty=false;
Counter=_smartp->Counter;
Data=_smartp->Data;
*Counter+=1;
return(true);
}
if(!Empty&&Counter!=NULL)
if(_smartp->Data!=Data)
{
Unref();
Empty=false;
Counter=_smartp->Counter;
Data=_smartp->Data;
*Counter+=1;
return(true);
}
return(false);
}

template<typename T>
void XE_SMART<T>::Unref()
{
if(!Empty&&Counter!=NULL)
{
*Counter-=1;
if(*Counter==0)
{
#ifdef XE_ALLOC_C
free(Counter);
#else
delete[] Counter;
#endif /* XE_ALLOC_C */
Counter=NULL;
if(!Static)
{
#ifdef XE_ALLOC_C
free(Data);
#else
delete[] Data;
#endif /* XE_ALLOC_C */
Data=NULL;
}
}
}
}

template<typename T>
bool XE_SMART<T>::AbstractRef()
{
if(!Empty&&Counter!=NULL)
{
*Counter+=1;
return(true);
}
return(false);
}

template<typename T>
void XE_SMART<T>::AbstractUnref()
{
if(!Empty&&Counter!=NULL)
*Counter-=1;
}

template<typename T>
T* XE_SMART<T>::Pointer()
{
if(!Empty&&Counter!=NULL)
return(Data);
return(NULL);
}

template<typename T>
T XE_SMART<T>::Clone()
{
if(!Empty&&Counter!=NULL)
return(*Data);
T _temp;
return(_temp);
}

template<typename T>
bool XE_SMART<T>::IsEmpty()
{
return(Empty);
}

template<typename T>
unsigned int XE_SMART<T>::Count()
{
if(!Empty&&Counter!=NULL)
return(*Counter);
return(0);
}

template<typename T>
void XE_SMART<T>::operator= (T* _pointer)
{
RefPointer(_pointer);
}

template<typename T>
void XE_SMART<T>::operator= (const T _clone)
{
RefClone(_clone);
}

template<typename T>
void XE_SMART<T>::operator= (const XE_SMART<T>& _smart)
{
RefSmart(_smart);
}

template<typename T>
void XE_SMART<T>::operator= (XE_SMART<T>* _smartp)
{
RefSmartP(_smartp);
}

template<typename T>
T* XE_SMART<T>::operator-> ()
{
return(Pointer());
}

/*--- XE_ELEMENTS & XE_ELEMENT_BASE & XE_ELEMENT_POINTER ---*/
template<typename T> class XE_ELEMENT_BASE;
template<typename T> class XE_ELEMENT_POINTER;
template<typename T> class XE_ELEMENTS;

#define XE_FOREACH(type,container,as)	for(XE_ELEMENT_POINTER< type > as=container.FirstPointer();!as.IsEmpty();as.Next())if(!as.IsEmpty())

/*-- XE_ELEMENT_BASE --*/
template<typename T>
class XE_ELEMENT_BASE
{
friend class XE_ELEMENT_POINTER<T>;
friend class XE_ELEMENTS<T>;
private:
T data;
bool empty;
bool original;
XE_ELEMENT_BASE<T> *parent;
XE_ELEMENT_BASE<T> *prev;
XE_ELEMENT_BASE<T> *next;
XE_ELEMENTS<T> *container;
public:
XE_ELEMENT_BASE();
~XE_ELEMENT_BASE();
};

template<typename T>
XE_ELEMENT_BASE<T>::XE_ELEMENT_BASE()
{
empty=true;
original=true;
parent=NULL;
prev=NULL;
next=NULL;
container=NULL;
}

template<typename T>
XE_ELEMENT_BASE<T>::~XE_ELEMENT_BASE()
{
}

/*-- XE_ELEMENT_POINTER --*/
/*! Klasa wskaznika elementu kontenera elementow.
Przechowuje wskaznik elementu kontenera elementow.
*/
template<typename T>
class XE_ELEMENT_POINTER
{
friend class XE_ELEMENT_BASE<T>;
friend class XE_ELEMENTS<T>;
private:
bool empty;
bool original;
XE_ELEMENT_BASE<T> *parent;
public:
/*! Konstruktor domyslny. */
XE_ELEMENT_POINTER();
/*! Zwraca posredni klon elementu (przechowywany w kontenerze).
\return Klon obiektu.
*/
T iClone();
/*! Zwraca posredni wskaznik na element (przechowywany w kontenerze).
\return Wskaznik na obiekt.
*/
T* iPointer();
/*! Inkrementuje element (o pozycje w przod).
Jesli natrafi za koncowy element kontenera, to przyjmuje wartosc pustego elementu, lub pierwszego w liscie.
\param cycle Ustala, czy iteracja ma zataczac cykl.
*/
void Next(bool cycle=false);
/*! Dekrementuje element (o pozycje wstecz).
Jesli natrafi przed poczatkowy element kontenera, to przyjmuje wartosc pustego elementu, lub ostatniego w liscie.
\param cycle Ustala, czy iteracja ma zataczac cykl.
*/
void Prev(bool cycle=false);
/*! Sprawdza czy element jest pusty.
\return true jesli element jest pusty.
*/
bool IsEmpty();
/*! Dowiazuje sie do obiektu danego wskaznika elementu.
\param elm Referencja na wskaznik elementu.
*/
void RefPointer(XE_ELEMENT_POINTER<T>& elm);
/*! Odwiazuje sie od kontenera. */
void Unref();
/*! Operator wyluskania.
Zwraca posredni wskaznik na obiekt (przechowywany w kontenerze).
\return Wskaznik na obiekt.
*/
T* operator-> ();
/*! Operator wyluskania.
Zwraca referencje na obiekt (przechowywany w kontenerze).
\return Referencja na obiekt.
*/
T& operator() ();
/*! Operator porownania.
Sprawdza czy oba wskazniki elementow sa rowne.
\param elm Wskaznik elementu.
\return true jesli wskazniki elementow sa rowne.
*/
bool operator== (XE_ELEMENT_POINTER<T>& elm);
/*! Operator porownania.
Sprawdza czy oba wskazniki elementow sa rozne.
\param elm Wskaznik elementu.
\return true jesli wskazniki elementow sa rozne.
*/
bool operator!= (XE_ELEMENT_POINTER<T>& elm);
/*! Operator preinkrementacji.
Inkrementuje element (o pozycje w przód).
\return Element.
*/
XE_ELEMENT_POINTER<T> operator++ ();
/*! Operator predekrementacji.
Dekrementuje element (o pozycje w przód).
\return Element.
*/
XE_ELEMENT_POINTER<T> operator-- ();
/*! Operator postinkrementacji.
Inkrementuje element (o pozycje w przód).
\return Element.
*/
XE_ELEMENT_POINTER<T> operator++ (int);
/*! Operator postdekrementacji.
Dekrementuje element (o pozycje w przód).
\return Element.
*/
XE_ELEMENT_POINTER<T> operator-- (int);
};

template<typename T>
T* XE_ELEMENT_POINTER<T>::operator-> ()
{
return(iPointer());
}

template<typename T>
T& XE_ELEMENT_POINTER<T>::operator() ()
{
return(*iPointer());
}

template<typename T>
bool XE_ELEMENT_POINTER<T>::operator== (XE_ELEMENT_POINTER<T>& elm)
{
if(iPointer()==elm.iPointer())
return(true);
return(false);
}

template<typename T>
bool XE_ELEMENT_POINTER<T>::operator!= (XE_ELEMENT_POINTER<T>& elm)
{
if(iPointer()!=elm.iPointer())
return(true);
return(false);
}

template<typename T>
XE_ELEMENT_POINTER<T> XE_ELEMENT_POINTER<T>::operator++ ()
{
this->Next();
return(*this);
}

template<typename T>
XE_ELEMENT_POINTER<T> XE_ELEMENT_POINTER<T>::operator-- ()
{
this->Prev();
return(*this);
}

template<typename T>
XE_ELEMENT_POINTER<T> XE_ELEMENT_POINTER<T>::operator++ (int)
{
XE_ELEMENT_POINTER<T> temp=*this;
this->Next();
return(temp);
}

template<typename T>
XE_ELEMENT_POINTER<T> XE_ELEMENT_POINTER<T>::operator-- (int)
{
XE_ELEMENT_POINTER<T> temp=*this;
this->Prev();
return(temp);
}

template<typename T>
XE_ELEMENT_POINTER<T>::XE_ELEMENT_POINTER()
{
empty=true;
original=false;
parent=NULL;
}

template<typename T>
T XE_ELEMENT_POINTER<T>::iClone()
{
if(!empty)
if(parent!=NULL)
return(parent->data);
return(T());
}

template<typename T>
T* XE_ELEMENT_POINTER<T>::iPointer()
{
if(!empty)
if(parent!=NULL)
return(&parent->data);
return(NULL);
}

template<typename T>
void XE_ELEMENT_POINTER<T>::Next(bool cycle)
{
if(parent!=NULL)
{
XE_ELEMENT_POINTER<T> temp;
if(parent->next!=NULL)
{
temp.empty=parent->next->empty;
temp.parent=parent->next;
}
else
if(cycle)temp=parent->container->FirstPointer();
empty=temp.empty;
parent=temp.parent;
}
}

template<typename T>
void XE_ELEMENT_POINTER<T>::Prev(bool cycle)
{
if(parent!=NULL)
{
XE_ELEMENT_POINTER<T> temp;
if(parent->prev!=NULL)
{
temp.empty=parent->prev->empty;
temp.parent=parent->prev;
}
else
if(cycle)temp=parent->container->LastPointer();
empty=temp.empty;
parent=temp.parent;
}
}

template<typename T>
bool XE_ELEMENT_POINTER<T>::IsEmpty()
{
if(empty||!parent)
return(true);
return(false);
}

template<typename T>
void XE_ELEMENT_POINTER<T>::RefPointer(XE_ELEMENT_POINTER<T>& elm)
{
if(elm.parent!=NULL)
{
empty=elm.parent->empty;
parent=elm.parent->parent;
}
}

template<typename T>
void XE_ELEMENT_POINTER<T>::Unref()
{
empty=true;
parent=NULL;
}

/*-- XE_ELEMENTS --*/
/*! Klasa kontenera elementow.
Przechowuje obiekty, do ktorych dostep daja klony oraz wskazniki elementow.
*/
template<typename T>
class XE_ELEMENTS
{
private:
XE_ELEMENT_BASE<T>* first;
XE_ELEMENT_BASE<T>* last;
unsigned int size;
public:
/*! Ustala czy pamiec kontenera jest statyczna (jesli true to podczas niszczenia kontenera nie zostaje zwalniana jego pamiec). */
bool Static;
/*! Ustala czy elementy maja byc dodawane poprzez klonowanie pamieci obiektu, a nie samego obiektu. (nie zaleca sie stosowac) */
bool AddingByCloning;
/*! Konstruktor domyslny. */
XE_ELEMENTS();
/*! Destruktor. */
~XE_ELEMENTS();
/*! Zwraca rozmiar kontenera.
\return Ilosc elementow w kontenerze.
*/
unsigned int Size();
/*! Czysci kontener. */
void Clear();
/*! Sortuje kontener funkcja foo.
\param foo Wskaznik na funkcje porownujaca 2 elementy (wartosc wieksza niz 0 oznacza wlasciwa kolejnosc, 0 oznacza identyczne wartosci, a mniej niz 0 oznacza niewlasciwa kolejnosc i koniecznosc zamiany miejsc).
*/
void Sort(int(*foo)(T*,T*));
/*! Zamienia elementy miejscami.
\param elm1 Pierwszy element.
\param elm2 Drugi element.
\return true jesli operacja powiodla sie.
*/
bool Swap(XE_ELEMENT_POINTER<T> elm1,XE_ELEMENT_POINTER<T> elm2);
/*! Przechwytuje element z innego kontenera (wskazniki na dane nie traca waznosci).
\param elm element do przechwycenia.
\return true jesli operacja powiodla sie.
*/
bool Capture(XE_ELEMENT_POINTER<T> elm);
/*! Zwraca wskaznik pierwszego elementu kontenera.
\return Wskaznik elementu.
*/
XE_ELEMENT_POINTER<T> FirstPointer();
/*! Zwraca wskaznik ostatniego elementu kontenera.
\return Wskaznik elementu.
*/
XE_ELEMENT_POINTER<T> LastPointer();
/*! Dodaje obiekt do kontenera i zwraca jego wskaznik elementu.
\param value Obiekt.
\return Wskaznik elementu.
*/
XE_ELEMENT_POINTER<T> AddPointer(const T& value);
/*! Dodaje obiekt do kontenera, tuz za okreslonym elementem i zwraca jego wskaznik elementu.
\param value Obiekt.
\param elm Wskaznik elementu.
\return Wskaznik elementu.
*/
XE_ELEMENT_POINTER<T> InsertPointer(const T& value,XE_ELEMENT_POINTER<T> elm);
/*! Usuwa obiekt z kontenera wskazywany przez wskaznik elementu.
\param elm Referencja na wskaznik elementu.
*/
void ErasePointer(XE_ELEMENT_POINTER<T>& elm);
};

template<typename T>
XE_ELEMENTS<T>::XE_ELEMENTS()
{
first=NULL;
last=NULL;
size=0;
Static=false;
AddingByCloning=false;
}

template<typename T>
XE_ELEMENTS<T>::~XE_ELEMENTS()
{
if(!Static)Clear();
}

template<typename T>
unsigned int XE_ELEMENTS<T>::Size()
{
return(size);
}

template<typename T>
void XE_ELEMENTS<T>::Clear()
{
if(size>0)
{
XE_ELEMENT_BASE<T>* temp=first;
XE_ELEMENT_BASE<T>* _temp;
while(temp!=NULL)
{
_temp=temp;
temp=temp->next;
#ifdef XE_ALLOC_C
free(_temp);
#else
delete[] _temp;
#endif /* XE_ALLOC_C */
}
first=NULL;
last=NULL;
size=0;
}
}

template<typename T>
void XE_ELEMENTS<T>::Sort(int(*foo)(T*,T*))
{
if(foo)
{
bool change=true;
XE_ELEMENT_POINTER<T> iter;
XE_ELEMENT_POINTER<T> temp;
while(change)
{
change=false;
iter=FirstPointer();
temp=iter;
temp.Next();
while(!iter.IsEmpty())
{
if(!temp.IsEmpty())
if(foo(iter.iPointer(),temp.iPointer())<0)
{
T toswap;
toswap=iter.parent->data;
iter.parent->data=temp.parent->data;
temp.parent->data=toswap;
change=true;
}
iter.Next();
temp.Next();
}
iter=LastPointer();
temp=iter;
temp.Prev();
if(change)
while(!iter.IsEmpty())
{
if(!temp.IsEmpty())
if(foo(iter.iPointer(),temp.iPointer())>0)
{
T toswap;
toswap=iter.parent->data;
iter.parent->data=temp.parent->data;
temp.parent->data=toswap;
change=true;
}
iter.Prev();
temp.Prev();
}
}
}
}

template<typename T>
bool XE_ELEMENTS<T>::Swap(XE_ELEMENT_POINTER<T> elm1,XE_ELEMENT_POINTER<T> elm2)
{
if(elm1.IsEmpty()||elm2.IsEmpty())return(false);
T toswap;
XE_ELEMENTS<T> *cont1=NULL;
XE_ELEMENTS<T> *cont2=NULL;
cont1=elm1.parent->container;
cont2=elm2.parent->container;
toswap=elm1.parent->data;
elm1.parent->data=elm2.parent->data;
elm2.parent->data=toswap;
elm1.parent->container=cont2;
elm2.parent->container=cont1;
return(true);
}

template<typename T>
bool XE_ELEMENTS<T>::Capture(XE_ELEMENT_POINTER<T> elm)
{
if(elm.IsEmpty())return(false);
if(elm.parent->container==this)return(false);
XE_ELEMENT_BASE<T>* _elm=elm.parent;
XE_ELEMENT_BASE<T>* _prev=_elm->prev;
XE_ELEMENT_BASE<T>* _next=_elm->next;
XE_ELEMENTS<T>* _con=_elm->container;
if(_prev)_prev->next=_next;
if(_next)_next->prev=_prev;
if(_con->first==_elm)(_next)?_con->first=_next:_con->first=NULL;
if(_con->last==_elm)(_prev)?_con->last=_prev:_con->last=NULL;
_elm->next=NULL;
_elm->prev=last;
if(last)last->next=_elm;
last=_elm;
if(!first)first=_elm;
size++;
if(_elm->container)_elm->container->size--;
_elm->container=this;
return(true);
}

template<typename T>
XE_ELEMENT_POINTER<T> XE_ELEMENTS<T>::FirstPointer()
{
XE_ELEMENT_POINTER<T> temp;
if(first!=NULL)
{
temp.empty=first->empty;
temp.parent=first;
}
return(temp);
}

template<typename T>
XE_ELEMENT_POINTER<T> XE_ELEMENTS<T>::LastPointer()
{
XE_ELEMENT_POINTER<T> temp;
if(last!=NULL)
{
temp.empty=last->empty;
temp.parent=last;
}
return(temp);
}

#ifdef XE_PROTOTYPE
template<typename T>
XE_ELEMENT_POINTER<T> XE_ELEMENTS<T>::AddPointer(const T& value)
{
if(size==0)
{
size++;
XE_ELEMENT_BASE<T>* _first=first;
XE_ELEMENT_BASE<T>* _last=last;
XE_ELEMENT_BASE<T>* temp=(XE_ELEMENT_BASE<T>*)malloc(sizeof(XE_ELEMENT_BASE<T>));
if(!temp)return(XE_ELEMENT_POINTER<T>());
temp->data=(T)value;
temp->empty=false;
temp->original=true;
temp->parent=NULL;
temp->next=NULL;
temp->prev=NULL;
#ifdef XE_ALLOC_C
first=(XE_ELEMENT_BASE<T>*)malloc(sizeof(XE_ELEMENT_BASE<T>));
#else
first=new XE_ELEMENT_BASE<T>[1];
#endif /* XE_ALLOC_C */
last=first;
XE_ELEMENT_POINTER<T> _temp;
if(last==NULL)
{
first=_first;
last=_last;
free(temp);
return(_temp);
}
if(AddingByCloning)
memcpy(last,temp,sizeof(XE_ELEMENT_BASE<T>));
else
*last=*temp;
_temp.empty=false;
_temp.original=false;
_temp.parent=last;
_temp.parent->container=this;
free(temp);
return(_temp);
}
else
{
size++;
XE_ELEMENT_BASE<T>* _first=first;
XE_ELEMENT_BASE<T>* _last=last;
XE_ELEMENT_BASE<T>* temp=(XE_ELEMENT_BASE<T>*)malloc(sizeof(XE_ELEMENT_BASE<T>));
if(!temp)return(XE_ELEMENT_POINTER<T>());
temp->data=(T)value;
temp->empty=false;
temp->original=true;
temp->parent=NULL;
temp->next=NULL;
temp->prev=last;
#ifdef XE_ALLOC_C
last->next=(XE_ELEMENT_BASE<T>*)malloc(sizeof(XE_ELEMENT_BASE<T>));
#else
last->next=new XE_ELEMENT_BASE<T>[1];
#endif /* XE_ALLOC_C */
last=last->next;
XE_ELEMENT_POINTER<T> _temp;
if(last==NULL)
{
first=_first;
last=_last;
free(temp);
return(_temp);
}
*last=*temp;
_temp.empty=false;
_temp.original=false;
_temp.parent=last;
_temp.parent->container=this;
free(temp);
return(_temp);
}
}
#else /* XE_PROTOTYPE */
template<typename T>
XE_ELEMENT_POINTER<T> XE_ELEMENTS<T>::AddPointer(const T& value)
{
if(size==0)
{
size++;
XE_ELEMENT_BASE<T>* _first=first;
XE_ELEMENT_BASE<T>* _last=last;
XE_ELEMENT_BASE<T> temp;
temp.data=(T)value;
temp.empty=false;
temp.original=true;
temp.parent=NULL;
temp.next=NULL;
temp.prev=NULL;
#ifdef XE_ALLOC_C
first=(XE_ELEMENT_BASE<T>*)malloc(sizeof(XE_ELEMENT_BASE<T>));
#else
first=new XE_ELEMENT_BASE<T>[1];
#endif /* XE_ALLOC_C */
last=first;
XE_ELEMENT_POINTER<T> _temp;
if(last==NULL)
{
first=_first;
last=_last;
return(_temp);
}
if(AddingByCloning)
memcpy(last,&temp,sizeof(XE_ELEMENT_BASE<T>));
else
*last=temp;
_temp.empty=false;
_temp.original=false;
_temp.parent=last;
_temp.parent->container=this;
return(_temp);
}
else
{
size++;
XE_ELEMENT_BASE<T>* _first=first;
XE_ELEMENT_BASE<T>* _last=last;
XE_ELEMENT_BASE<T> temp;
temp.data=(T)value;
temp.empty=false;
temp.original=true;
temp.parent=NULL;
temp.next=NULL;
temp.prev=last;
#ifdef XE_ALLOC_C
last->next=(XE_ELEMENT_BASE<T>*)malloc(sizeof(XE_ELEMENT_BASE<T>));
#else
last->next=new XE_ELEMENT_BASE<T>[1];
#endif /* XE_ALLOC_C */
last=last->next;
XE_ELEMENT_POINTER<T> _temp;
if(last==NULL)
{
first=_first;
last=_last;
return(_temp);
}
*last=temp;
_temp.empty=false;
_temp.original=false;
_temp.parent=last;
_temp.parent->container=this;
return(_temp);
}
}
#endif /* XE_PROTOTYPE */

template<typename T>
XE_ELEMENT_POINTER<T> XE_ELEMENTS<T>::InsertPointer(const T& value,XE_ELEMENT_POINTER<T> elm)
{
if(!elm.IsEmpty())
if(elm.parent->container!=this)
return(XE_ELEMENT_POINTER<T>());
if(size==0)
{
size++;
XE_ELEMENT_BASE<T>* _first=first;
XE_ELEMENT_BASE<T>* _last=last;
XE_ELEMENT_BASE<T> temp;
temp.data=value;
temp.empty=false;
temp.original=true;
temp.parent=NULL;
temp.next=NULL;
temp.prev=NULL;
#ifdef XE_ALLOC_C
first=(XE_ELEMENT_BASE<T>*)malloc(sizeof(XE_ELEMENT_BASE<T>));
#else
first=new XE_ELEMENT_BASE<T>[1];
#endif /* XE_ALLOC_C */
last=first;
XE_ELEMENT_POINTER<T> _temp;
if(last==NULL)
{
first=_first;
last=_last;
return(_temp);
}
if(AddingByCloning)
memcpy(last,&temp,sizeof(XE_ELEMENT_BASE<T>));
else
*last=temp;
_temp.empty=false;
_temp.original=false;
_temp.parent=last;
_temp.parent->container=this;
return(_temp);
}
else
{
size++;
XE_ELEMENT_BASE<T>* _first=first;
XE_ELEMENT_BASE<T>* _last=last;
XE_ELEMENT_BASE<T> temp;
temp.data=value;
temp.empty=false;
temp.original=true;
temp.parent=NULL;
XE_ELEMENT_BASE<T>* _base;
#ifdef XE_ALLOC_C
_base=(XE_ELEMENT_BASE<T>*)malloc(sizeof(XE_ELEMENT_BASE<T>));
#else
_base=new XE_ELEMENT_BASE<T>[1];
#endif /* XE_ALLOC_C */
XE_ELEMENT_POINTER<T> _temp;
if(_base==NULL)
{
first=_first;
last=_last;
return(_temp);
}
*_base=temp;
if(elm.parent==last)
{
temp.next=NULL;
temp.prev=last;
last->next=_base;
last=_base;
}
if(elm.IsEmpty())
{
temp.next=first;
temp.prev=NULL;
first->prev=_base;
first=_base;
}
if(elm.parent!=last&&!elm.IsEmpty())
{
temp.next=elm.parent->next;
temp.prev=elm.parent;
elm.parent->next=_base;
}
_temp.empty=false;
_temp.original=false;
_temp.parent=_base;
_temp.parent->container=this;
return(_temp);
}
}

template<typename T>
void XE_ELEMENTS<T>::ErasePointer(XE_ELEMENT_POINTER<T>& elm)
{
if(size>0)
if(!elm.empty)
if(elm.parent!=NULL)
{
size--;
XE_ELEMENT_BASE<T>* temp=elm.parent;
XE_ELEMENT_BASE<T>* _prev=temp->prev;
XE_ELEMENT_BASE<T>* _next=temp->next;
if(_prev!=NULL)_prev->next=_next;
if(_prev==NULL)first=_next;
if(_next!=NULL)_next->prev=_prev;
if(_next==NULL)last=_prev;
#ifdef XE_ALLOC_C
free(temp);
#else
delete[] temp;
#endif /* XE_ALLOC_C */
}
}

/*--- XE_ARRAY ---*/
/*! Klasa kontenera tablicy. */
template<typename T>
class XE_ARRAY
{
public:
enum SEEK
{
SET,
END,
CUR,
RET,
OFF
};
private:
unsigned int size;
T* data;
T* position;
T temporary;
public:
//! Ustala czy przy zniszczeniu obiektu ma nie zwalniac tablicy z pamieci.
bool Static;
/*! Konstruktor domyslny. */
XE_ARRAY();
/*! Konstruktor.
\param count Ilosc elementow do zarezerwowania.
*/
XE_ARRAY(unsigned int count);
/*! Destruktor. */
~XE_ARRAY();
/*! Sprawdza czy kontener jest pusty.
\return true jesli kontener jest pusty, false jesli nie jest.
*/
bool IsEmpty();
/*! Zwraca rozmiar kontenera.
\return Ilosc elementow zdolnych do przechowania.
*/
unsigned int Size();
/*! Zwraca rozmiar kontenera w bajtach.
\return Wielkosc danych zdolnych do przechowania.
*/
unsigned int SizeBytes();
/*! Usuwa pola tablicy i rezerwuje nowa ilosc pol.
\param count Ilosc pol do rezerwacji.
*/
void Reserve(unsigned int count);
/*! Zmienia rozmiar tablicy, kopiujac jej elementy.
\param count Ilosc pol.
\param offset Indeks pola od ktorego zaczyna kopiowac.
*/
void Resize(unsigned int count,unsigned int offset=0);
/*! Usuwa wszystkie pola tablicy. */
void Free();
/*! Zwraca wskaznik na tablice danych.
\param current Ustala czy ma zwrocic wskaznik na bierzaca pozycje kursora w tablicy.
\return Wskaznik na dane tablicy.
*/
T* Get(bool current=false);
/*! Zwraca aktualna pozycje kursora.
\return Pozycja w bajtach.
*/
unsigned int Current();
/*! Sprawdza czy kursor znajduje sie na koncu tablicy.
\return true jesli kursor znajduje sie na koncu tablicy.
*/
bool EndOf();
/*! Podlacza zewnetrzny bufor pod tablice.
\param buffer Wskaznik na bufor danych do podlaczenia.
\param count Liczba elementow bufora do podlaczenia.
*/
void Link(T* buffer,unsigned int count);
/*! Odlacza zewnetrzny bufor od tablicy i rezerwuje nowe pola.
\param count Liczba pol do zarezerwowania.
*/
void Unlink(unsigned int count=0);
/*! Ustala pozycje kursora w buforze tablicy.
\param pos Pozycja.
\param type Typ pozycji. Dostepne wartosci:<br>
SEEK::SET: Indeks pola wzgledem poczatku tablicy.<br>
SEEK::END: Indeks pola wzgledem konca tablicy.<br>
SEEK::CUR: Indeks pola wzgledem aktualnej pozycji kursora.<br>
SEEK::RET: Indeks pola wzgledem aktualnej pozycji kursora, przemieszczany wstecz.<br>
SEEK::OFF: Pozycja w bajtach wzgledem poczatku bufora tablicy.
*/
void Seek(unsigned int pos,SEEK type=SET);
/*! Odczytuje pola tablicy wzgledem aktualnej pozycji kursora.
\param dest Wskaznik na docelowa tablice danych.
\param count Ilosc pol do odczytania.
\return Ilosc odczytanych bajtow.
*/
unsigned int Read(T* dest,unsigned int count=1);
/*! Wczytuje pola tablicy wzgledem aktualnej pozycji kursora.
\param src Wskaznik na zrodlowa tablice danych.
\param count Ilosc pol do wczytania.
\return Ilosc wczytanych bajtow.
*/
unsigned int Write(T* src,unsigned int count=1);
/*! Odczytuje dane tablicy wzgledem aktualnej pozycji kursora.
\param dest Wskaznik na docelowy bufor danych.
\param sizebyte Ilosc bajtow do odczytania.
\return Ilosc odczytanych bajtow.
*/
unsigned int ReadBytes(void* dest,unsigned int sizebyte);
/*! Wczytuje dane tablicy wzgledem aktualnej pozycji kursora.
\param src Wskaznik na zrodlowy bufor danych.
\param sizebyte Ilosc bajtow do wczytania.
\return Ilosc wczytanych bajtow.
*/
unsigned int WriteBytes(void* src,unsigned int sizebyte);
/*! Operator dostepu do pola tablicy. */
T& operator[] (unsigned int index);
};

typedef XE_ARRAY<char> XE_BUFFER;

template<typename T>
XE_ARRAY<T>::XE_ARRAY()
{
data=NULL;
position=data;
size=0;
Static=false;
}

template<typename T>
XE_ARRAY<T>::XE_ARRAY(unsigned int count)
{
data=new T[count];
position=data;
size=count;
Static=false;
}

template<typename T>
XE_ARRAY<T>::~XE_ARRAY()
{
if(!Static)Free();
}

template<typename T>
bool XE_ARRAY<T>::IsEmpty()
{
if(!data||!size)return(true);
return(false);
}

template<typename T>
unsigned int XE_ARRAY<T>::Size()
{
return(size);
}

template<typename T>
unsigned int XE_ARRAY<T>::SizeBytes()
{
return(size*sizeof(T));
}

template<typename T>
void XE_ARRAY<T>::Reserve(unsigned int count)
{
if(!count)
{
if(data)delete[] data;
data=NULL;
position=data;
size=0;
return;
}
if(data)delete[] data;
data=new T[count];
position=data;
size=count;
}

template<typename T>
void XE_ARRAY<T>::Resize(unsigned int count,unsigned int offset)
{
if(!count)
{
if(data)delete[] data;
data=NULL;
position=data;
size=0;
return;
}
T* temp=new T[count];
for(unsigned int i=0;i<min(size,count);i++)
temp[i]=data[min(i+offset,count-1)];
if(data)delete[] data;
data=temp;
position=data;
size=count;
}

template<typename T>
void XE_ARRAY<T>::Free()
{
if(data)delete[] data;
data=NULL;
position=data;
size=0;
}

template<typename T>
T* XE_ARRAY<T>::Get(bool current)
{
if(!current)
return(data);
else
return(position);
}

template<typename T>
unsigned int XE_ARRAY<T>::Current()
{
return((unsigned int)position-(unsigned int)data);
}

template<typename T>
bool XE_ARRAY<T>::EndOf()
{
if(!data||!size)return(true);
if((unsigned int)position<(unsigned int)data||(unsigned int)position>(unsigned int)data+size*sizeof(T))
return(true);
return(false);
}

template<typename T>
void XE_ARRAY<T>::Link(T* buffer,unsigned int count)
{
if(!buffer||!count)return;
data=buffer;
position=data;
size=count;
}

template<typename T>
void XE_ARRAY<T>::Unlink(unsigned int count)
{
if(!count)
{
data=NULL;
position=data;
size=0;
}
else
{
data=new T[count];
position=data;
size=count;
}
}

template<typename T>
void XE_ARRAY<T>::Seek(unsigned int pos,SEEK type)
{
if(!data||!size)return;
if(type==SET)
position=(T*)((unsigned int)data+min(pos,size)*sizeof(T));
else
if(type==END)
position=(T*)((unsigned int)data+size*sizeof(T)-min(pos,size));
else
if(type==CUR)
{
position+=pos*sizeof(T);
if((unsigned int)position<(unsigned int)data)position=data;
if((unsigned int)position>(unsigned int)data+size*sizeof(T))position=(T*)((unsigned int)data+size*sizeof(T));
}
else
if(type==RET)
{
position-=min(pos*sizeof(T),(unsigned int)(position-data));
if((unsigned int)position<(unsigned int)data)position=data;
if((unsigned int)position>(unsigned int)data+size*sizeof(T))position=(T*)((unsigned int)data+size*sizeof(T));
}
else
if(type==OFF)
position=(T*)((unsigned int)data+min(pos,size*sizeof(T)));
}

template<typename T>
unsigned int XE_ARRAY<T>::Read(T* dest,unsigned int count)
{
if(!count||!data||!size)return(0);
unsigned int bytes=min(count*sizeof(T),(unsigned int)data+size*sizeof(T)-(unsigned int)position);
memcpy(dest,position,bytes);
position+=bytes;
return(bytes);
}

template<typename T>
unsigned int XE_ARRAY<T>::Write(T* src,unsigned int count)
{
if(!count||!data||!size)return(0);
unsigned int bytes=min(count*sizeof(T),(unsigned int)data+size*sizeof(T)-(unsigned int)position);
memcpy(position,src,bytes);
position+=bytes;
return(bytes);
}

template<typename T>
unsigned int XE_ARRAY<T>::ReadBytes(void* dest,unsigned int sizebyte)
{
if(!sizebyte||!data||!size)return(0);
unsigned int bytes=min(sizebyte,(unsigned int)data+size*sizeof(T)-(unsigned int)position);
memcpy(dest,position,bytes);
position+=bytes;
return(bytes);
}

template<typename T>
unsigned int XE_ARRAY<T>::WriteBytes(void* src,unsigned int sizebyte)
{
if(!sizebyte||!data||!size)return(0);
unsigned int bytes=min(sizebyte,(unsigned int)data+size*sizeof(T)-(unsigned int)position);
memcpy(position,src,bytes);
position+=bytes;
return(bytes);
}

template<typename T>
T& XE_ARRAY<T>::operator[] (unsigned int index)
{
if(!data||!size||index>=size)return(temporary);
return(data[index]);
}

/*--- XE_NODE ---*/
/*! Klasa wezla. */
template<typename T>
class XE_NODE
{
public:
typedef XE_ELEMENT_POINTER<XE_NODE<T> > NODE;
private:
T Data;
XE_NODE<T>* Parent;
XE_ELEMENTS<XE_NODE<T> > Childs;
public:
/*! Konstruktor domyslny. */
XE_NODE();
/*! Konstruktor wypelniajacy. */
XE_NODE(const T& val);
/*! Destruktor. */
~XE_NODE();
/*! Zwraca klon danych wezla.
\return Klon danych wezla.
*/
T Clone();
/*! Zwraca wskaznik na dane wezla.
\return Wskaznik na dane wezla.
*/
T* Pointer();
/*! Zwraca wskaznik na rodzica wezla.
\return Wskaznik na rodzica wezla.
*/
XE_NODE<T>* GetParent();
/*! Zwraca wezel do pierwszego dziecka.
\return Wezel.
*/
NODE FirstChild();
/*! Zwraca wezel do ostatniego dziecka.
\return Wezel.
*/
NODE LastChild();
/*! Dodaje dziecko i zwraca wezel do niego.
\param value Wartosc danych wezla.
\return Wezel dodanego dziecka.
*/
NODE AddChild(const T& value);
/*! Usuwa dziecko.
\param child Element wezla dziecka.
*/
void RemoveChild(NODE child);
/*! Usuwa dziecko o danym wskazniku na dane wezla.
\param child Wskaznik na dane wezla dziecka.
*/
void RemoveChildPtr(XE_NODE<T>* child);
/*! Usuwa siebie z wezlow rodzica. */
void RemoveSelf();
/*! Zwraca liczbe dzieci.
\return Ilosc wezlow dzieci.
*/
unsigned int Count();
/*! Usuwa wszystkie dzieci. */
void Free();
/*! Przechwytuje dziecko z innego wezla.
\param child Wezel dziecka do przechwycenia.
\return true jesli operacja powiodla sie.
*/
bool CaptureChild(NODE child);
/*! Przechwytuje dziecko z innego wezla o danym wskazniku na dane wezla.
\param child Wskaznik na dane wezla dziecka.
\return true jesli operacja powiodla sie.
*/
bool CaptureChildPtr(XE_NODE<T>* child);
/*! Sortuje dzieci wedle funkcji foo przyjmujacej wskazniki na dwoje sasiednich dzieci i zwracajaca 1 jesli kolejnosc jest poprawna, 0 jesli kolejnosc jest ta sama i -1 jesli kolejnosc jest zla i trzeba ja zamienic.
\param foo Wskaznik na funkcje sortujaca: int foo(XE_NODE<T>* a,XE_NODE<T>* b).
*/
void SortChilds(int(*foo)(XE_NODE<T>*,XE_NODE<T>*));
/*! Operator wyluskania.
\return Wskaznik na dane wezla.
*/
T* operator-> ();
};

template<typename T>
XE_NODE<T>::XE_NODE()
{
Parent=NULL;
}

template<typename T>
XE_NODE<T>::XE_NODE(const T& val)
{
Data=val;
Parent=NULL;
}

template<typename T>
XE_NODE<T>::~XE_NODE()
{
Free();
}

template<typename T>
T XE_NODE<T>::Clone()
{
return(Data);
}

template<typename T>
T* XE_NODE<T>::Pointer()
{
return(&Data);
}

template<typename T>
XE_NODE<T>* XE_NODE<T>::GetParent()
{
return(Parent);
}

template<typename T>
XE_ELEMENT_POINTER<XE_NODE<T> > XE_NODE<T>::FirstChild()
{
return(Childs.FirstPointer());
}

template<typename T>
XE_ELEMENT_POINTER<XE_NODE<T> > XE_NODE<T>::LastChild()
{
return(Childs.LastPointer());
}

template<typename T>
XE_ELEMENT_POINTER<XE_NODE<T> > XE_NODE<T>::AddChild(const T& value)
{
NODE elm=Childs.AddPointer(XE_NODE<T>(value));
if(!elm.IsEmpty())elm->Parent=this;
return(elm);
}

template<typename T>
void XE_NODE<T>::RemoveChild(NODE child)
{
if(child->Parent!=this)return;
child->Free();
Childs.ErasePointer(child);
}

template<typename T>
void XE_NODE<T>::RemoveChildPtr(XE_NODE<T>* child)
{
if(!child)return;
for(NODE elm=Childs.FirstPointer();!elm.IsEmpty();elm.Next())
if(elm.iPointer()==child)
{
if(elm->Parent!=this)break;
elm->Free();
Childs.ErasePointer(elm);
break;
}
}

template<typename T>
void XE_NODE<T>::RemoveSelf()
{
if(!Parent)return;
Parent->RemoveChildPtr(this);
}

template<typename T>
unsigned int XE_NODE<T>::Count()
{
return(Childs.Size());
}

template<typename T>
void XE_NODE<T>::Free()
{
XE_ELEMENT_POINTER<XE_NODE<T> > elm=Childs.FirstPointer();
while(!elm.IsEmpty())
{
elm->Free();
Childs.ErasePointer(elm);
elm=Childs.FirstPointer();
}
}

template<typename T>
bool XE_NODE<T>::CaptureChild(NODE child)
{

XE_NODE<T>* par=Parent;
while(par)
{
if(par==child.iPointer())
{
XE_NODE<T>* tpar=child->Parent;
if(!tpar)return(false);
while(child->Count())
{
NODE tn=child->FirstChild();
if(!tpar->Childs.Capture(tn))return(false);
if(!tn.IsEmpty())tn->Parent=tpar;
}
if(Childs.Capture(child))
{
if(!child.IsEmpty())child->Parent=this;
return(true);
}
}
par=par->Parent;
}

if(Childs.Capture(child))
{
if(!child.IsEmpty())child->Parent=this;
return(true);
}
return(false);
}

template<typename T>
bool XE_NODE<T>::CaptureChildPtr(XE_NODE<T>* child)
{
if(!child)return(false);
if(!child->GetParent())return(false);
for(NODE elm=child->GetParent()->FirstChild();!elm.IsEmpty();elm.Next())
if(elm.iPointer()==child)
{

XE_NODE<T>* par=Parent;
while(par)
{
if(par==child)
{
XE_NODE<T>* tpar=child->Parent;
if(!tpar)return(false);
while(child->Count())
{
NODE tn=child->FirstChild();
if(!tpar->Childs.Capture(tn))return(false);
if(!tn.IsEmpty())tn->Parent=tpar;
}
if(Childs.Capture(elm))
{
if(!elm.IsEmpty())elm->Parent=this;
return(true);
}
}
par=par->Parent;
}
if(Childs.Capture(elm))
{
if(!elm.IsEmpty())child->Parent=this;
return(true);
}
}
return(false);
}

template<typename T>
void XE_NODE<T>::SortChilds(int(*foo)(XE_NODE<T>*,XE_NODE<T>*))
{
Childs.Sort(foo);
}

template<typename T>
T* XE_NODE<T>::operator-> ()
{
return(Pointer());
}

/*--- XE_NUMBER ---*/
/*! Klasa liczby.
Bufor wartości liczbowej.
*/
class XE_NUMBER
{
private:
double Value;
public:
/*! Konstruktor domyślny. */
XE_NUMBER();
/*! Konstruktor wypelniajacy. */
XE_NUMBER(double val);
/*! Konstruktor wypelniajacy. */
XE_NUMBER(int val);
/*! Zwraca double. */
double Get();
/*! Ustala wartość. */
void Set(double val);
//! operator dodawania
XE_NUMBER operator+ (XE_NUMBER& n);
//! operator odejmowania
XE_NUMBER operator- (XE_NUMBER& n);
//! operator mnozenia
XE_NUMBER operator* (XE_NUMBER& n);
//! operator dzielenia
XE_NUMBER operator/ (XE_NUMBER& n);
//! operator dodawania
XE_NUMBER& operator+= (XE_NUMBER& n);
//! operator odejmowania
XE_NUMBER& operator-= (XE_NUMBER& n);
//! operator mnozenia
XE_NUMBER& operator*= (XE_NUMBER& n);
//! operator dzielenia
XE_NUMBER& operator/= (XE_NUMBER& n);
/*! Operator przypisania.
Ustala nowa wartosc.
\param val Wartosc double.
*/
double& operator= (double& val);
/*! Operator przypisania.
Ustala nowa wartosc.
\param val Wartosc float.
*/
float& operator= (float& val);
/*! Operator przypisania.
Ustala nowa wartosc.
\param val Wartosc int.
*/
int& operator= (int& val);
/*! Operator przypisania.
Ustala nowa wartosc.
\param val Wartosc unsigned int.
*/
unsigned int& operator= (unsigned int& val);
/*! Operator przypisania.
Ustala nowa wartosc.
\param val Wartosc short.
*/
short& operator= (short& val);
/*! Operator przypisania.
Ustala nowa wartosc.
\param val Wartosc unsigned short.
*/
unsigned int& operator= (unsigned short& val);
/*! Operator przypisania.
Ustala nowa wartosc.
\param val Wartosc byte (char).
*/
char& operator= (char& val);
/*! Operator przypisania.
Ustala nowa wartosc.
\param val Wartosc unsigned byte (char).
*/
unsigned char& operator= (unsigned char& val);
/*! Operator przypisania.
Ustala nowa wartosc.
\param val Wartosc bool.
*/
bool& operator= (bool& val);
/*! Operator porownania.
Sprawdza czy wartosci sa rowne.
\param val Wartosc liczbowa.
\return true jesli wartosci sa rowne.
*/
bool operator== (XE_NUMBER& val);
/*! Operator porownania.
Sprawdza czy wartosci sa rowne.
\param val Wartosc double.
\return true jesli wartosci sa rowne.
*/
bool operator== (double& val);
/*! Operator porownania.
Sprawdza czy wartosci sa rowne.
\param val Wartosc float.
\return true jesli wartosci sa rowne.
*/
bool operator== (float& val);
/*! Operator porownania.
Sprawdza czy wartosci sa rowne.
\param val Wartosc int.
\return true jesli wartosci sa rowne.
*/
bool operator== (int& val);
/*! Operator porownania.
Sprawdza czy wartosci sa rowne.
\param val Wartosc unsigned int.
\return true jesli wartosci sa rowne.
*/
bool operator== (unsigned int& val);
/*! Operator porownania.
Sprawdza czy wartosci sa rowne.
\param val Wartosc short.
\return true jesli wartosci sa rowne.
*/
bool operator== (short& val);
/*! Operator porownania.
Sprawdza czy wartosci sa rowne.
\param val Wartosc unsigned short.
\return true jesli wartosci sa rowne.
*/
bool operator== (unsigned short& val);
/*! Operator porownania.
Sprawdza czy wartosci sa rowne.
\param val Wartosc byte (char).
\return true jesli wartosci sa rowne.
*/
bool operator== (char& val);
/*! Operator porownania.
Sprawdza czy wartosci sa rowne.
\param val Wartosc unsigned byte (char).
\return true jesli wartosci sa rowne.
*/
bool operator== (unsigned char& val);
/*! Operator porownania.
Sprawdza czy wartosci sa rowne.
\param val Wartosc bool.
\return true jesli wartosci sa rowne.
*/
bool operator== (bool& val);
/*! Operator porownania.
Sprawdza czy wartosci sa rozne.
\param val Wartosc liczbowa.
\return true jesli wartosci sa rowne.
*/
bool operator!= (XE_NUMBER& val);
/*! Operator porownania.
Sprawdza czy wartosci sa rozne.
\param val Wartosc double.
\return true jesli wartosci sa rowne.
*/
bool operator!= (double& val);
/*! Operator porownania.
Sprawdza czy wartosci sa rozne.
\param val Wartosc float.
\return true jesli wartosci sa rowne.
*/
bool operator!= (float& val);
/*! Operator porownania.
Sprawdza czy wartosci sa rozne.
\param val Wartosc int.
\return true jesli wartosci sa rowne.
*/
bool operator!= (int& val);
/*! Operator porownania.
Sprawdza czy wartosci sa rozne.
\param val Wartosc unsigned int.
\return true jesli wartosci sa rowne.
*/
bool operator!= (unsigned int& val);
/*! Operator porownania.
Sprawdza czy wartosci sa rozne.
\param val Wartosc short.
\return true jesli wartosci sa rowne.
*/
bool operator!= (short& val);
/*! Operator porownania.
Sprawdza czy wartosci sa rozne.
\param val Wartosc unsigned short.
\return true jesli wartosci sa rowne.
*/
bool operator!= (unsigned short& val);
/*! Operator porownania.
Sprawdza czy wartosci sa rozne.
\param val Wartosc byte (char).
\return true jesli wartosci sa rowne.
*/
bool operator!= (char& val);
/*! Operator porownania.
Sprawdza czy wartosci sa rozne.
\param val Wartosc unsigned byte (char).
\return true jesli wartosci sa rowne.
*/
bool operator!= (unsigned char& val);
/*! Operator porownania.
Sprawdza czy wartosci sa rozne.
\param val Wartosc bool.
\return true jesli wartosci sa rowne.
*/
bool operator!= (bool& val);
/*! Operator konwersji.
Rzutuje wartosc na double.
*/
operator double() const;
/*! Operator konwersji.
Rzutuje wartosc na float.
*/
operator float() const;
/*! Operator konwersji.
Rzutuje wartosc na int.
*/
operator int() const;
/*! Operator konwersji.
Rzutuje wartosc na unsigned int.
*/
operator unsigned int() const;
/*! Operator konwersji.
Rzutuje wartosc na short.
*/
operator short() const;
/*! Operator konwersji.
Rzutuje wartosc na unsigned short.
*/
operator unsigned short() const;
/*! Operator konwersji.
Rzutuje wartosc na byte (char).
*/
operator char() const;
/*! Operator konwersji.
Rzutuje wartosc na unsigned byte (char).
*/
operator unsigned char() const;
/*! Operator konwersji.
Rzutuje wartosc na bool.
*/
operator bool() const;
};

#ifdef XE_CANIMP

XE_NUMBER::XE_NUMBER()
{
Value=0;
}

XE_NUMBER::XE_NUMBER(double val)
{
Value=val;
}

XE_NUMBER::XE_NUMBER(int val)
{
Value=(double)val;
}

double XE_NUMBER::Get()
{
return(Value);
}

void XE_NUMBER::Set(double val)
{
Value=val;
}

XE_NUMBER XE_NUMBER::operator+ (XE_NUMBER& n)
{
return(Value+n.Value);
}

XE_NUMBER XE_NUMBER::operator- (XE_NUMBER& n)
{
return(Value-n.Value);
}

XE_NUMBER XE_NUMBER::operator* (XE_NUMBER& n)
{
return(Value*n.Value);
}

XE_NUMBER XE_NUMBER::operator/ (XE_NUMBER& n)
{
return(Value/n.Value);
}

XE_NUMBER& XE_NUMBER::operator+= (XE_NUMBER& n)
{
*this=*this+n;
return(*this);
}

XE_NUMBER& XE_NUMBER::operator-= (XE_NUMBER& n)
{
*this=*this-n;
return(*this);
}

XE_NUMBER& XE_NUMBER::operator*= (XE_NUMBER& n)
{
*this=*this*n;
return(*this);
}

XE_NUMBER& XE_NUMBER::operator/= (XE_NUMBER& n)
{
*this=*this/n;
return(*this);
}

double& XE_NUMBER::operator= (double& val)
{
Set(val);
return(Value);
}

float& XE_NUMBER::operator= (float& val)
{
Set((double&)val);
return((float&)Value);
}

int& XE_NUMBER::operator= (int& val)
{
Set((double&)val);
return((int&)Value);
}

unsigned int& XE_NUMBER::operator= (unsigned int& val)
{
Set((double&)val);
return((unsigned int&)Value);
}

char& XE_NUMBER::operator= (char& val)
{
Set((double&)val);
return((char&)Value);
}

unsigned char& XE_NUMBER::operator= (unsigned char& val)
{
Set((double&)val);
return((unsigned char&)Value);
}

bool& XE_NUMBER::operator= (bool& val)
{
Set((double&)val);
return((bool&)Value);
}

bool XE_NUMBER::operator== (XE_NUMBER& val)
{
return((Value==val.Value)?true:false);
}

bool XE_NUMBER::operator== (double& val)
{
return((Value==val)?true:false);
}

bool XE_NUMBER::operator== (float& val)
{
return((Value==(double)val)?true:false);
}

bool XE_NUMBER::operator== (int& val)
{
return((Value==(double)val)?true:false);
}

bool XE_NUMBER::operator== (unsigned int& val)
{
return((Value==(double)val)?true:false);
}

bool XE_NUMBER::operator== (short& val)
{
return((Value==(double)val)?true:false);
}

bool XE_NUMBER::operator== (unsigned short& val)
{
return((Value==(double)val)?true:false);
}

bool XE_NUMBER::operator== (char& val)
{
return((Value==(double)val)?true:false);
}

bool XE_NUMBER::operator== (unsigned char& val)
{
return((Value==(double)val)?true:false);
}

bool XE_NUMBER::operator== (bool& val)
{
return((Value==(double)val)?true:false);
}

bool XE_NUMBER::operator!= (XE_NUMBER& val)
{
return((Value!=val.Value)?true:false);
}

bool XE_NUMBER::operator!= (double& val)
{
return((Value!=val)?true:false);
}

bool XE_NUMBER::operator!= (float& val)
{
return((Value!=(double)val)?true:false);
}

bool XE_NUMBER::operator!= (int& val)
{
return((Value!=(double)val)?true:false);
}

bool XE_NUMBER::operator!= (unsigned int& val)
{
return((Value!=(double)val)?true:false);
}

bool XE_NUMBER::operator!= (short& val)
{
return((Value!=(double)val)?true:false);
}

bool XE_NUMBER::operator!= (unsigned short& val)
{
return((Value!=(double)val)?true:false);
}

bool XE_NUMBER::operator!= (char& val)
{
return((Value!=(double)val)?true:false);
}

bool XE_NUMBER::operator!= (unsigned char& val)
{
return((Value!=(double)val)?true:false);
}

bool XE_NUMBER::operator!= (bool& val)
{
return((Value!=(double)val)?true:false);
}

XE_NUMBER::operator double() const
{
return(Value);
}

XE_NUMBER::operator float() const
{
return((float)Value);
}

XE_NUMBER::operator int() const
{
return((int)Value);
}

XE_NUMBER::operator unsigned int() const
{
return((unsigned int)Value);
}

XE_NUMBER::operator short() const
{
return((short)Value);
}

XE_NUMBER::operator unsigned short() const
{
return((unsigned short)Value);
}

XE_NUMBER::operator char() const
{
return((char)Value);
}

XE_NUMBER::operator unsigned char() const
{
return((unsigned char)Value);
}

XE_NUMBER::operator bool() const
{
return(Value?true:false);
}

#endif /* XE_CANIMP */

/*--- XE_STRING ---*/
/*! Klasa stringa.
Bufor wartosci tekstowych.
*/
class XE_STRING
{
private:
char* Text;
public:
/*! Ustala czy pamiec bufora jest statyczna (jesli true to podczas niszczenia stringa nie zostaje zwalniana pamiec bufora). */
bool Static;
/*! Konstruktor domyslny. */
XE_STRING();
/*! Konstruktor wypelniajacy. */
XE_STRING(const XE_STRING& str);
/*! Konstruktor wypelniajacy. */
XE_STRING(char* str);
/*! Konstruktor wypelniajacy. */
XE_STRING(wchar_t* str);
/*! Destruktor. */
~XE_STRING();
/*! Zwraca znak w buforze tekstu.
\param index Pozycja w buforze tekstu.
\param invert Okresla czy indeks zaczyna sie od konca tekstu.
\return Cstring zawartości bufora tekstu.
*/
char Index(unsigned int index,bool invert=false);
/*! Zwraca cstring.
\param index Pozycja w buforze tekstu.
\param invert Okresla czy indeks zaczyna sie od konca tekstu.
\return Cstring zawartości bufora tekstu.
*/
char* Get(unsigned int index=0,bool invert=false);
/*! Zwraca wcstring.
\param result Wskaznik na docelowy bufor Wcstring.
\param buffsize Rozmiar bufora w bajtach.
\return true jesli operacja powiodla sie.
*/
bool GetUnicode(wchar_t* result,unsigned int buffsize);
/*! Kopiuje lancuch znakow do bufora tekstu.
\param str Wskaznik na cstring.
*/
void Set(char* str);
/*! Kopiuje lancuch znakow do bufora tekstu.
\param str Wskaznik na wcstring.
*/
void Set(wchar_t* str);
/*! Kopiuje bufor znakow do bufora tekstu.
\param src Zrodlowy bufor znakow.
\param size Rozmiar bufora znakow (w bajtach).
*/
void Copy(char* src,unsigned int size);
/*! Ustala zawartosc bufora tekstu formatowanego.
\param str Wskaznik na cstring formatu.
\param ... Kolejne wartosci.
*/
void Format(char* str,...);
/*! Rozdziela zawartosc bufora tekstu w miejscu znakow sep i wysyla je do tablicy dynamicznej buforow tekstu. Wazne aby po uzyciu zwolnic tablice z pamieci uzywajac operatora: delete[].
\param sep Wskaznik na cstring ze znakami rozdzielajacymi.
\param count Referencja na wynik operacji (liczba elementow utworzonej tablicy).
\param block Wskaznikna cstring ze znakami oznaczajacymi blok tekstu bez rozdzielania.
\return Wskaznik na tablice dynamiczna buforow tekstu jelsi operacja powiodla sie, lub wartosc NULL jesli nie.
*/
XE_STRING* Explode(char* sep,unsigned int& count,char* block=0);
/*! Rozdziela zawartosc bufora w miejscu znakow sep i wysyla je do tablicy dynamicznej buforow tekstu. Wazne aby po uzyciu zwolnic tablice z pamieci uzywajac operatora: delete[].
\param buff Wskaznik na bufor.
\param size Rozmiar bufora w bajtach.
\param sep Wskaznik na cstring ze znakami rozdzielajacymi.
\param count Referencja na wynik operacji (liczba elementow utworzonej tablicy).
\param block Wskaznikna cstring ze znakami oznaczajacymi blok tekstu bez rozdzielania.
\return Wskaznik na tablice dynamiczna buforow tekstu jelsi operacja powiodla sie, lub wartosc NULL jesli nie.
*/
static XE_STRING* ExplodeBuff(char* buff,unsigned int size,char* sep,unsigned int& count,char* block=0);
/*! Konwertuje liczbe w postaci tekstu na liczbe numeryczna.
\param out Wskaznik na zmienna do ktorej zostanie zapisany wynik.
\param deftype Jesli rozne od 0, ustala jakiego typu ma byc liczba bez symbolu.
\return true jesli operacja powiodla sie,
*/
bool Convert(void* out,int deftype=0);
/*! Zwraca typ liczby.
\return Typ (rozszerzenie) liczby.
*/
int ConvertType();
/*! Zwraca rozmiar liczby.
\return Rozmiar liczby w bajtach.
*/
unsigned int ConvertSize();
/*! Zamienia slowa formatowalne ("\") na odpowiednie znaki. */
void EscapeSeq();
/*! Zwraca dlugosc tekstu.
\return Rozmiar bufora tekstu.
*/
unsigned int Length();
/*! Zwraca dlugosc tekstu dla konwersji unicode.
\param extrachars Liczba dodatkowych znakow.
\return Rozmiar bufora tekstu unicode.
*/
unsigned int LengthUnicode(unsigned int extrachars=0);
/*! Szuka dany ciag znakow w stringu i zwraca jego pozycje.
\param offset Poczatkowa pozycja odczytu.
\param substr Szukany ciag znakow.
\param position Wskaznik na zwracana pozycje w stringu (jesli rozny od NULL).
\return true jesli operacja powiodla sie.
*/
bool Find(unsigned int offset,char* substr,unsigned int* position=NULL);
/*! Szuka i pobiera ciag znakow pomiedzy poczatkiem a pierwszym znalezionym ciagiem znakow rozdzielajacych.
\param result Referencja na wynikowy string.
\param delim Ciag znakow rozdzielajacych.
\param offset Poczatkowa pozycja odczytu.
\return Pozycja na ktorej zakonczyl odczyt, od ktorej powinien szukac dalej, lub 0 jesli nie znaleziono ciagu, badz dalsze poszukiwania sa niemozliwe..
*/
unsigned int Search(XE_STRING& result,char* delim,unsigned int offset=0);
/*! Zamienia wszystkie stare ciagi znakow na nowe.
\param result Referencja na wynikowy string.
\param oldstr Stary ciag znakow.
\param newstr Nowy ciag znakow.
\return Ilosc zamienionych ciagow znakow.
*/
unsigned int Replace(XE_STRING& result,char* oldstr,char* newstr);
/*! Zamienia wszystkie stare znaki na nowe.
\param result Referencja na wynikowy string.
\param oldchar Stary znak.
\param newchar Nowy znak.
\return Ilosc zamienionych znakow.
*/
unsigned int ReplaceChars(XE_STRING& result,char oldchar,char newchar);
/*! Czysci bufor tekstu. */
void Clear();
/*! Operator przypisania.
Ustala nowa zawartosc bufora tekstu.
\param str Referencja na string.
*/
void operator= (const XE_STRING& str);
/*! Operator przypisania.
Ustala nowa zawartosc bufora tekstu.
\param str Wskaznik na cstring.
*/
void operator= (char* str);
/*! Operator konwersji.
Rzutuje zawartosc bufora tekstu na cstring.
*/
operator char*() const;
/*! Operator strumieniowania.
Konwertuje i dodaje wartosc double do bufora tekstu.
\param value Wartosc do dodania.
\return Referencja na string str.
*/
XE_STRING& operator<< (double value);
/*! Operator strumieniowania.
Konwertuje i dodaje wartosc int do bufora tekstu.
\param value Wartosc do dodania.
\return Referencja na string str.
*/
XE_STRING& operator<< (int value);
/*! Operator strumieniowania.
Konwertuje i dodaje wartosc unsigned int do bufora tekstu.
\param value Wartosc do dodania.
\return Referencja na string str.
*/
XE_STRING& operator<< (unsigned int value);
/*! Operator strumieniowania.
Dodaje cstring do bufora tekstu.
\param value Wskaznik na cstring.
\return Referencja na string str.
*/
XE_STRING& operator<< (char* value);
/*! Operator porownania.
Sprawdza czy oba stringi sa rowne.
\param str String do porownania.
\return true jesli oba stringi sa rowne.
*/
bool operator== (XE_STRING& str);
/*! Operator porownania.
Sprawdza czy oba stringi sa rozne.
\param str String do porownania.
\return true jesli oba stringi sa rozne.
*/
bool operator!= (XE_STRING& str);
/*! Operator porownania.
Sprawdza czy string i cstring sa rowne.
\param str String do porownania.
\return true jesli string i cstring sa rowne.
*/
bool operator== (char* str);
/*! Operator porownania.
Sprawdza czy string i cstring sa rozne.
\param str String do porownania.
\return true jesli string i cstring sa rozne.
*/
bool operator!= (char* str);
};

#define XeEndL "\r\n"
#define XeEndS "\0"

#ifdef XE_CANIMP

XE_STRING::XE_STRING()
{
#ifdef XE_ALLOC_C
Text=(char*)malloc(sizeof(char)*1);
#else
Text=new char[1];
#endif /* XE_ALLOC_C */
Text[0]=0;
Static=false;
}

XE_STRING::XE_STRING(const XE_STRING& str)
{
#ifdef XE_ALLOC_C
Text=(char*)malloc(sizeof(char)*1);
#else
Text=new char[1];
#endif /* XE_ALLOC_C */
Text[0]=0;
Static=false;
Set(str);
}

XE_STRING::XE_STRING(char* str)
{
#ifdef XE_ALLOC_C
Text=(char*)malloc(sizeof(char)*1);
#else
Text=new char[1];
#endif /* XE_ALLOC_C */
Text[0]=0;
Static=false;
Set(str);
}

XE_STRING::XE_STRING(wchar_t* str)
{
#ifdef XE_ALLOC_C
Text=(char*)malloc(sizeof(char)*1);
#else
Text=new char[1];
#endif /* XE_ALLOC_C */
Text[0]=0;
Static=false;
Set(str);
}

XE_STRING::~XE_STRING()
{
if(!Static)
{
if(Text!=NULL)
#ifdef XE_ALLOC_C
free(Text);
#else
delete[] Text;
#endif /* XE_ALLOC_C */
Text=NULL;
}
}

char XE_STRING::Index(unsigned int index,bool invert)
{
if(!Text)return(0);
unsigned int len=strlen(Text);
if(!len)return(Text[0]);
if(!invert)
return((Text+min(index,len))[0]);
else
return((Text+len-1-min(index,len))[0]);
}

char* XE_STRING::Get(unsigned int index,bool invert)
{
if(!Text)return(0);
unsigned int len=strlen(Text);
if(!len)return(Text);
if(!invert)
return(Text+min(index,len));
else
return(Text+len-1-min(index,len));
}

bool XE_STRING::GetUnicode(wchar_t* result,unsigned int buffsize)
{
if(!result||!buffsize)return(false);
unsigned int len=strlen(Text);
mbstowcs_s(0,result,buffsize,Text,len);
return(true);
}

void XE_STRING::Set(char* str)
{
if(!str)return;
unsigned int len=strlen(str);
#ifdef XE_ALLOC_C
if(Text)free(Text);
Text=(char*)malloc(sizeof(char)*(len+1));
#else
if(Text)delete[] Text;
Text=new char[len+1];
#endif /* XE_ALLOC_C */
Text[0]=0;
strcpy_s(Text,len+1,str);
Text[len]=0;
}

void XE_STRING::Set(wchar_t* str)
{
if(!str)return;
unsigned int len=wcslen(str);
#ifdef XE_ALLOC_C
if(Text)free(Text);
Text=(char*)malloc(sizeof(char)*(len+1));
#else
if(Text)delete[] Text;
Text=new char[len+1];
#endif /* XE_ALLOC_C */
Text[0]=0;
wcstombs_s(0,Text,len+1,str,len);
Text[len]=0;
}

void XE_STRING::Copy(char* src,unsigned int size)
{
if(!src)return;
unsigned int len=min(size,strlen(src));
#ifdef XE_ALLOC_C
if(Text)free(Text);
Text=(char*)malloc(sizeof(char)*(len+1));
#else
if(Text)delete[] Text;
Text=new char[len+1];
#endif /* XE_ALLOC_C */
Text[0]=0;
memcpy(Text,src,len);
Text[len]=0;
}

void XE_STRING::Format(char* str,...)
{
unsigned int count=0;
va_list vl;
va_start(vl,str);
count=_vscprintf(str,vl);
va_end(vl);
#ifdef XE_ALLOC_C
if(Text)free(Text);
Text=(char*)malloc(sizeof(char)*(count+1));
#else
if(Text)delete[] Text;
Text=new char[count+1];
#endif /* XE_ALLOC_C */
Text[0]=0;
va_start(vl,str);
vsprintf_s(Text,count+1,str,vl);
va_end(vl);
}

XE_STRING* XE_STRING::Explode(char* sep,unsigned int& count,char* block)
{
XE_STRING str;
str.Set(Text);
count=0;
unsigned int len=strlen(str.Get());
unsigned int slen=strlen(sep);
unsigned int ulen=0;
if(block)ulen=strlen(block);
unsigned int num=0;
unsigned int lastp=0;
bool un=false;
bool fi=true;
for(unsigned int i=0;i<len;i++)
{
for(unsigned int k=0;k<ulen;k++)
if(str.Get()[i]==block[k])un=!un;
if(un)continue;
for(unsigned int j=0;j<slen;j++)
if(str.Get()[i]==sep[j]||str.Get()[i]==0)
{
if(i-lastp>1||(fi&&i>0))
num++;
lastp=i;
if(fi)fi=false;
break;
}
}
XE_STRING* result=new XE_STRING[num+1];
char* last=str;
lastp=0;
un=false;
fi=true;
for(unsigned int i=0;i<len+1;i++)
{
for(unsigned int k=0;k<ulen;k++)
if(str.Get()[i]==block[k])un=!un;
if(un)continue;
for(unsigned int j=0;j<slen;j++)
if(str.Get()[i]==sep[j]||str.Get()[i]==0)
{
if(i-lastp>1||(fi&&i>0))
{
str.Get()[i]=0;
result[count++].Set(last);
}
lastp=i;
last=&str.Get()[i+1];
if(fi)fi=false;
break;
}
}
return(result);
}

XE_STRING* XE_STRING::ExplodeBuff(char* buff,unsigned int size,char* sep,unsigned int& count,char* block)
{
count=0;
unsigned int slen=strlen(sep);
unsigned int ulen=0;
if(block)ulen=strlen(block);
unsigned int num=0;
unsigned int lastp=0;
bool un=false;
bool fi=true;
for(unsigned int i=0;i<size;i++)
{
for(unsigned int k=0;k<ulen;k++)
if(buff[i]==block[k])un=!un;
if(un)continue;
for(unsigned int j=0;j<slen;j++)
if(buff[i]==sep[j]||buff[i]==0)
{
if(i-lastp>1||(fi&&i>0))
num++;
lastp=i;
if(fi)fi=false;
break;
}
}
XE_STRING* result=new XE_STRING[num+1];
char* last=buff;
lastp=0;
un=false;
fi=true;
for(unsigned int i=0;i<size+1;i++)
{
for(unsigned int k=0;k<ulen;k++)
if(buff[i]==block[k])un=!un;
if(un)continue;
for(unsigned int j=0;j<slen;j++)
if(buff[i]==sep[j]||buff[i]==0)
{
if(i-lastp>1||(fi&&i>0))
{
char chr=buff[i];
buff[i]=0;
result[count++].Set(last);
buff[i]=chr;
}
lastp=i;
last=&buff[i+1];
if(fi)fi=false;
break;
}
}
return(result);
}

bool XE_STRING::Convert(void* out,int deftype)
{
unsigned int len=strlen(Text);
if((deftype&&len<=0)||(!deftype&&len<=1))return(false);
char type=Text[len-1];
if(deftype)type=deftype;
if(type=='x')
{
unsigned int sp=0;
unsigned char val=0;
unsigned int start=len-1;
if(deftype)start=len;
bool g=0;
for(unsigned int p=start;p>0;p--,g=!g)
{
if(g==0)((unsigned char*)out)[sp]=0;
if(Text[p-1]>='0'&&Text[p-1]<='9')
val=Text[p-1]-'0';
else
if(Text[p-1]>='a'&&Text[p-1]<='f')
val=Text[p-1]-'a'+10;
else
if(Text[p-1]>='A'&&Text[p-1]<='F')
val=Text[p-1]-'A'+10;
else
val=0;
((unsigned char*)out)[sp]|=val<<4*g;
if(g==1)sp++;
}
return(true);
}
else
if(type=='c')
{
if(!deftype)Text[len-1]=0;
*(char*)out=(char)strtol(Text,NULL,0);
if(!deftype)Text[len-1]='c';
return(true);
}
else
if(type=='b')
{
if(!deftype)Text[len-1]=0;
*(unsigned char*)out=(unsigned char)strtoul(Text,NULL,0);
if(!deftype)Text[len-1]='b';
return(true);
}
else
if(type=='s')
{
if(!deftype)Text[len-1]=0;
*(short*)out=(short)strtol(Text,NULL,0);
if(!deftype)Text[len-1]='s';
return(true);
}
else
if(type=='w')
{
if(!deftype)Text[len-1]=0;
*(unsigned short*)out=(unsigned short)strtoul(Text,NULL,0);
if(!deftype)Text[len-1]='w';
return(true);
}
else
if(type=='i')
{
if(!deftype)Text[len-1]=0;
*(int*)out=(int)strtol(Text,NULL,0);
if(!deftype)Text[len-1]='i';
return(true);
}
else
if(type=='d')
{
if(!deftype)Text[len-1]=0;
*(unsigned int*)out=(unsigned int)strtoul(Text,NULL,0);
if(!deftype)Text[len-1]='d';
return(true);
}
else
if(type=='h')
{
if(!deftype)Text[len-1]=0;
*(float*)out=(float)strtod(Text,NULL);
if(!deftype)Text[len-1]='h';
return(true);
}
else
if(type=='r')
{
if(!deftype)Text[len-1]=0;
*(double*)out=(double)strtod(Text,NULL);
if(!deftype)Text[len-1]='r';
return(true);
}
else
if(type=='f')
{
if(!deftype)Text[len-1]=0;
XE_STRING temp;
unsigned int p=0;
unsigned int i=0;
p=Search(temp,":",p);
((float*)out)[i++]=(float)strtod(temp.Get(),NULL);
while(p&&i<4)
{
p=Search(temp,":",p);
((float*)out)[i++]=(float)strtod(temp.Get(),NULL);
}
if(!deftype)Text[len-1]='f';
return(true);
}
else
if(type=='v')
{
if(!deftype)Text[len-1]=0;
XE_STRING temp;
unsigned int p=0;
unsigned int i=0;
p=Search(temp,":",p);
((double*)out)[i++]=(double)strtod(temp.Get(),NULL);
while(p&&i<4)
{
p=Search(temp,":",p);
((double*)out)[i++]=(double)strtod(temp.Get(),NULL);
}
if(!deftype)Text[len-1]='v';
return(true);
}
return(false);
}

int XE_STRING::ConvertType()
{
unsigned int len=strlen(Text);
if(len<=1)return(0);
char type=Text[len-1];
if(type=='x'||type=='c'||type=='b'||type=='s'||type=='w'||type=='i'||type=='d'||type=='h'||type=='r'||type=='f'||type=='v')return(type);
return(0);
}

unsigned int XE_STRING::ConvertSize()
{
unsigned int len=strlen(Text);
if(len<=1)return(0);
char type=Text[len-1];
if(type=='x')return(len/2);
if(type=='c'||type=='b')return(1);
if(type=='s'||type=='w')return(2);
if(type=='i'||type=='d')return(4);
if(type=='h')return(4);
if(type=='r')return(8);
if(type=='f')return(16);
if(type=='v')return(32);
return(0);
}

void XE_STRING::EscapeSeq()
{
unsigned int len=strlen(Text);
if(len<=0)return;
#ifdef XE_ALLOC_C
char* temp=(char*)malloc(sizeof(char)*len);
#else
char* temp=new char[len+1];
#endif /* XE_ALLOC_C */
if(!temp)return;
char* spos=Text;
char* dpos=temp;
while(*spos&&(unsigned int)spos<=(unsigned int)Text+len)
{
if(*spos!='\\')
{
*dpos=*spos;
spos++;
dpos++;
}
else
{
spos++;
if(!*spos)continue;
if(*spos=='0')*dpos='\0';
if(*spos=='a')*dpos='\a';
if(*spos=='b')*dpos='\b';
if(*spos=='f')*dpos='\f';
if(*spos=='n')*dpos='\n';
if(*spos=='r')*dpos='\r';
if(*spos=='t')*dpos='\t';
if(*spos=='v')*dpos='\v';
if(*spos=='\'')*dpos='\'';
if(*spos=='"')*dpos='\"';
if(*spos=='\\')*dpos='\\';
if(*spos=='?')*dpos='\?';
if(*spos=='x')
{
char res=0;
char tmp=0;
spos++;
if(!*spos)continue;
if(*spos>='0'&&*spos<='9')tmp=*spos-'0';
else
if(*spos>='a'&&*spos<='f')tmp=*spos-'a'+10;
else
if(*spos>='A'&&*spos<='F')tmp=*spos-'A'+10;
res|=tmp<<4;
spos++;
if(!*spos)continue;
if(*spos>='0'&&*spos<='9')tmp=*spos-'0';
else
if(*spos>='a'&&*spos<='f')tmp=*spos-'a'+10;
else
if(*spos>='A'&&*spos<='F')tmp=*spos-'A'+10;
res|=tmp;
*dpos=res;
}
spos++;
dpos++;
}
}
*dpos=0;
Set(temp);
delete[] temp;
}

unsigned int XE_STRING::Length()
{
return(strlen(Text));
}

unsigned int XE_STRING::LengthUnicode(unsigned int extrachars)
{
return((strlen(Text)+extrachars)*sizeof(wchar_t));
}

bool XE_STRING::Find(unsigned int offset,char* substr,unsigned int* position)
{
if(offset>=strlen(Text))return(false);
void* pos=strstr(&Text[offset],substr);
if((unsigned int)pos<(unsigned int)Text)return(false);
if(position)*position=(unsigned int)pos-(unsigned int)Text;
return(true);
}

unsigned int XE_STRING::Search(XE_STRING& result,char* delim,unsigned int offset)
{
if(offset>strlen(Text))return(0);
char* begin=(char*)((unsigned int)Text+offset);
char* end=strstr(begin,delim);
if(!end)end=(char*)((unsigned int)Text+(unsigned int)strlen(Text));
if((unsigned int)end<(unsigned int)Text||(unsigned int)end<(unsigned int)begin)return(0);
result.Copy(begin,(unsigned int)end-(unsigned int)begin);
unsigned int pos=(unsigned int)(end-Text)+strlen(delim);
if(pos>strlen(Text))pos=0;
return(pos);
}

unsigned int XE_STRING::Replace(XE_STRING& result,char* oldstr,char* newstr)
{
result.Clear();
unsigned int count=0;
char* pos=Text;
char* last=Text;
char* end=&Text[strlen(Text)];
XE_STRING part;
while(pos<end)
{
pos=strstr(last,oldstr);
if(!pos)
{
pos=end;
part.Copy(last,pos-last);
result<<part;
}
else
{
count++;
part.Copy(last,pos-last);
result<<part<<newstr;
}
part.Clear();
last=pos+strlen(oldstr);
}
return(count);
}

unsigned int XE_STRING::ReplaceChars(XE_STRING& result,char oldchar,char newchar)
{
unsigned int count=0;
for(unsigned int i=0;i<Length();i++)
if(Text[i]==oldchar)
{
Text[i]=newchar;
count++;
}
return(count);
}

void XE_STRING::Clear()
{
#ifdef XE_ALLOC_C
if(Text)free(Text);
Text=(char*)malloc(sizeof(char)*1);
#else
if(Text)delete[] Text;
Text=new char[1];
#endif /* XE_ALLOC_C */
Text[0]=0;
}

void XE_STRING::operator= (const XE_STRING& str)
{
Set(str);
}

void XE_STRING::operator= (char* str)
{
Set(str);
}

XE_STRING::operator char*() const
{
return(Text);
}

XE_STRING& XE_STRING::operator<< (double value)
{
#ifdef XE_ALLOC_C
char* buff=(char*)malloc(sizeof(char)*33);
#else
char* buff=new char[33];
#endif /* XE_ALLOC_C */
_gcvt_s(buff,32,value,8);
unsigned int _len=strlen(Text)+strlen(buff);
#ifdef XE_ALLOC_C
char* newtext=(char*)malloc(sizeof(char)*(_len+1));
#else
char* newtext=new char[_len+1];
#endif /* XE_ALLOC_C */
strcpy_s(newtext,(_len+1),Text);
strcat_s(newtext,(_len+1),buff);
#ifdef XE_ALLOC_C
if(buff)free(buff);
if(Text)free(Text);
#else
if(buff)delete[] buff;
if(Text)delete[] Text;
#endif /* XE_ALLOC_C */
Text=newtext;
Text[_len]=0;
return(*this);
}

XE_STRING& XE_STRING::operator<< (int value)
{
#ifdef XE_ALLOC_C
char* buff=(char*)malloc(sizeof(char)*33);
#else
char* buff=new char[33];
#endif /* XE_ALLOC_C */
_itoa_s(value,buff,32,XE_BASE_DEC);
unsigned int _len=strlen(Text)+strlen(buff);
#ifdef XE_ALLOC_C
char* newtext=(char*)malloc(sizeof(char)*(_len+1));
#else
char* newtext=new char[_len+1];
#endif /* XE_ALLOC_C */
strcpy_s(newtext,_len+1,Text);
strcat_s(newtext,_len+1,buff);
#ifdef XE_ALLOC_C
if(buff)free(buff);
if(Text)free(Text);
#else
if(buff)delete[] buff;
if(Text)delete[] Text;
#endif /* XE_ALLOC_C */
Text=newtext;
Text[_len]=0;
return(*this);
}

XE_STRING& XE_STRING::operator<< (unsigned int value)
{
#ifdef XE_ALLOC_C
char* buff=(char*)malloc(sizeof(char)*33);
#else
char* buff=new char[33];
#endif /* XE_ALLOC_C */
_ultoa_s((unsigned long)value,buff,32,XE_BASE_DEC);
unsigned int _len=strlen(Text)+strlen(buff);
#ifdef XE_ALLOC_C
char* newtext=(char*)malloc(sizeof(char)*(_len+1));
#else
char* newtext=new char[_len+1];
#endif /* XE_ALLOC_C */
strcpy_s(newtext,_len+1,Text);
strcat_s(newtext,_len+1,buff);
#ifdef XE_ALLOC_C
if(buff)free(buff);
if(Text)free(Text);
#else
if(buff)delete[] buff;
if(Text)delete[] Text;
#endif /* XE_ALLOC_C */
Text=newtext;
Text[_len]=0;
return(*this);
}

XE_STRING& XE_STRING::operator<< (char* value)
{
unsigned int _len=strlen(Text)+strlen(value);
#ifdef XE_ALLOC_C
char* newtext=(char*)malloc(sizeof(char)*(_len+1));
#else
char* newtext=new char[_len+1];
#endif /* XE_ALLOC_C */
strcpy_s(newtext,_len+1,Text);
strcat_s(newtext,_len+1,value);
#ifdef XE_ALLOC_C
if(Text)free(Text);
#else
if(Text)delete[] Text;
#endif /* XE_ALLOC_C */
Text=newtext;
Text[_len]=0;
return(*this);
}

bool XE_STRING::operator== (XE_STRING& str)
{
if(strcmp(Text,str.Get())==0)
return(true);
return(false);
}

bool XE_STRING::operator!= (XE_STRING& str)
{
if(strcmp(Text,str.Get())!=0)
return(true);
return(false);
}

bool XE_STRING::operator== (char* str)
{
if(strcmp(Text,str)==0)
return(true);
return(false);
}

bool XE_STRING::operator!= (char* str)
{
if(strcmp(Text,str)!=0)
return(true);
return(false);
}

#endif /* XE_CANIMP */

/*--- Data types ---*/
//! 32 bit pointer
typedef void* XE_POINTER;
//! 8 bit signed
typedef char XE_CHAR;
//! 8 bit unsigned
typedef unsigned char XE_BYTE;
//! 16 bit signed
typedef short XE_SHORT;
//! 16 bit unsigned
typedef unsigned short XE_WORD;
//! 32 bit signed
typedef int XE_INTEGER;
//! 32 bit unsigned
typedef unsigned int XE_DWORD;
//! 32 bit float
typedef float XE_HALF;
//! 64 bit float
typedef double XE_REAL;
//! 32 bit handle
typedef XE_DWORD XE_HANDLE;

/*! Klasa typu QWORD (64bit). */
class XE_QWORD
{
public:
/*! Wartosc nizsza (32bit). */
unsigned int Low;
/*! Wartosc wyzsza (32bit). */
unsigned int High;
/*! Konstruktor kopiujacy.
\param h Wartosc wyzsza (32bit).
\param l Wartosc nizsza (32bit).
*/
XE_QWORD(XE_DWORD h=0,XE_DWORD l=0);
/*! Operator porownania.
Sprawdza czy oba QWORD sa rowne.
\param value Wartosc do porownania.
\return true jesli oba QWORD sa rowne.
*/
bool operator== (XE_QWORD& value);
/*! Operator porownania.
Sprawdza czy oba QWORD sa rozne.
\param value Wartosc do porownania.
\return true jesli oba QWORD sa rozne.
*/
bool operator!= (XE_QWORD& value);
/*! Operator porownania.
Sprawdza czy QWORD jest wiekszy badz rowny niz drugi.
\param value Wartosc do porownania.
\return true jesli pierwszy QWORD jest wiekszy badz rowny niz drugi.
*/
bool operator>= (XE_QWORD& value);
/*! Operator porownania.
Sprawdza czy QWORD jest mniejszy badz rowny niz drugi.
\param value Wartosc do porownania.
\return true jesli pierwszy QWORD jest mniejszy badz rowny niz drugi.
*/
bool operator<= (XE_QWORD& value);
/*! Operator porownania.
Sprawdza czy QWORD jest wiekszy niz drugi.
\param value Wartosc do porownania.
\return true jesli pierwszy QWORD jest wiekszy niz drugi.
*/
bool operator> (XE_QWORD& value);
/*! Operator porownania.
Sprawdza czy QWORD jest mniejszy niz drugi.
\param value Wartosc do porownania.
\return true jesli pierwszy QWORD jest mniejszy niz drugi.
*/
bool operator< (XE_QWORD& value);
/*! Operator bitowy.
Wykonuje sume bitowa na obu QWORD.
\param value Wartosc.
\return QWORD wynikowy.
*/
XE_QWORD operator| (XE_QWORD value);
/*! Operator bitowy.
Wykonuje iloczyn bitowy na obu QWORD.
\param value Wartosc.
\return QWORD wynikowy.
*/
XE_QWORD operator& (XE_QWORD value);
/*! Operator bitowy.
Wykonuje roznice bitowa na obu QWORD.
\param value Wartosc.
\return QWORD wynikowy.
*/
XE_QWORD operator^ (XE_QWORD value);
/*! Operator bitowy przypisania.
Wykonuje przypisanie sumy bitowej obu QWORD.
\param value Wartosc.
\return QWORD wynikowy.
*/
XE_QWORD operator|= (XE_QWORD& value);
/*! Operator bitowy przypisania.
Wykonuje przypisanie iloczynu bitowego obu QWORD.
\param value Wartosc.
\return QWORD wynikowy.
*/
XE_QWORD operator&= (XE_QWORD& value);
/*! Operator bitowy przypisania.
Wykonuje przypisanie roznicy bitowej obu QWORD.
\param value Wartosc.
\return QWORD wynikowy.
*/
XE_QWORD operator^= (XE_QWORD& value);
};

#ifdef XE_CANIMP

XE_QWORD::XE_QWORD(XE_DWORD h,XE_DWORD l)
{
Low=l;
High=h;
}

bool XE_QWORD::operator== (XE_QWORD& value)
{
if(High==value.High&&Low==value.Low)
return(true);
return(false);
}

bool XE_QWORD::operator!= (XE_QWORD& value)
{
if(High!=value.High||Low!=value.Low)
return(true);
return(false);
}

bool XE_QWORD::operator>= (XE_QWORD& value)
{
if(High>value.High)return(true);
if(High<value.High)return(false);
if(High==value.High)
{
if(Low>value.Low)return(true);
if(Low==value.Low)return(true);
return(false);
}
return(false);
}

bool XE_QWORD::operator<= (XE_QWORD& value)
{
if(High<value.High)return(true);
if(High>value.High)return(false);
if(High==value.High)
{
if(Low<value.Low)return(true);
if(Low==value.Low)return(true);
return(false);
}
return(false);
}

bool XE_QWORD::operator> (XE_QWORD& value)
{
if(High>value.High)return(true);
if(High<value.High)return(false);
if(High==value.High)
{
if(Low>value.Low)return(true);
return(false);
}
return(false);
}

bool XE_QWORD::operator< (XE_QWORD& value)
{
if(High<value.High)return(true);
if(High>value.High)return(false);
if(High==value.High)
{
if(Low<value.Low)return(true);
return(false);
}
return(false);
}

XE_QWORD XE_QWORD::operator| (XE_QWORD value)
{
XE_QWORD _res;
_res.High=High|value.High;
_res.Low=Low|value.Low;
return(_res);
}

XE_QWORD XE_QWORD::operator& (XE_QWORD value)
{
XE_QWORD _res;
_res.High=High&value.High;
_res.Low=Low&value.Low;
return(_res);
}

XE_QWORD XE_QWORD::operator^ (XE_QWORD value)
{
XE_QWORD _res;
_res.High=High^value.High;
_res.Low=Low^value.Low;
return(_res);
}

XE_QWORD XE_QWORD::operator|= (XE_QWORD& value)
{
High|=value.High;
Low|=value.Low;
return(*this);
}

XE_QWORD XE_QWORD::operator&= (XE_QWORD& value)
{
High&=value.High;
Low&=value.Low;
return(*this);
}

XE_QWORD XE_QWORD::operator^= (XE_QWORD& value)
{
High^=value.High;
Low^=value.Low;
return(*this);
}

#endif /* XE_CANIMP */

/* Vectors & Matrices */
template<typename T> class XE_VECTOR_TEMPLATE;
typedef XE_VECTOR_TEMPLATE<double> XE_VECTOR;
typedef XE_VECTOR_TEMPLATE<float> XE_HALFVECTOR;
template<typename T> class XE_ANGLEVECTOR_TEMPLATE;
typedef XE_ANGLEVECTOR_TEMPLATE<double> XE_ANGLEVECTOR;
typedef XE_ANGLEVECTOR_TEMPLATE<float> XE_ANGLEHALFVECTOR;
template<typename T> class XE_MATRIX_TEMPLATE;
typedef XE_MATRIX_TEMPLATE<double> XE_MATRIX;
typedef XE_MATRIX_TEMPLATE<float> XE_HALFMATRIX;

#define XE_HALFTOVEC(val) XE_VECTOR((double)(val).X,(double)(val).Y,(double)(val).Z,(double)(val).W)
#define XE_VECTOHALF(val) XE_HALFVECTOR((float)(val).X,(float)(val).Y,(float)(val).Z,(float)(val).W)
#define XE_FORLINE(begin,end,as,size,offset,whennomove)	for(XE_VECTOR __d_i_r__=((end)-(begin)).Normalize(),__d_i_s__=XE_VECTOR(offset),__l_e_n__=XE_VECTOR(((end)-(begin)).Length()),as=(begin)+__d_i_r__*XE_NUMBER(offset);(whennomove)?(__d_i_s__.X<=__l_e_n__.X):(__d_i_s__.X<__l_e_n__.X);__d_i_s__.X+=size,as+=__d_i_r__*XE_NUMBER(size))
#define XE_FORLINEHALF(begin,end,as,size,offset,whennomove)	for(XE_HALFVECTOR __d_i_r__=((end)-(begin)).Normalize(),__d_i_s__=XE_HALFVECTOR(offset),__l_e_n__=XE_HALFVECTOR(((end)-(begin)).Length()),as=(begin)+__d_i_r__*XE_NUMBER(offset);(whennomove)?(__d_i_s__.X<=__l_e_n__.X):(__d_i_s__.X<__l_e_n__.X);__d_i_s__.X+=size,as+=__d_i_r__*XE_NUMBER(size))

/*--- XE_VECTOR_TEMPLATE ---*/
//! Klasa szablonu wektora
template<typename T>
class XE_VECTOR_TEMPLATE
{
public:
//! Wspolrzedna X
T X;
//! Wspolrzedna Y
T Y;
//! Wspolrzedna Z
T Z;
//! Wspolczedna W
T W;
//! Konstruktor przypisujacy poczatkowe wartosci
XE_VECTOR_TEMPLATE(T _x=0,T _y=0,T _z=0,T _w=1);
//! Przypisanie wartosci
void Set(T _x=0,T _y=0,T _z=0,T _w=1);
//! Zwraca dlugosc wektora
T Length();
//! Zwraca znormalizowana wartosc wektora
XE_VECTOR_TEMPLATE<T> Normalize();
//! Zwraca minimum
XE_VECTOR_TEMPLATE<T> Min(XE_VECTOR_TEMPLATE<T>& v);
//! Zwraca maksimum
XE_VECTOR_TEMPLATE<T> Max(XE_VECTOR_TEMPLATE<T>& v);
//! Zwraca wektor ze zresetowanymi wybranymi komponentami
XE_VECTOR_TEMPLATE<T> Without(unsigned int flags);
//! Zwraca wektor pomiedzy dwoma danymi, uwzgledniajac ich wagi.
XE_VECTOR_TEMPLATE<T> Amount(XE_VECTOR_TEMPLATE<T>& v,T amount);
//! Rzutuje wektory.
XE_VECTOR_TEMPLATE<T> Project(XE_VECTOR_TEMPLATE<T>& v);
//! Operator dodawania
XE_VECTOR_TEMPLATE<T> operator+ (XE_VECTOR_TEMPLATE<T>& v);
//! Operator dodawania
XE_VECTOR_TEMPLATE<T> operator+ (XE_NUMBER& n);
//! Operator odejmowania
XE_VECTOR_TEMPLATE<T> operator- (XE_VECTOR_TEMPLATE<T>& v);
//! Operator odejmowania
XE_VECTOR_TEMPLATE<T> operator- (XE_NUMBER& n);
//! Operator mnozenia
XE_VECTOR_TEMPLATE<T> operator* (XE_VECTOR_TEMPLATE<T>& v);
//! Operator mnozenia
XE_VECTOR_TEMPLATE<T> operator* (XE_NUMBER& n);
//! Operator dzielenia
XE_VECTOR_TEMPLATE<T> operator/ (XE_VECTOR_TEMPLATE<T>& v);
//! Operator dzielenia
XE_VECTOR_TEMPLATE<T> operator/ (XE_NUMBER& n);
//! Operator iloczynu wektorowego
XE_VECTOR_TEMPLATE<T> operator^ (XE_VECTOR_TEMPLATE<T>& v);
//! Operator iloczynu skalarnego
T operator% (XE_VECTOR_TEMPLATE<T>& v);
//! Operator dodawania
XE_VECTOR_TEMPLATE<T>& operator+= (XE_VECTOR_TEMPLATE<T>& v);
//! Operator dodawania
XE_VECTOR_TEMPLATE<T>& operator+= (XE_NUMBER& n);
//! Operator odejmowania
XE_VECTOR_TEMPLATE<T>& operator-= (XE_VECTOR_TEMPLATE<T>& v);
//! Operator odejmowania
XE_VECTOR_TEMPLATE<T>& operator-= (XE_NUMBER& n);
//! Operator mnozenia
XE_VECTOR_TEMPLATE<T>& operator*= (XE_VECTOR_TEMPLATE<T>& v);
//! Operator mnozenia
XE_VECTOR_TEMPLATE<T>& operator*= (XE_NUMBER& n);
//! Operator dzielenia
XE_VECTOR_TEMPLATE<T>& operator/= (XE_VECTOR_TEMPLATE<T>& v);
//! Operator dzielenia
XE_VECTOR_TEMPLATE<T>& operator/= (XE_NUMBER& n);
//! Operator iloczynu wektorowego
XE_VECTOR_TEMPLATE<T>& operator^= (XE_VECTOR_TEMPLATE<T>& v);
//! Operator zmiany znaku
XE_VECTOR_TEMPLATE<T>& operator- ();
//! Operator rownosci
bool operator== (XE_VECTOR_TEMPLATE<T>& v);
//! Operator roznosci
bool operator!= (XE_VECTOR_TEMPLATE<T>& v);
};

template<typename T>
XE_VECTOR_TEMPLATE<T>::XE_VECTOR_TEMPLATE(T _x,T _y,T _z,T _w)
{
X=_x;
Y=_y;
Z=_z;
W=_w;
}

template<typename T>
void XE_VECTOR_TEMPLATE<T>::Set(T _x,T _y,T _z,T _w)
{
X=_x;
Y=_y;
Z=_z;
W=_w;
}

template<typename T>
T XE_VECTOR_TEMPLATE<T>::Length()
{
return(sqrt((X*X)+(Y*Y)+(Z*Z)));
}

template<typename T>
XE_VECTOR_TEMPLATE<T> XE_VECTOR_TEMPLATE<T>::Normalize()
{
T l=Length();
if(l==0)return(XE_VECTOR_TEMPLATE());
XE_VECTOR_TEMPLATE<T> vr;
vr.X=X/l;
vr.Y=Y/l;
vr.Z=Z/l;
return(vr);
}

template<typename T>
XE_VECTOR_TEMPLATE<T> XE_VECTOR_TEMPLATE<T>::Min(XE_VECTOR_TEMPLATE<T>& v)
{
XE_VECTOR_TEMPLATE<T> vr;
vr.X=min(X,v.X);
vr.Y=min(Y,v.Y);
vr.Z=min(Z,v.Z);
vr.W=min(W,v.W);
return(vr);
}

template<typename T>
XE_VECTOR_TEMPLATE<T> XE_VECTOR_TEMPLATE<T>::Max(XE_VECTOR_TEMPLATE<T>& v)
{
XE_VECTOR_TEMPLATE<T> vr;
vr.X=max(X,v.X);
vr.Y=max(Y,v.Y);
vr.Z=max(Z,v.Z);
vr.W=max(W,v.W);
return(vr);
}

template<typename T>
XE_VECTOR_TEMPLATE<T> XE_VECTOR_TEMPLATE<T>::Without(unsigned int flags)
{
XE_VECTOR_TEMPLATE<T> vr=*this;
if(flags&XE_FLAG_X)vr.X=0;
if(flags&XE_FLAG_Y)vr.Y=0;
if(flags&XE_FLAG_Z)vr.Z=0;
if(flags&XE_FLAG_W)vr.W=1;
return(vr);
}

template<typename T>
XE_VECTOR_TEMPLATE<T> XE_VECTOR_TEMPLATE<T>::Amount(XE_VECTOR_TEMPLATE<T>& v,T amount)
{
XE_VECTOR_TEMPLATE<T> vr;
vr.X=(X-v.X)*min(1,max(0,amount))+v.X;
vr.Y=(Y-v.Y)*min(1,max(0,amount))+v.Y;
vr.Z=(Z-v.Z)*min(1,max(0,amount))+v.Z;
vr.W=(W-v.W)*min(1,max(0,amount))+v.W;
return(vr);
}

template<typename T>
XE_VECTOR_TEMPLATE<T> XE_VECTOR_TEMPLATE<T>::Project(XE_VECTOR_TEMPLATE<T>& v)
{
XE_VECTOR_TEMPLATE<T> vr;
T dp=*this%v;
T mp=v.X*v.X+v.Y*v.Y+v.Z*v.Z;
vr.X=(dp/mp)*v.X;
vr.Y=(dp/mp)*v.Y;
vr.Z=(dp/mp)*v.Z;
return(vr);
}

template<typename T>
XE_VECTOR_TEMPLATE<T> XE_VECTOR_TEMPLATE<T>::operator+ (XE_VECTOR_TEMPLATE<T>& v)
{
XE_VECTOR_TEMPLATE<T> vr;
vr.X=X+v.X;
vr.Y=Y+v.Y;
vr.Z=Z+v.Z;
return(vr);
}

template<typename T>
XE_VECTOR_TEMPLATE<T> XE_VECTOR_TEMPLATE<T>::operator+ (XE_NUMBER& n)
{
XE_VECTOR_TEMPLATE<T> vr;
vr.X=X+(T)n.Get();
vr.Y=Y+(T)n.Get();
vr.Z=Z+(T)n.Get();
return(vr);
}

template<typename T>
XE_VECTOR_TEMPLATE<T> XE_VECTOR_TEMPLATE<T>::operator- (XE_VECTOR_TEMPLATE<T>& v)
{
XE_VECTOR_TEMPLATE<T> vr;
vr.X=X-v.X;
vr.Y=Y-v.Y;
vr.Z=Z-v.Z;
return(vr);
}

template<typename T>
XE_VECTOR_TEMPLATE<T> XE_VECTOR_TEMPLATE<T>::operator- (XE_NUMBER& n)
{
XE_VECTOR_TEMPLATE<T> vr;
vr.X=X-(T)n.Get();
vr.Y=Y-(T)n.Get();
vr.Z=Z-(T)n.Get();
return(vr);
}

template<typename T>
XE_VECTOR_TEMPLATE<T> XE_VECTOR_TEMPLATE<T>::operator* (XE_VECTOR_TEMPLATE<T>& v)
{
XE_VECTOR_TEMPLATE<T> vr;
vr.X=X*v.X;
vr.Y=Y*v.Y;
vr.Z=Z*v.Z;
return(vr);
}

template<typename T>
XE_VECTOR_TEMPLATE<T> XE_VECTOR_TEMPLATE<T>::operator* (XE_NUMBER& n)
{
XE_VECTOR_TEMPLATE<T> vr;
vr.X=X*(T)n.Get();
vr.Y=Y*(T)n.Get();
vr.Z=Z*(T)n.Get();
return(vr);
}

template<typename T>
XE_VECTOR_TEMPLATE<T> XE_VECTOR_TEMPLATE<T>::operator/ (XE_VECTOR_TEMPLATE<T>& v)
{
XE_VECTOR_TEMPLATE<T> vr;
vr.X=X/v.X;
vr.Y=Y/v.Y;
vr.Z=Z/v.Z;
return(vr);
}

template<typename T>
XE_VECTOR_TEMPLATE<T> XE_VECTOR_TEMPLATE<T>::operator/ (XE_NUMBER& n)
{
XE_VECTOR_TEMPLATE<T> vr;
vr.X=X/(T)n.Get();
vr.Y=Y/(T)n.Get();
vr.Z=Z/(T)n.Get();
return(vr);
}

template<typename T>
XE_VECTOR_TEMPLATE<T> XE_VECTOR_TEMPLATE<T>::operator^ (XE_VECTOR_TEMPLATE<T>& v)
{
return(XE_VECTOR_TEMPLATE((Y*v.Z)-(v.Y*Z),(v.X*Z)-(X*v.Z),(X*v.Y)-(v.X*Y)));
}

template<typename T>
T XE_VECTOR_TEMPLATE<T>::operator% (XE_VECTOR_TEMPLATE<T>& v)
{
return(X*v.X+Y*v.Y+Z*v.Z);
}

template<typename T>
XE_VECTOR_TEMPLATE<T>& XE_VECTOR_TEMPLATE<T>::operator+= (XE_VECTOR_TEMPLATE<T>& v)
{
*this=*this+v;
return(*this);
}

template<typename T>
XE_VECTOR_TEMPLATE<T>& XE_VECTOR_TEMPLATE<T>::operator+= (XE_NUMBER& n)
{
*this=*this+n;
return(*this);
}

template<typename T>
XE_VECTOR_TEMPLATE<T>& XE_VECTOR_TEMPLATE<T>::operator-= (XE_VECTOR_TEMPLATE<T>& v)
{
*this=*this-v;
return(*this);
}

template<typename T>
XE_VECTOR_TEMPLATE<T>& XE_VECTOR_TEMPLATE<T>::operator-= (XE_NUMBER& n)
{
*this=*this-n;
return(*this);
}

template<typename T>
XE_VECTOR_TEMPLATE<T>& XE_VECTOR_TEMPLATE<T>::operator*= (XE_VECTOR_TEMPLATE<T>& v)
{
*this=*this*v;
return(*this);
}

template<typename T>
XE_VECTOR_TEMPLATE<T>& XE_VECTOR_TEMPLATE<T>::operator*= (XE_NUMBER& n)
{
*this=*this*n;
return(*this);
}

template<typename T>
XE_VECTOR_TEMPLATE<T>& XE_VECTOR_TEMPLATE<T>::operator/= (XE_VECTOR_TEMPLATE<T>& v)
{
*this=*this/v;
return(*this);
}

template<typename T>
XE_VECTOR_TEMPLATE<T>& XE_VECTOR_TEMPLATE<T>::operator/= (XE_NUMBER& n)
{
*this=*this/n;
return(*this);
}

template<typename T>
XE_VECTOR_TEMPLATE<T>& XE_VECTOR_TEMPLATE<T>::operator^= (XE_VECTOR_TEMPLATE<T>& v)
{
*this=*this^v;
return(*this);
}

template<typename T>
XE_VECTOR_TEMPLATE<T>& XE_VECTOR_TEMPLATE<T>::operator- ()
{
X=-X;
Y=-Y;
Z=-Z;
return(*this);
}

template<typename T>
bool XE_VECTOR_TEMPLATE<T>::operator== (XE_VECTOR_TEMPLATE<T>& v)
{
if(X==v.X&&Y==v.Y&&Z==v.Z&&W==v.W)
return(true);
return(false);
}

template<typename T>
bool XE_VECTOR_TEMPLATE<T>::operator!= (XE_VECTOR_TEMPLATE<T>& v)
{
if(X!=v.X||Y!=v.Y||Z!=v.Z||W!=v.W)
return(true);
return(false);
}

/*--- XE_PERVERTEX ---*/
//! Struktura danych wierzcholka
struct XE_PERVERTEX
{
//! Koordynaty tekstury
XE_HALFVECTOR Coord;
//! Kolor
XE_HALFVECTOR Color;
//! Normalna (X)
float NormalX;
//! Normalna (Y)
float NormalY;
//! Normalna (Z)
float NormalZ;
//! Pozycja wierzcholka
XE_HALFVECTOR Vertex;
};

/*--- XE_ANGLEVECTOR_TEMPLATE ---*/
//! Klasa szablonu wektora sferycznego
template<typename T>
class XE_ANGLEVECTOR_TEMPLATE
{
public:
//! Kat Alpha (os Z)
T Alpha;
//! Kat Beta (os Y)
T Beta;
//! Kat Gamma (os X)
T Gamma;
//! Dlugosc
T Length;
//! Konstruktor przypisujacy poczatkowe wartosci
XE_ANGLEVECTOR_TEMPLATE(T _alpha=0,T _beta=0,T _gamma=0,T _length=0);
//! Przypisanie wartosci
void Set(T _alpha=0,T _beta=0,T _gamma=0,T _length=0);
//! Moduluje wartosci wzgledem innego wektora sferycznego.
void Mod(XE_ANGLEVECTOR_TEMPLATE angle);
//! Operator dodawania
XE_ANGLEVECTOR_TEMPLATE<T> operator+ (XE_ANGLEVECTOR_TEMPLATE<T>& v);
//! Operator odejmowania
XE_ANGLEVECTOR_TEMPLATE<T> operator- (XE_ANGLEVECTOR_TEMPLATE<T>& v);
//! Operator mnozenia
XE_ANGLEVECTOR_TEMPLATE<T> operator* (XE_ANGLEVECTOR_TEMPLATE<T>& v);
//! Operator dzielenia
XE_ANGLEVECTOR_TEMPLATE<T> operator/ (XE_ANGLEVECTOR_TEMPLATE<T>& v);
//! Operator dodawania
XE_ANGLEVECTOR_TEMPLATE<T>& operator+= (XE_ANGLEVECTOR_TEMPLATE<T>& v);
//! Operator odejmowania
XE_ANGLEVECTOR_TEMPLATE<T>& operator-= (XE_ANGLEVECTOR_TEMPLATE<T>& v);
//! Operator mnozenia
XE_ANGLEVECTOR_TEMPLATE<T>& operator*= (XE_ANGLEVECTOR_TEMPLATE<T>& v);
//! Operator dzielenia
XE_ANGLEVECTOR_TEMPLATE<T>& operator/= (XE_ANGLEVECTOR_TEMPLATE<T>& v);
//! Operator zmiany znaku
XE_ANGLEVECTOR_TEMPLATE<T>& operator- ();
//! Operator rownosci
bool operator== (XE_ANGLEVECTOR_TEMPLATE<T>& v);
//! Operator roznosci
bool operator!= (XE_ANGLEVECTOR_TEMPLATE<T>& v);
};

template<typename T>
XE_ANGLEVECTOR_TEMPLATE<T>::XE_ANGLEVECTOR_TEMPLATE(T _alpha,T _beta,T _gamma,T _length)
{
Alpha=_alpha;
Beta=_beta;
Gamma=_gamma;
Length=_length;
}

template<typename T>
void XE_ANGLEVECTOR_TEMPLATE<T>::Set(T _alpha,T _beta,T _gamma,T _length)
{
Alpha=_alpha;
Beta=_beta;
Gamma=_gamma;
Length=_length;
}

template<typename T>
void XE_ANGLEVECTOR_TEMPLATE<T>::Mod(XE_ANGLEVECTOR_TEMPLATE<T> angle)
{
Alpha=fmod(Alpha,angle.Alpha);
Beta=fmod(Beta,angle.Beta);
Gamma=fmod(Gamma,angle.Gamma);
Length=fmod(Length,angle.Length);
}

template<typename T>
XE_ANGLEVECTOR_TEMPLATE<T> XE_ANGLEVECTOR_TEMPLATE<T>::operator+ (XE_ANGLEVECTOR_TEMPLATE<T>& v)
{
XE_ANGLEVECTOR vr;
vr.Alpha=Alpha+v.Alpha;
vr.Beta=Beta+v.Beta;
vr.Gamma=Gamma+v.Gamma;
vr.Length=Length+v.Length;
return(vr);
}

template<typename T>
XE_ANGLEVECTOR_TEMPLATE<T> XE_ANGLEVECTOR_TEMPLATE<T>::operator- (XE_ANGLEVECTOR_TEMPLATE<T>& v)
{
XE_ANGLEVECTOR vr;
vr.Alpha=Alpha-v.Alpha;
vr.Beta=Beta-v.Beta;
vr.Gamma=Gamma-v.Gamma;
vr.Length=Length-v.Length;
return(vr);
}

template<typename T>
XE_ANGLEVECTOR_TEMPLATE<T> XE_ANGLEVECTOR_TEMPLATE<T>::operator* (XE_ANGLEVECTOR_TEMPLATE<T>& v)
{
XE_ANGLEVECTOR vr;
vr.Alpha=Alpha*v.Alpha;
vr.Beta=Beta*v.Beta;
vr.Gamma=Gamma*v.Gamma;
vr.Length=Length*v.Length;
return(vr);
}

template<typename T>
XE_ANGLEVECTOR_TEMPLATE<T> XE_ANGLEVECTOR_TEMPLATE<T>::operator/ (XE_ANGLEVECTOR_TEMPLATE<T>& v)
{
XE_ANGLEVECTOR vr;
vr.Alpha=Alpha/v.Alpha;
vr.Beta=Beta/v.Beta;
vr.Gamma=Gamma/v.Gamma;
if(v.Length!=0)vr.Length=Length/v.Length;
return(vr);
}

template<typename T>
XE_ANGLEVECTOR_TEMPLATE<T>& XE_ANGLEVECTOR_TEMPLATE<T>::operator+= (XE_ANGLEVECTOR_TEMPLATE<T>& v)
{
*this=*this+v;
return(*this);
}

template<typename T>
XE_ANGLEVECTOR_TEMPLATE<T>& XE_ANGLEVECTOR_TEMPLATE<T>::operator-= (XE_ANGLEVECTOR_TEMPLATE<T>& v)
{
*this=*this-v;
return(*this);
}

template<typename T>
XE_ANGLEVECTOR_TEMPLATE<T>& XE_ANGLEVECTOR_TEMPLATE<T>::operator*= (XE_ANGLEVECTOR_TEMPLATE<T>& v)
{
*this=*this*v;
return(*this);
}

template<typename T>
XE_ANGLEVECTOR_TEMPLATE<T>& XE_ANGLEVECTOR_TEMPLATE<T>::operator/= (XE_ANGLEVECTOR_TEMPLATE<T>& v)
{
*this=*this/v;
return(*this);
}

template<typename T>
XE_ANGLEVECTOR_TEMPLATE<T>& XE_ANGLEVECTOR_TEMPLATE<T>::operator- ()
{
Alpha=-Alpha;
Beta=-Beta;
Gamma=-Gamma;
return(*this);
}

template<typename T>
bool XE_ANGLEVECTOR_TEMPLATE<T>::operator== (XE_ANGLEVECTOR_TEMPLATE<T>& v)
{
if(Alpha==v.Alpha&&Beta==v.Beta&&Gamma==v.Gamma&&Length==v.Length)
return(true);
return(false);
}

template<typename T>
bool XE_ANGLEVECTOR_TEMPLATE<T>::operator!= (XE_ANGLEVECTOR_TEMPLATE<T>& v)
{
if(Alpha!=v.Alpha||Beta!=v.Beta||Gamma!=v.Gamma||Length!=v.Length)
return(true);
return(false);
}

/*--- XE_MATRIX ---*/
//! Klasa szablonu macierzy 4x4.
template<typename T>
class XE_MATRIX_TEMPLATE
{
public:
//! Komorki macierzy.
T Cell[16];
//! Konstruktor domyslny.
XE_MATRIX_TEMPLATE();
/*! Ustala maciez jednostkowa. */
void Identity();
/*! Dodanie macierzy.
\param mat Macierz.
\return Macierz wynikowa.
*/
XE_MATRIX_TEMPLATE<T> Add(XE_MATRIX_TEMPLATE<T>& mat);
/*! Odejmmowanie macierzy.
\param mat Macierz.
\return Macierz wynikowa.
*/
XE_MATRIX_TEMPLATE<T> Substract(XE_MATRIX_TEMPLATE<T>& mat);
/*! Mnozenie macierzy przez skalar.
\param value Wartosc.
\return Macierz wynikowa.
*/
XE_MATRIX_TEMPLATE<T> ScalarMultiply(T value);
/*! Transpozycja macierzy.
\return Macierz wynikowa.
*/
XE_MATRIX_TEMPLATE<T> Transpose();
/*! Mnozenie macierzy przez macierz.
\param mat Macierz.
\return Macierz wynikowa.
*/
XE_MATRIX_TEMPLATE<T> Multiply(XE_MATRIX_TEMPLATE<T>& mat);
/*! Mnozenie macierzy przez vector.
\param vec Vector.
\return Vector wynikowy.
*/
XE_VECTOR_TEMPLATE<T> Multiply(XE_VECTOR_TEMPLATE<T>& vec);
/*! Oblicza wyznacznik macierzy.
\return Wyznacznik.
*/
T Determinant();
/*! Oblicza odwrotnosc macierzy.
\return Macierz odwrotna.
*/
XE_MATRIX_TEMPLATE<T> Inverse();
/*! Przesuniecie.
\param x Wartosc X.
\param y Wartosc Y.
\param z Wartosc Z.
\return Macierz wynikowa.
*/
XE_MATRIX_TEMPLATE<T> Translate(T x,T y,T z);
/*! Skalowanie.
\param x Wartosc X.
\param y Wartosc Y.
\param z Wartosc Z.
\return Macierz wynikowa.
*/
XE_MATRIX_TEMPLATE<T> Scale(T x,T y,T z);
/*! Obrot w osi X.
\param angle Wartosc obrotu.
\return Macierz wynikowa.
*/
XE_MATRIX_TEMPLATE<T> RotateX(T angle);
/*! Obrot w osi Y.
\param angle Wartosc obrotu.
\return Macierz wynikowa.
*/
XE_MATRIX_TEMPLATE<T> RotateY(T angle);
/*! Obrot w osi Z.
\param angle Wartosc obrotu.
\return Macierz wynikowa.
*/
XE_MATRIX_TEMPLATE<T> RotateZ(T angle);
/*! Operator dodawania. */
XE_MATRIX_TEMPLATE<T> operator+ (XE_MATRIX_TEMPLATE<T>& mat);
/*! Operator odejmowania. */
XE_MATRIX_TEMPLATE<T> operator- (XE_MATRIX_TEMPLATE<T>& mat);
/*! Operator mnozenia przez skalar. */
XE_MATRIX_TEMPLATE<T> operator* (T value);
/*! Operator mnozenia. */
XE_MATRIX_TEMPLATE<T> operator* (XE_MATRIX_TEMPLATE<T>& mat);
/*! Operator mnozenia przez Vektor. */
XE_VECTOR_TEMPLATE<T> operator* (XE_VECTOR_TEMPLATE<T>& vec);
};

template<typename T>
XE_MATRIX_TEMPLATE<T>::XE_MATRIX_TEMPLATE()
{
Identity();
}

template<typename T>
void XE_MATRIX_TEMPLATE<T>::Identity()
{
Cell[0]=1;	Cell[4]=0;	Cell[8]=0;	Cell[12]=0;
Cell[1]=0;	Cell[5]=1;	Cell[9]=0;	Cell[13]=0;
Cell[2]=0;	Cell[6]=0;	Cell[10]=1;	Cell[14]=0;
Cell[3]=0;	Cell[7]=0;	Cell[11]=0;	Cell[15]=1;
}

template<typename T>
XE_MATRIX_TEMPLATE<T> XE_MATRIX_TEMPLATE<T>::Add(XE_MATRIX_TEMPLATE<T>& mat)
{
XE_MATRIX_TEMPLATE<T> temp;
for(int i=0;i<16;i++)
temp.Cell[i]=Cell[i]+mat.Cell[i];
return(temp);
}

template<typename T>
XE_MATRIX_TEMPLATE<T> XE_MATRIX_TEMPLATE<T>::Substract(XE_MATRIX_TEMPLATE<T>& mat)
{
XE_MATRIX_TEMPLATE<T> temp;
for(int i=0;i<16;i++)
temp.Cell[i]=Cell[i]-mat.Cell[i];
return(temp);
}

template<typename T>
XE_MATRIX_TEMPLATE<T> XE_MATRIX_TEMPLATE<T>::ScalarMultiply(T value)
{
XE_MATRIX_TEMPLATE<T> temp;
for(int i=0;i<16;i++)
temp.Cell[i]=Cell[i]*value;
return(temp);
}

template<typename T>
XE_MATRIX_TEMPLATE<T> XE_MATRIX_TEMPLATE<T>::Transpose()
{
XE_MATRIX_TEMPLATE<T> temp;
temp.Cell[0]=Cell[0];	temp.Cell[4]=Cell[1];	temp.Cell[8]=Cell[2];	temp.Cell[12]=Cell[3];
temp.Cell[1]=Cell[4];	temp.Cell[5]=Cell[5];	temp.Cell[9]=Cell[6];	temp.Cell[13]=Cell[7];
temp.Cell[2]=Cell[8];	temp.Cell[6]=Cell[9];	temp.Cell[10]=Cell[10];	temp.Cell[14]=Cell[11];
temp.Cell[3]=Cell[12];	temp.Cell[7]=Cell[13];	temp.Cell[11]=Cell[14];	temp.Cell[15]=Cell[15];
return(temp);
}

template<typename T>
XE_MATRIX_TEMPLATE<T> XE_MATRIX_TEMPLATE<T>::Multiply(XE_MATRIX_TEMPLATE<T>& mat)
{
XE_MATRIX_TEMPLATE<T> temp;
temp.Cell[0]=Cell[0]*mat.Cell[0]+Cell[4]*mat.Cell[1]+Cell[8]*mat.Cell[2]+Cell[12]*mat.Cell[3];
temp.Cell[1]=Cell[1]*mat.Cell[0]+Cell[5]*mat.Cell[1]+Cell[9]*mat.Cell[2]+Cell[13]*mat.Cell[3];
temp.Cell[2]=Cell[2]*mat.Cell[0]+Cell[6]*mat.Cell[1]+Cell[10]*mat.Cell[2]+Cell[14]*mat.Cell[3];
temp.Cell[3]=Cell[3]*mat.Cell[0]+Cell[7]*mat.Cell[1]+Cell[11]*mat.Cell[2]+Cell[15]*mat.Cell[3];
temp.Cell[4]=Cell[0]*mat.Cell[4]+Cell[4]*mat.Cell[5]+Cell[8]*mat.Cell[6]+Cell[12]*mat.Cell[7];
temp.Cell[5]=Cell[1]*mat.Cell[4]+Cell[5]*mat.Cell[5]+Cell[9]*mat.Cell[6]+Cell[13]*mat.Cell[7];
temp.Cell[6]=Cell[2]*mat.Cell[4]+Cell[6]*mat.Cell[5]+Cell[10]*mat.Cell[6]+Cell[14]*mat.Cell[7];
temp.Cell[7]=Cell[3]*mat.Cell[4]+Cell[7]*mat.Cell[5]+Cell[11]*mat.Cell[6]+Cell[15]*mat.Cell[7];
temp.Cell[8]=Cell[0]*mat.Cell[8]+Cell[4]*mat.Cell[9]+Cell[8]*mat.Cell[10]+Cell[12]*mat.Cell[11];
temp.Cell[9]=Cell[1]*mat.Cell[8]+Cell[5]*mat.Cell[9]+Cell[9]*mat.Cell[10]+Cell[13]*mat.Cell[11];
temp.Cell[10]=Cell[2]*mat.Cell[8]+Cell[6]*mat.Cell[9]+Cell[10]*mat.Cell[10]+Cell[14]*mat.Cell[11];
temp.Cell[11]=Cell[3]*mat.Cell[8]+Cell[7]*mat.Cell[9]+Cell[11]*mat.Cell[10]+Cell[15]*mat.Cell[11];
temp.Cell[12]=Cell[0]*mat.Cell[12]+Cell[4]*mat.Cell[13]+Cell[8]*mat.Cell[14]+Cell[12]*mat.Cell[15];
temp.Cell[13]=Cell[1]*mat.Cell[12]+Cell[5]*mat.Cell[13]+Cell[9]*mat.Cell[14]+Cell[13]*mat.Cell[15];
temp.Cell[14]=Cell[2]*mat.Cell[12]+Cell[6]*mat.Cell[13]+Cell[10]*mat.Cell[14]+Cell[14]*mat.Cell[15];
temp.Cell[15]=Cell[3]*mat.Cell[12]+Cell[7]*mat.Cell[13]+Cell[11]*mat.Cell[14]+Cell[15]*mat.Cell[15];
return(temp);
}

template<typename T>
XE_VECTOR_TEMPLATE<T> XE_MATRIX_TEMPLATE<T>::Multiply(XE_VECTOR_TEMPLATE<T>& vec)
{
XE_VECTOR_TEMPLATE<T> temp;
temp.X=Cell[0]*vec.X+Cell[4]*vec.Y+Cell[8]*vec.Z+Cell[12]*vec.W;
temp.Y=Cell[1]*vec.X+Cell[5]*vec.Y+Cell[9]*vec.Z+Cell[13]*vec.W;
temp.Z=Cell[2]*vec.X+Cell[6]*vec.Y+Cell[10]*vec.Z+Cell[14]*vec.W;
temp.W=Cell[3]*vec.X+Cell[7]*vec.Y+Cell[11]*vec.Z+Cell[15]*vec.W;
temp.X/=temp.W;
temp.Y/=temp.W;
temp.Z/=temp.W;
return(temp);
}

template<typename T>
T XE_MATRIX_TEMPLATE<T>::Determinant()
{
T temp
=Cell[0]*Cell[5]*Cell[10]*Cell[15]+Cell[0]*Cell[9]*Cell[14]*Cell[7]+Cell[0]*Cell[13]*Cell[6]*Cell[11]
+Cell[4]*Cell[1]*Cell[14]*Cell[11]+Cell[4]*Cell[9]*Cell[2]*Cell[15]+Cell[4]*Cell[13]*Cell[10]*Cell[3]
+Cell[8]*Cell[1]*Cell[6]*Cell[15]+Cell[8]*Cell[5]*Cell[14]*Cell[3]+Cell[8]*Cell[13]*Cell[2]*Cell[7]
+Cell[12]*Cell[1]*Cell[10]*Cell[7]+Cell[12]*Cell[5]*Cell[2]*Cell[11]+Cell[12]*Cell[9]*Cell[6]*Cell[3]
-Cell[0]*Cell[5]*Cell[14]*Cell[11]-Cell[0]*Cell[9]*Cell[6]*Cell[15]-Cell[0]*Cell[13]*Cell[10]*Cell[7]
-Cell[4]*Cell[1]*Cell[10]*Cell[15]-Cell[4]*Cell[9]*Cell[14]*Cell[3]-Cell[4]*Cell[13]*Cell[2]*Cell[11]
-Cell[8]*Cell[1]*Cell[14]*Cell[7]-Cell[8]*Cell[5]*Cell[2]*Cell[15]-Cell[8]*Cell[13]*Cell[6]*Cell[3]
-Cell[12]*Cell[1]*Cell[6]*Cell[11]-Cell[12]*Cell[5]*Cell[10]*Cell[3]-Cell[12]*Cell[9]*Cell[2]*Cell[7];
return(temp);
}

template<typename T>
XE_MATRIX_TEMPLATE<T> XE_MATRIX_TEMPLATE<T>::Inverse()
{
T det=Determinant();
XE_MATRIX_TEMPLATE<T> _temp;
XE_MATRIX_TEMPLATE<T> temp=_temp.ScalarMultiply(((T)1)/det);
return(temp);
}

template<typename T>
XE_MATRIX_TEMPLATE<T> XE_MATRIX_TEMPLATE<T>::Translate(T x,T y,T z)
{
XE_MATRIX_TEMPLATE<T> temp;
XE_MATRIX_TEMPLATE<T> mat;
mat.Cell[12]=(T)x;
mat.Cell[13]=(T)y;
mat.Cell[14]=(T)z;
temp=Multiply(mat);
return(temp);
}

template<typename T>
XE_MATRIX_TEMPLATE<T> XE_MATRIX_TEMPLATE<T>::Scale(T x,T y,T z)
{
XE_MATRIX_TEMPLATE<T> temp;
XE_MATRIX_TEMPLATE<T> mat;
mat.Cell[0]=(T)x;
mat.Cell[5]=(T)y;
mat.Cell[10]=(T)z;
temp=Multiply(mat);
return(temp);
}

template<typename T>
XE_MATRIX_TEMPLATE<T> XE_MATRIX_TEMPLATE<T>::RotateX(T angle)
{
XE_MATRIX_TEMPLATE<T> temp;
XE_MATRIX_TEMPLATE<T> mat;
mat.Cell[5]=cos(XE_MATH::DegToRad((T)angle));
mat.Cell[6]=sin(XE_MATH::DegToRad((T)angle));
mat.Cell[9]=-sin(XE_MATH::DegToRad((T)angle));
mat.Cell[10]=cos(XE_MATH::DegToRad((T)angle));
temp=Multiply(mat);
return(temp);
}

template<typename T>
XE_MATRIX_TEMPLATE<T> XE_MATRIX_TEMPLATE<T>::RotateY(T angle)
{
XE_MATRIX_TEMPLATE<T> temp;
XE_MATRIX_TEMPLATE<T> mat;
mat.Cell[0]=cos(XE_MATH::DegToRad((T)angle));
mat.Cell[2]=-sin(XE_MATH::DegToRad((T)angle));
mat.Cell[8]=sin(XE_MATH::DegToRad((T)angle));
mat.Cell[10]=cos(XE_MATH::DegToRad((T)angle));
temp=Multiply(mat);
return(temp);
}

template<typename T>
XE_MATRIX_TEMPLATE<T> XE_MATRIX_TEMPLATE<T>::RotateZ(T angle)
{
XE_MATRIX_TEMPLATE<T> temp;
XE_MATRIX_TEMPLATE<T> mat;
mat.Cell[0]=cos(XE_MATH::DegToRad((T)angle));
mat.Cell[1]=sin(XE_MATH::DegToRad((T)angle));
mat.Cell[4]=-sin(XE_MATH::DegToRad((T)angle));
mat.Cell[5]=cos(XE_MATH::DegToRad((T)angle));
temp=Multiply(mat);
return(temp);
}

template<typename T>
XE_MATRIX_TEMPLATE<T> XE_MATRIX_TEMPLATE<T>::operator+ (XE_MATRIX_TEMPLATE<T>& mat)
{
return(Add(mat));
}

template<typename T>
XE_MATRIX_TEMPLATE<T> XE_MATRIX_TEMPLATE<T>::operator- (XE_MATRIX_TEMPLATE<T>& mat)
{
return(Substract(mat));
}

template<typename T>
XE_MATRIX_TEMPLATE<T> XE_MATRIX_TEMPLATE<T>::operator* (T value)
{
return(ScalarMultiply((T)value));
}

template<typename T>
XE_MATRIX_TEMPLATE<T> XE_MATRIX_TEMPLATE<T>::operator* (XE_MATRIX_TEMPLATE<T>& mat)
{
return(Multiply(mat));
}

template<typename T>
XE_VECTOR_TEMPLATE<T> XE_MATRIX_TEMPLATE<T>::operator* (XE_VECTOR_TEMPLATE<T>& vec)
{
return(Multiply(vec));
}

/*--- XE_PAIR ---*/
//! Klasa pary klucz-wartośść.
template <typename TK,typename TV>
class XE_PAIR
{
public:
TK Key;
TV Value;
};

/*--- XE_ASSOC ---*/
//! Klasa tablicy asocjacyjnej.
template <typename TK,typename TV>
class XE_ASSOC
{
private:
XE_ELEMENTS< XE_PAIR<TK,TV> > Array;
public:
/*! Zwraca pierwszy element tablicy.
\return Element pola tablicy.
*/
XE_ELEMENT_POINTER< XE_PAIR<TK,TV> > FirstPointer();
/*! Czysci zawartosc tablicy. */
void Clear();
/*! Kasuje element tablicy.
\param key Klucz.
\return true jesli operacja powiodla sie.
*/
bool Erase(TK key);
/*! Zmienia klucz danej pary.
\param oldkey Stary klucz.
\param newkey Nowy klucz.
\return true jesli operacja powiodla sie.
*/
bool Change(TK oldkey,TK newkey);
/*! Zwraca ilosc elementow w tablicy.
\return Ilosc elementow.
*/
unsigned int Size();
/*! Operator wyłuskania wartości po kluczu.
\param key Klucz.
*/
TV& operator[] (TK key);
};

template <typename TK,typename TV>
XE_ELEMENT_POINTER< XE_PAIR<TK,TV> > XE_ASSOC<TK,TV>::FirstPointer()
{
return(Array.FirstPointer());
}

template <typename TK,typename TV>
void XE_ASSOC<TK,TV>::Clear()
{
Array.Clear();
}

template <typename TK,typename TV>
bool XE_ASSOC<TK,TV>::Erase(TK key)
{
XE_ELEMENT_POINTER< XE_PAIR<TK,TV> > elm;
for(elm=Array.FirstPointer();!elm.IsEmpty();elm.Next())
if(elm.iPointer()->Key==key)
{
Array.ErasePointer(elm);
return(true);
}
return(false);
}

template <typename TK,typename TV>
bool XE_ASSOC<TK,TV>::Change(TK oldkey,TK newkey)
{
XE_ELEMENT_POINTER< XE_PAIR<TK,TV> > elm;
for(elm=Array.FirstPointer();!elm.IsEmpty();elm.Next())
if(elm.iPointer()->Key==oldkey)
{
elm.iPointer()->Key=newkey;
return(true);
}
return(false);
}

template <typename TK,typename TV>
unsigned int XE_ASSOC<TK,TV>::Size()
{
return(Array.Size());
}

template <typename TK,typename TV>
TV& XE_ASSOC<TK,TV>::operator[] (TK key)
{
XE_ELEMENT_POINTER< XE_PAIR<TK,TV> > elm;
for(elm=Array.FirstPointer();!elm.IsEmpty();elm.Next())
if(elm.iPointer()->Key==key)
return(elm.iPointer()->Value);
XE_PAIR<TK,TV> pair;
TV temp;
pair.Key=key;
pair.Value=temp;
elm=Array.AddPointer(pair);
return(elm.iPointer()->Value);
}

/*--- XE_UNI ---*/
/*! Klasa wartosci uniwersalnej (tekstowo-liczbowa). */
class XE_UNI
{
private:
void* Data;
unsigned char Type; // 0: undefined; 1: number; 2: string; 3: assoc;
public:
typedef XE_ASSOC< XE_UNI, XE_UNI > ASSOC;
/*! Ustala czy pamiec bufora jest statyczna (jesli true to podczas niszczenia unii nie zostaje zwalniana pamiec bufora). */
bool Static;
/*! Konstruktor domyslny. */
XE_UNI();
/*! Konstruktor kopiujacy. */
XE_UNI(const XE_UNI& value);
/*! Konstruktor wypelniajacy. */
XE_UNI(XE_NUMBER value);
/*! Konstruktor wypelniajacy. */
XE_UNI(XE_STRING& value);
/*! Konstruktor wypelniajacy. */
XE_UNI(double value);
/*! Konstruktor wypelniajacy. */
XE_UNI(char* value);
/*! Konstruktor wypelniajacy. */
XE_UNI(ASSOC& value);
/*! Destruktor. */
~XE_UNI();
/*! Pobiera wartosc liczbowa.
\return Wartosc liczbowa.
*/
XE_NUMBER GetNumber();
/*! Pobiera wartosc tekstowa.
\return Wartosc tekstowa.
*/
XE_STRING GetString();
/*! Pobiera wartosc liczbowa (double).
\return Wartosc liczbowa.
*/
double GetPureNumber();
/*! Pobiera wartosc tekstowa (cstring).
\return Wartosc tekstowa.
*/
char* GetPureString();
/*! Pobiera zawartosc tablicy asocjacyjnej.
\param result Referencja na docelowa tablice asocjacyjna.
*/
void GetAssoc(ASSOC& result);
/*! Pobiera wskaznik na tablice asocjacyjna.
\return Wskaznik na tablice asocjacyjna.
*/
ASSOC* GetPureAssoc();
/*! Zwraca typ przechowywanej wartosci.
\return Typ wartosci. Dostepne wartosci:<br>
'n': Liczbowy.<br>
's': Tekstowy.<br>
'a': Asocjacyjny.
*/
unsigned char GetType();
/*! Sprawdza czy ma typ niezdefiniowany.
\return true jesli typ niezdefiniowany.
*/
bool IsUndefined();
/*! Sprawdza czy ma typ liczbowy.
\return true jesli typ liczbowy.
*/
bool IsNumber();
/*! Sprawdza czy ma typ tekstowy.
\return true jesli typ tekstowy.
*/
bool IsString();
/*! Sprawdza czy ma typ asocjacyjny.
\return true jesli typ asocjacyjny.
*/
bool IsAssoc();
/*! Ustala wartosc liczbowa.
\param value Referencja na wartosc liczbowa.
*/
void SetNumber(XE_NUMBER& value);
/*! Ustala wartosc tekstowa.
\param value Referencja na wartosc tekstowa.
*/
void SetString(XE_STRING& value);
/*! Ustala zawartosc tablicy asocjacyjnej.
\param value Referencja na tablice asocjacyjna.
*/
void SetAssoc(ASSOC& value);
/*! Kasuje wartosc. */
void Unset();
/*! Porownuje z inna wartoscia. */
bool Compare(XE_UNI& value);
/*! Operator przypisania.
\param value Wartosc uniwersalna.
*/
void operator= (const XE_UNI& value);
/*! Operator przypisania.
\param value Referencja na wartosc liczbowa.
*/
void operator= (XE_NUMBER& value);
/*! Operator przypisania.
\param value Referencja na wartosc tekstowa.
*/
void operator= (XE_STRING& value);
/*! Operator przypisania.
\param value Referencja na tablice asocjacyjna.
*/
void operator= (ASSOC& value);
/*! Operator przypisania.
\param value Wartosc double.
*/
void operator= (double value);
/*! Operator przypisania.
\param value Wartosc cstring.
*/
void operator= (char* value);
/*! Operator rzutowania. */
operator XE_NUMBER() const;
/*! Operator rzutowania. */
operator XE_STRING() const;
/*! Operator rzutowania. */
operator double() const;
/*! Operator rzutowania. */
operator char*() const;
/*! Operator porownania.
\param value Wartosc uniwersalna.
*/
bool operator== (XE_UNI& value);
/*! Operator porownania.
\param value Wartosc uniwersalna.
*/
bool operator!= (XE_UNI& value);
/*! Operator dostepu do pola tablicy asocjacyjnej.
\param key Referencja na klucz.
\return Referencja na wartosc.
*/
XE_UNI& operator[] (XE_UNI& key);
};

#ifdef XE_CANIMP

XE_UNI::XE_UNI()
{
Data=0;
Type=0;
Static=false;
}

XE_UNI::XE_UNI(const XE_UNI& value)
{
Data=0;
Type=0;
Static=false;
if(value.Type==1)
SetNumber(((XE_UNI&)value).GetNumber());
else
if(value.Type==2)
SetString(((XE_UNI&)value).GetString());
else
if(value.Type==3)
SetAssoc(*(ASSOC*)(((XE_UNI&)value).Data));
}

XE_UNI::XE_UNI(XE_NUMBER value)
{
Data=0;
Type=0;
Static=false;
SetNumber(value);
}

XE_UNI::XE_UNI(XE_STRING& value)
{
Data=0;
Type=0;
Static=false;
SetString(value);
}

XE_UNI::XE_UNI(double value)
{
Data=0;
Type=0;
Static=false;
SetNumber(XE_NUMBER(value));
}

XE_UNI::XE_UNI(char* value)
{
Data=0;
Type=0;
Static=false;
SetString(XE_STRING(value));
}

XE_UNI::XE_UNI(ASSOC& value)
{
Data=0;
Type=0;
Static=false;
SetAssoc(value);
}

XE_UNI::~XE_UNI()
{
if(!Static)
Unset();
}

XE_NUMBER XE_UNI::GetNumber()
{
if(Data)
{
if(Type==1)
return(*(XE_NUMBER*)Data);
if(Type==2)
{
double temp=0;
((XE_STRING*)Data)->Convert(&temp,'r');
return(XE_NUMBER(temp));
}
if(Type==3)
{
if(((ASSOC*)Data)->Size())
return(((ASSOC*)Data)->FirstPointer()->Value.GetNumber());
return(XE_NUMBER());
}
}
return(XE_NUMBER());
}

XE_STRING XE_UNI::GetString()
{
if(Data)
{
if(Type==1)
{
XE_STRING temp;
temp.Format("%f",((XE_NUMBER*)Data)->Get());
return(temp);
}
if(Type==2)
return(*(XE_STRING*)Data);
if(Type==3)
{
if(((ASSOC*)Data)->Size())
return(((ASSOC*)Data)->FirstPointer()->Value.GetString());
return(XE_STRING());
}
}
return(XE_STRING());
}

double XE_UNI::GetPureNumber()
{
if(Data)
{
if(Type==1)
return(((XE_NUMBER*)Data)->Get());
}
return(0);
}

char* XE_UNI::GetPureString()
{
if(Data)
{
if(Type==2)
return(((XE_STRING*)Data)->Get());
}
return(0);
}

void XE_UNI::GetAssoc(XE_UNI::ASSOC& result)
{
if(Data)
{
if(Type==1)
{
result.Clear();
result[XE_NUMBER(0)]=*this;
}
if(Type==2)
{
result.Clear();
result[XE_NUMBER(0)]=*this;
}
if(Type==3)
{
result.Clear();
for(XE_ELEMENT_POINTER<XE_PAIR<XE_UNI,XE_UNI> > elm=((ASSOC*)Data)->FirstPointer();!elm.IsEmpty();elm.Next())
result[elm->Key]=elm->Value;
}
}
}

XE_UNI::ASSOC* XE_UNI::GetPureAssoc()
{
if(Data)
{
if(Type==3)
return((ASSOC*)Data);
}
return(0);
}

unsigned char XE_UNI::GetType()
{
if(Type==1)return('n');
if(Type==2)return('s');
if(Type==3)return('a');
return(0);
}

bool XE_UNI::IsUndefined()
{
return((Type==0)?true:false);
}

bool XE_UNI::IsNumber()
{
return((Type==1)?true:false);
}

bool XE_UNI::IsString()
{
return((Type==2)?true:false);
}

bool XE_UNI::IsAssoc()
{
return((Type==3)?true:false);
}

void XE_UNI::SetNumber(XE_NUMBER& value)
{
Unset();
Data=new XE_NUMBER;
((XE_NUMBER*)Data)->Set(value.Get());
Type=1;
}

void XE_UNI::SetString(XE_STRING& value)
{
Unset();
Data=new XE_STRING;
((XE_STRING*)Data)->Set(value.Get());
Type=2;
}

void XE_UNI::SetAssoc(ASSOC& value)
{
Unset();
Data=new ASSOC;
for(XE_ELEMENT_POINTER<XE_PAIR<XE_UNI,XE_UNI> > elm=value.FirstPointer();!elm.IsEmpty();elm.Next())
((ASSOC*)Data)->operator[](elm->Key)=elm->Value;
Type=3;
}

void XE_UNI::Unset()
{
if(Data)
{
if(Type==1)
{
delete (XE_NUMBER*)Data;
}
if(Type==2)
{
((XE_STRING*)Data)->Clear();
delete (XE_STRING*)Data;
}
if(Type==3)
{
((ASSOC*)Data)->Clear();
delete (ASSOC*)Data;
}
}
Data=0;
Type=0;
}

bool XE_UNI::Compare(XE_UNI& value)
{
if(Type==value.Type)
{
if(Type==0)return(true);
if(Type==1)if((*(XE_NUMBER*)Data)==(*(XE_NUMBER*)value.Data))return(true);
if(Type==2)if((*(XE_STRING*)Data)==(*(XE_STRING*)value.Data))return(true);
if(Type==3)
{
if(((ASSOC*)Data)->Size()!=((ASSOC*)value.Data)->Size())
return(false);
for(XE_ELEMENT_POINTER<XE_PAIR<XE_UNI,XE_UNI> > elm1=((ASSOC*)value.Data)->FirstPointer(),elm2=((ASSOC*)Data)->FirstPointer();!elm1.IsEmpty()&&!elm2.IsEmpty();elm1.Next(),elm2.Next())
if(elm1->Key!=elm2->Key||elm1->Value!=elm2->Value)
return(false);
return(true);
}
}
return(false);
}

void XE_UNI::operator= (const XE_UNI& value)
{
if(value.Type==1)
SetNumber(((XE_UNI&)value).GetNumber());
else
if(value.Type==2)
SetString(((XE_UNI&)value).GetString());
else
if(value.Type==3)
SetAssoc(*((XE_UNI&)value).GetPureAssoc());
else
Unset();
}

void XE_UNI::operator= (XE_NUMBER& value)
{
SetNumber(value);
}

void XE_UNI::operator= (XE_STRING& value)
{
SetString(value);
}

void XE_UNI::operator= (double value)
{
SetNumber(XE_NUMBER(value));
}

void XE_UNI::operator= (char* value)
{
SetString(XE_STRING(value));
}

void XE_UNI::operator= (ASSOC& value)
{
SetAssoc(value);
}

XE_UNI::operator XE_NUMBER() const
{
return(((XE_UNI*)this)->GetNumber());
}

XE_UNI::operator XE_STRING() const
{
return(((XE_UNI*)this)->GetString());
}

XE_UNI::operator double() const
{
if(Type==0)return(0);
return(((XE_UNI*)this)->GetNumber().Get());
}

XE_UNI::operator char*() const
{
if(Type==0)return(0);
return(((XE_UNI*)this)->GetString().Get());
}

bool XE_UNI::operator== (XE_UNI& value)
{
return(Compare(value));
}

bool XE_UNI::operator!= (XE_UNI& value)
{
return(!Compare(value));
}

XE_UNI& XE_UNI::operator[] (XE_UNI& key)
{
return(GetPureAssoc()->operator[](key));
}

#endif /* XE_CANIMP */

/*--- XE_STACK_BUFFER ---*/
//! Klasa bufora stosu.
template<typename T>
class XE_STACK_BUFFER
{
private:
T* stack;
int maxsize;
int count;
public:
/*! Konstruktor domyslny. */
XE_STACK_BUFFER();
/*! Konstruktor inicjujacy. */
XE_STACK_BUFFER(int size);
/*! Destruktor. */
~XE_STACK_BUFFER();
/*! Uklada obiekt na stosie. */
void Push(T val);
/*! Zdejmuje obiekt ze stosu. */
T Pop();
/*! Zwraca ilosc elementow na stosie. */
int Count();
/*! Zwraca rozmiar stosu. */
int MaxSize();
/*! Zwraca wskaznik na bufor stosu. */
T* Buffer();
/*! Czysci stos. */
void Clear();
/*! Odklada wiecej niz jeden obiekt na stosie. */
void PushMulti(int size,...);
/*! Czysci stos, nastepnie odklada wiecej niz jeden obiekt na stosie. */
void ClearPushMulti(int size,...);
};

template<typename T>
XE_STACK_BUFFER<T>::XE_STACK_BUFFER()
{
stack=new T[1];
maxsize=1;
count=0;
}

template<typename T>
XE_STACK_BUFFER<T>::XE_STACK_BUFFER(int size)
{
stack=new T[max(size,1)];
maxsize=max(size,1);
count=0;
}

template<typename T>
XE_STACK_BUFFER<T>::~XE_STACK_BUFFER()
{
delete stack;
maxsize=0;
count=0;
}

template<typename T>
void XE_STACK_BUFFER<T>::Push(T val)
{
if(count>=maxsize)return;
stack[count++]=val;
}

template<typename T>
T XE_STACK_BUFFER<T>::Pop()
{
if(count<=0)return(T());
return(stack[count--]);
}

template<typename T>
int XE_STACK_BUFFER<T>::Count()
{
return(count);
}

template<typename T>
int XE_STACK_BUFFER<T>::MaxSize()
{
return(maxsize);
}

template<typename T>
T* XE_STACK_BUFFER<T>::Buffer()
{
return(stack);
}

template<typename T>
void XE_STACK_BUFFER<T>::Clear()
{
count=0;
}

template<typename T>
void XE_STACK_BUFFER<T>::PushMulti(int size,...)
{
va_list vl;
va_start(vl,size);
for(int i=0;i<size;i++)Push(va_arg(vl,T));
va_end(vl);
}

template<typename T>
void XE_STACK_BUFFER<T>::ClearPushMulti(int size,...)
{
count=0;
va_list vl;
va_start(vl,size);
for(int i=0;i<size;i++)Push(va_arg(vl,T));
va_end(vl);
}

/*--- XE_DYNAMIC_BUFFER ---*/
//! Klasa dynamicznych buforow
template<typename T>
class XE_DYNAMIC_BUFFER
{
private:
XE_ELEMENTS<XE_PAIR<T*,unsigned int> > Buffers;
public:
//! Defenicja typu iteratora.
typedef XE_ELEMENT_POINTER<XE_PAIR<T*,unsigned int> > ITERATOR;
/*! Konstruktor domyslny. */
XE_DYNAMIC_BUFFER();
/*! Destruktor. */
~XE_DYNAMIC_BUFFER();
/*! Dodanie bufora.
\param source Wskaznik na tablice danych.
\param count Ilosc elementow do pobrania.
\return Element iteratora bufora.
*/
ITERATOR Add(T* source,unsigned int count);
/*! Zamiana bufora.
\param elm Element iteratora bufora.
\param source Wskaznik na tablice danych.
\param count Ilosc elementow do pobrania.
\return true jesli operacja powiodla sie.
*/
bool Replace(ITERATOR elm,T* source,unsigned int count);
/*! Usuniecie bufora.
\param elm Element iteratora bufora.
*/
void Erase(ITERATOR elm);
/*! Zwraca liczbe buforow.
\return Ilosc buforow.
*/
unsigned int Count();
/*! Zwraca wielkosc bufora, sumujac rozmiary wszystkich buforow.
\return Suma rozmiaru buforow.
*/
unsigned int Size();
/*! Zwraca wielkosc bufora w bajtach, sumujac rozmiary wszystkich buforow.
\return Suma rozmiaru buforow w bajtach.
*/
unsigned int Bytesize();
/*! Buduje jeden bufor laczac wszystkie i zwraca jego wskaznik lub wartosc NULL jesli nie utworzy bufora.
\param extra Ilosc dodatkowego miejsca dla elementow (liczba elementow).
\param offset Liczba bajtow przesuniecia (mniejsza badz rowna ilosci dodatkowego miejsca w bajtach).
\return Wskaznik na utworzony bufor.
*/
T* Build(unsigned int extra=0,unsigned int offset=0);
/*! Zwraca iterator pierwszego bufora.
\return Iterator pierwszego bufora.
*/
ITERATOR First();
/*! Zwraca iterator ostatniego bufora.
\return Iterator ostatniego bufora.
*/
ITERATOR Last();
/*! Usuwa wszystkie bufory. */
void Clear();
};

template<typename T>
XE_DYNAMIC_BUFFER<T>::XE_DYNAMIC_BUFFER()
{
}

template<typename T>
XE_DYNAMIC_BUFFER<T>::~XE_DYNAMIC_BUFFER()
{
Clear();
}

template<typename T>
XE_ELEMENT_POINTER<XE_PAIR<T*,unsigned int> > XE_DYNAMIC_BUFFER<T>::Add(T* source,unsigned int count)
{
if(count==0)return(ITERATOR());
XE_PAIR<T*,unsigned int> temp;
temp.Key=new T[count];
memcpy(temp.Key,source,count);
temp.Value=count;
return(Buffers.AddPointer(temp));
}

template<typename T>
bool XE_DYNAMIC_BUFFER<T>::Replace(ITERATOR elm,T* source,unsigned int count)
{
if(elm.IsEmpty())return(false);
if(elm->Key)delete[] elm->Key;
XE_PAIR<T*,unsigned int> temp;
temp.Key=new T[count];
memcpy(temp.Key,source,count);
temp.Value=count;
*elm.iPointer()=temp;
return(true);
}

template<typename T>
void XE_DYNAMIC_BUFFER<T>::Erase(ITERATOR elm)
{
if(elm.IsEmpty())return;
if(elm->Key)delete[] elm->Key;
Buffers.ErasePointer(elm);
}

template<typename T>
unsigned int XE_DYNAMIC_BUFFER<T>::Count()
{
return(Buffers.Size());
}

template<typename T>
unsigned int XE_DYNAMIC_BUFFER<T>::Size()
{
unsigned int size=0;
for(XE_ELEMENT_POINTER<XE_PAIR<T*,unsigned int> > elm=Buffers.FirstPointer();!elm.IsEmpty();elm.Next())
size+=elm->Value;
return(size);
}

template<typename T>
unsigned int XE_DYNAMIC_BUFFER<T>::Bytesize()
{
return(Size()*sizeof(T));
}

template<typename T>
T* XE_DYNAMIC_BUFFER<T>::Build(unsigned int extra,unsigned int offset)
{
unsigned int size=Size();
unsigned int off=min(extra*sizeof(T),offset);
if(size+extra==0)return(NULL);
T* buff=NULL;
buff=new T[size+extra];
if(!buff)return(NULL);
T* pos=buff+off;
for(XE_ELEMENT_POINTER<XE_PAIR<T*,unsigned int> > elm=Buffers.FirstPointer();!elm.IsEmpty();elm.Next())
{
memcpy(pos,elm->Key,elm->Value*sizeof(T));
pos+=elm->Value*sizeof(T);
}
return(buff);
}

template<typename T>
XE_ELEMENT_POINTER<XE_PAIR<T*,unsigned int> > XE_DYNAMIC_BUFFER<T>::First()
{
return(Buffers.FirstPointer());
}

template<typename T>
XE_ELEMENT_POINTER<XE_PAIR<T*,unsigned int> > XE_DYNAMIC_BUFFER<T>::Last()
{
return(Buffers.LastPointer());
}

template<typename T>
void XE_DYNAMIC_BUFFER<T>::Clear()
{
for(XE_ELEMENT_POINTER<XE_PAIR<T*,unsigned int> > elm=Buffers.FirstPointer();!elm.IsEmpty();elm.Next())
if(elm->Key)delete[] elm->Key;
Buffers.Clear();
}

/*--- XE_FONTMAP ---*/
/*! Klasa mapy czcionki. */
class XE_FONTMAP
{
public:
//! Klasa opisu znaku.
class CHAR
{
public:
//! Szerokosc.
float Width;
//! Wysokosc.
float Height;
//! Przesuniecie poziome.
float Xoffset;
//! Przesuniecie pionowe.
float Yoffset;
//! Przesuniecie pozycji po wypisaniu znaku.
float Advance;
//! Poczatkowy koordynat X.
float Xbegin;
//! Poczatkowy koordynat Y.
float Ybegin;
//! Koncowy koordynat X.
float Xend;
//! Koncowy koordynat Y.
float Yend;
/*! Konstruktor domyslny. */
CHAR();
};
//! Pozioma separacja.
float Xsep;
//! Pionowa separacja.
float Ysep;
//! Znak nowej linii;
char NewLine;
//! Rozmiar czcionki.
float Size;
//! Opis wszystkich znakow.
CHAR Desc[256];
/*! Konstruktor domyslny. */
XE_FONTMAP();
/*! Kompiluje tekstowy opis mapy z pliku.
\param fname Sciezka do pliku.
\return true jesli operacja powiodla sie.
*/
bool Compile(char* fname);
/*! Kompiluje tekstowy opis mapy z pamieci.
\param buff Wskaznik na bufor danych tekstowe opisu.
\param size Rozmiar bufora (w bajtach).
\return true jesli operacja powiodla sie.
*/
bool CompileFromMemory(void* buff,unsigned int size);
};

#ifdef XE_CANIMP

XE_FONTMAP::CHAR::CHAR()
{
Width=0;
Height=0;
Xoffset=0;
Yoffset=0;
Advance=0;
Xbegin=0;
Ybegin=0;
Xend=0;
Yend=0;
}

XE_FONTMAP::XE_FONTMAP()
{
Xsep=0;
Ysep=0;
NewLine='\n';
Size=0;
}

bool XE_FONTMAP::Compile(char* fname)
{
FILE* f=0;
fopen_s(&f,fname,"rb");
if(!f)return(false);
fseek(f,0,SEEK_END);
unsigned int size=ftell(f);
rewind(f);
char* buff=new char[size];
if(!buff)
{
fclose(f);
return(false);
}
fread(buff,size,1,f);
fclose(f);
bool status=CompileFromMemory(buff,size);
delete[] buff;
return(status);
}

bool XE_FONTMAP::CompileFromMemory(void* buff,unsigned int size)
{
if(!buff||!size)return(false);
int tw=0;
int th=0;
int fs=0;
XE_STRING param;
XE_STRING value;
unsigned int linesc=0;
XE_STRING* lines=XE_STRING::ExplodeBuff((char*)buff,size,"\r\n",linesc);
if(!lines||!linesc)return(false);
for(unsigned int i=0;i<linesc;i++)
{
unsigned int wordsc=0;
XE_STRING* words=lines[i].Explode(" \t",wordsc,"\"");
if(!words||!wordsc)continue;
if(words[0]=="common")
{
for(unsigned int j=1;j<wordsc;j++)
{
param.Clear();
value.Clear();
words[j].Search(value,"\n",words[j].Search(param,"="));
if(param=="lineHeight")value.Convert(&fs,'i');
if(param=="scaleW")value.Convert(&tw,'i');
if(param=="scaleH")value.Convert(&th,'i');
}
Size=(float)fs;
}
if(words[0]=="char")
{
int id=0;
int x=0;
int y=0;
int width=0;
int height=0;
int xoffset=0;
int yoffset=0;
int xadvance=0;
for(unsigned int j=1;j<wordsc;j++)
{
param.Clear();
value.Clear();
words[j].Search(value,"\n",words[j].Search(param,"="));
if(param=="id")
value.Convert(&id,'i');
if(param=="x")value.Convert(&x,'i');
if(param=="y")value.Convert(&y,'i');
if(param=="width")value.Convert(&width,'i');
if(param=="height")value.Convert(&height,'i');
if(param=="xoffset")value.Convert(&xoffset,'i');
if(param=="yoffset")value.Convert(&yoffset,'i');
if(param=="xadvance")value.Convert(&xadvance,'i');
}
if(id>=0&&id<256)
{
Desc[id].Width=(float)width;
Desc[id].Height=(float)height;
Desc[id].Xoffset=(float)xoffset;
Desc[id].Yoffset=(float)yoffset;
Desc[id].Advance=(float)xadvance;
Desc[id].Xbegin=(float)x/(float)tw;
Desc[id].Ybegin=(float)y/(float)th;
Desc[id].Xend=((float)x+(float)width)/(float)tw;
Desc[id].Yend=((float)y+(float)height)/(float)th;
}
}
delete[] words;
}
delete[] lines;
return(true);
}

#endif /* XE_CANIMP */

/*--- XE_MARKUP ---*/
//! Klasa drzewa znacznikow.
class XE_MARKUP
{
public:
//! Typ wyliczeniowy.
enum ENUM
{
CREATE,
DESTROY,
FIND
};
//! Klasa znacznika.
class TAG
{
public:
//! Nazwa.
XE_STRING Name;
//! Tresc znacznika.
XE_STRING Text;
//! Kontener atrybutow.
XE_ELEMENTS< XE_PAIR<XE_STRING,XE_STRING> > Attribs;
/*! Konstruktor domyslny. */
TAG();
};
//! Definicja typu elementu atrybutu.
typedef XE_ELEMENT_POINTER< XE_PAIR<XE_STRING,XE_STRING> > ATTRIB;
//! Definicja typu elementu tagu.
typedef XE_NODE<TAG>::NODE ETAG;
//! Definicja typu wskaznika tagu.
typedef XE_NODE<TAG>* PTAG;
private:
void TagsToString(XE_STRING& result,PTAG tag,int level);
public:
//! Drzewo znacznikow.
XE_NODE<TAG> Tags;
/*! Konstruktor domyslny. */
XE_MARKUP();
/*! Destruktor. */
~XE_MARKUP();
/*! Parsuje bufor.
\param buff Wskaznik na bufor danych.
\param size Rozmiar bufora.
\param prepost Znaki ignorowane na poczatku i koncu tresci znacznika.
\return true jesli operacja powiodla sie.
*/
bool Parse(char* buff,unsigned int size,char* prepost=0);
/*! Tworzy i wpisuje do bufora tekstowa tresc drzewa znacznikow.
\param buff Referencja na bufor docelowy.
\return true jesli operacja powiodla sie.
*/
bool Make(XE_BUFFER& buff);
/*! Zwalnia dane tagow z pamieci. */
void Free();
/*! Operator wyluskania.
\param path Sciezka do znacznika (tagi oddzielane znakiem: "/", opcjowane znakiem: "#". Dostepne opcje: new; last; id:0; next; prev ). Przyklad: "html/body/table#last#prev/td#id:2/tr#new". Tag "." oznacza ze operujemy na znaczniku z ragumentu tag, nie zas na jego dziecku. Tag ".." oznacza przejscie do znacznika nadrzednego.
\param action Typ akcji wykonywanej na tagu.
\param tag Wskaznik na znacznik w ktorym szukamy, badz wartosc NULL jesli szukamy w calym drzewie.
\return Element wezla znacznika.
*/
ETAG operator() (char* path,ENUM action=CREATE,PTAG tag=0);
/*! Operator wyluskania.
\param path Sciezka do znacznika (tagi oddzielane znakiem: "/", opcjowane znakiem: "#". Dostepne opcje: new; last; id:0; next; prev ). Przyklad: "html/body/table#last#prev/td#id:2/tr#new". Tag "." oznacza ze operujemy na znaczniku z ragumentu tag, nie zas na jego dziecku. Tag ".." oznacza przejscie do znacznika nadrzednego.
\param name Nazwa atrybutu.
\param action Typ akcji wykonywanej na tagu.
\param tag Wskaznik na znacznik w ktorym szukamy, badz wartosc NULL jesli szukamy w calym drzewie.
\return Element atrybutu znacznika.
*/
ATTRIB operator() (char* path,char* name,ENUM action=CREATE,PTAG tag=0);
};

#ifdef XE_CANIMP

XE_MARKUP::TAG::TAG()
{
}

void XE_MARKUP::TagsToString(XE_STRING& result,PTAG tag,int level)
{
if(!tag)return;
XE_STRING temp;
if(tag->Pointer()->Name.Length())
{
for(int _counter_=0;_counter_<level;_counter_++)temp<<"\t";
temp<<"<"<<tag->Pointer()->Name.Get();
if(tag->Pointer()->Attribs.Size())
for(XE_ELEMENT_POINTER< XE_PAIR<XE_STRING,XE_STRING> > elm=tag->Pointer()->Attribs.FirstPointer();!elm.IsEmpty();elm.Next())
if(elm->Key.Length())
temp<<" "<<elm->Key.Get()<<"=\""<<elm->Value.Get()<<"\"";
if(!tag->Count())
{
temp<<" />\r\n";
result<<temp.Get();
temp.Clear();
}
else
{
temp<<">\r\n";
result<<temp.Get();
temp.Clear();
for(XE_NODE<TAG>::NODE elm=tag->FirstChild();!elm.IsEmpty();elm.Next())
TagsToString(result,elm.iPointer(),level+1);
for(int _counter_=0;_counter_<level;_counter_++)temp<<"\t";
temp<<"</"<<tag->Pointer()->Name.Get()<<">\r\n";
result<<temp.Get();
temp.Clear();
}
}
else
{
if(tag->Pointer()->Text.Length())
{
temp.Clear();
for(int _counter_=0;_counter_<level;_counter_++)temp<<"\t";
temp<<tag->Pointer()->Text.Get()<<"\r\n";
result<<temp.Get();
temp.Clear();
}
}
}

XE_MARKUP::XE_MARKUP()
{
}

XE_MARKUP::~XE_MARKUP()
{
Free();
}

bool XE_MARKUP::Parse(char* buff,unsigned int size,char* prepost)
{
if(!buff||!size)return(false);
XE_STRING pp=prepost;
unsigned int ppl=pp.Length();
XE_STRING src;
src.Copy(buff,size);
XE_ELEMENTS<XE_PAIR<XE_STRING,int> > parts;
XE_STRING temp;
unsigned int srclen=src.Length();
unsigned int pos=0;
pos=src.Search(temp,"<",pos);
XE_PAIR<XE_STRING,int> pair;
while(pos)
{
XE_STRING pref;
pref.Copy(src.Get(pos),8);
if(pos+8+3<=srclen&&pref=="![CDATA[")
{
pos+=8;
temp.Clear();
pair.Key.Clear();
pos=src.Search(temp,"]]>",pos);
pair.Key=temp;
pair.Value=2;
if(pair.Key.Length())parts.AddPointer(pair);
}
else
{
temp.Clear();
pair.Key.Clear();
pos=src.Search(temp,">",pos);
pair.Key=temp;
pair.Value=0;
if(pair.Key.Length())parts.AddPointer(pair);
}
temp.Clear();
pair.Key.Clear();
pos=src.Search(temp,"<",pos);
if(temp.Length())
{
if(prepost)
{
char t;
bool doit;
unsigned int pre=0;
doit=true;
for(unsigned int i=0;i<temp.Length()&&doit;i++)
{
t=temp.Index(i);
for(unsigned int j=0;j<ppl&&doit;j++)
{
if(t!=pp.Index(j))
doit=false;
else
pre++;
}
}
unsigned int post=0;
doit=true;
for(unsigned int i=0;i<temp.Length()&&doit;i++)
{
t=temp.Index(i,true);
for(unsigned int j=0;j<ppl&&doit;j++)
{
if(t!=pp.Index(j))
doit=false;
else
post++;
}
}
if(pre+post<temp.Length())
pair.Key.Copy(temp.Get(pre),temp.Length()-pre-post);
}
else
pair.Key=temp;
pair.Value=1;
if(pair.Key.Length())parts.AddPointer(pair);
}
}
src.Clear();
XE_NODE<TAG>* parent=0;
for(XE_ELEMENT_POINTER<XE_PAIR<XE_STRING,int> > part=parts.FirstPointer();!part.IsEmpty();part.Next())
{
if(!parent)parent=&Tags;
if(part->Value==0)
{
if(part->Key.Length()>=5&&part->Key.Index(0)=='!'&&part->Key.Index(1)=='-'&&part->Key.Index(2)=='-'&&part->Key.Index(0,true)=='-'&&part->Key.Index(1,true)=='-')
continue;
else
if(part->Key.Index(0)!='/')
{
XE_NODE<TAG>::NODE tag=parent->AddChild(TAG());
XE_STRING prop;
if(part->Key.Length()>=2&&part->Key.Index(0)=='?'&&part->Key.Index(0,true)=='?')
prop.Copy(part->Key.Get(1),part->Key.Length()-2);
else
if(part->Key.Length()>=1&&part->Key.Index(0,true)=='/')
prop.Copy(part->Key.Get(),part->Key.Length()-1);
else
{
parent=tag.iPointer();
prop=part->Key;
}
unsigned int argc=0;
XE_STRING* args=prop.Explode(" \t\r\n",argc,"\"");
if(argc&&args)
{
tag->Pointer()->Name=args[0];
for(unsigned int i=1;i<argc;i++)
{
XE_PAIR<XE_STRING,XE_STRING> pair;
XE_STRING attr;
unsigned int p=args[i].Search(pair.Key,"=");
args[i].Search(attr,"\n",p);
if(attr.Length()>=2&&attr.Index(0)=='"'&&attr.Index(0,true)=='"')
pair.Value.Copy(attr.Get(1),attr.Length()-2);
else
pair.Value=attr;
if(pair.Key.Length())tag->Pointer()->Attribs.AddPointer(pair);
}
delete[] args;
}
else
{
parent=parent->GetParent();
parent->RemoveChild(tag);
}
}
else
parent=parent->GetParent();
}
else
if(part->Value==1)
{
XE_NODE<TAG>::NODE tag=parent->AddChild(TAG());
tag->Pointer()->Text=part->Key;
}
else
if(part->Value==2)
{
XE_NODE<TAG>::NODE tag=parent->AddChild(TAG());
tag->Pointer()->Name="<CDATA>";
tag->Pointer()->Text=part->Key;
}
}
parts.Clear();
return(true);
}

bool XE_MARKUP::Make(XE_BUFFER& buff)
{
if(!Tags.Count())return(false);
XE_STRING result;
for(XE_NODE<TAG>::NODE elm=Tags.FirstChild();!elm.IsEmpty();elm.Next())
TagsToString(result,elm.iPointer(),0);
buff.Reserve(result.Length());
buff.Write(result,result.Length());
result.Clear();
return(true);
}

void XE_MARKUP::Free()
{
Tags.Free();
}

XE_MARKUP::ETAG XE_MARKUP::operator() (char* path,ENUM action,PTAG tag)
{
XE_NODE<TAG>::NODE res;
XE_NODE<TAG>::NODE elm;
XE_STRING temp=path;
XE_STRING word;
if(!tag)tag=&Tags;
unsigned int pos=temp.Search(word,"/");
bool optnew=false;
bool optlast=false;
unsigned int optid=0;
if(word.Find(0,"#"))
{
temp.Clear();
XE_STRING option;
unsigned int op=word.Search(temp,"#");
while(op)
{
op=word.Search(option,"#",op);
if(option=="new")optnew=true;
if(option=="last")optlast=true;
if(option=="prev")if(optid)optid--;
if(option=="next")optid++;
if(option.Find(0,":"))
{
XE_STRING subopt;
XE_STRING subval;
option.Search(subval,":",option.Search(subopt,":"));
if(subopt=="id")subval.Convert(&optid,'d');
}
}
word=temp;
}
PTAG par=tag;
if(word==".")
{
if(tag->GetParent())
tag=tag->GetParent();
else
tag=&Tags;
}
if(word=="..")
{
if(tag->GetParent())
{
if(tag->GetParent()->GetParent())
tag=tag->GetParent()->GetParent();
else
tag=0;
}
else
tag=0;
}
if(!tag)return(ETAG());
ETAG start;
if(word!=".")
start=(!optlast)?(tag->FirstChild()):(tag->LastChild());
else
{
for(ETAG elm=tag->FirstChild();!elm.IsEmpty();elm.Next())
if(elm.iPointer()==par)
{
start=elm;
break;
}
}
if(optnew)goto __marknew__;
for(elm=start;!elm.IsEmpty();(!optlast)?(elm.Next()):(elm.Prev()))
if((word!="."&&word!="..")?(elm->Pointer()->Name==word):(true))
{
if(optid)
{
optid--;
continue;
}
if(pos)
return((*this)(&path[pos],action,elm.iPointer()));
else
{
if(action!=DESTROY)
return(elm);
else
{
tag->RemoveChild(elm);
return(ETAG());
}
}
}
__marknew__:
if(action==CREATE)
{
res=tag->AddChild(TAG());
if(res.IsEmpty())return(XE_NODE<TAG>::NODE());
res->Pointer()->Name=word;
if(pos)
return((*this)(&path[pos],action,res.iPointer()));
else
return(res);
}
return(XE_NODE<TAG>::NODE());
}

XE_MARKUP::ATTRIB XE_MARKUP::operator() (char* path,char* name,ENUM action,PTAG tag)
{
XE_NODE<TAG>::NODE res=(*this)(path,action,tag);
if(res.IsEmpty())return(ATTRIB());
for(ATTRIB elm=res->Pointer()->Attribs.FirstPointer();!elm.IsEmpty();elm.Next())
if(elm->Key==name)
return(elm);
ATTRIB atr=res->Pointer()->Attribs.AddPointer(XE_PAIR<XE_STRING,XE_STRING>());
if(atr.IsEmpty())return(ATTRIB());
atr->Key=name;
return(atr);
}

#endif /* XE_CANIMP */

#ifndef XE_NOT_USE_LIBRARY

/*--- XenoN Core Elements declaration ---*/
/*! Definicja elementu: Psyche. */
class XENON_CORE_PSYCHE;
typedef XE_ELEMENT_POINTER<XENON_CORE_PSYCHE> XE_ELM_PSYCHE;
/*! Definicja elementu: RenderTarget. */
class XENON_CORE_RENDERTARGET;
typedef XE_ELEMENT_POINTER<XENON_CORE_RENDERTARGET> XE_ELM_RENDERTARGET;
/*! Definicja elementu: FrameBuffer. */
class XENON_CORE_FRAMEBUFFER;
typedef XE_ELEMENT_POINTER<XENON_CORE_FRAMEBUFFER> XE_ELM_FRAMEBUFFER;
/*! Definicja elementu: VertexBuffer. */
class XENON_CORE_VERTEXBUFFER;
typedef XE_ELEMENT_POINTER<XENON_CORE_VERTEXBUFFER> XE_ELM_VERTEXBUFFER;
/*! Definicja elementu: Shader. */
class XENON_CORE_SHADER;
typedef XE_ELEMENT_POINTER<XENON_CORE_SHADER> XE_ELM_SHADER;
/*! Definicja elementu: RenderQueue. */
class XENON_CORE_RENDERQUEUE;
typedef XE_ELEMENT_POINTER<XENON_CORE_RENDERQUEUE> XE_ELM_RENDERQUEUE;
/*! Definicja elementu: Texture. */
class XENON_CORE_TEXTURE;
typedef XE_ELEMENT_POINTER<XENON_CORE_TEXTURE> XE_ELM_TEXTURE;
/*! Definicja elementu: Displaylist. */
class XENON_CORE_DISPLAYLIST;
typedef XE_ELEMENT_POINTER<XENON_CORE_DISPLAYLIST> XE_ELM_DISPLAYLIST;
/*! Definicja elementu: Substance. */
class XENON_CORE_SUBSTANCE;
typedef XE_ELEMENT_POINTER<XENON_CORE_SUBSTANCE> XE_ELM_SUBSTANCE;
/*! Definicja elementu: Ray. */
class XENON_CORE_RAY;
typedef XE_ELEMENT_POINTER<XENON_CORE_RAY> XE_ELM_RAY;
/*! Definicja elementu: Covalence Force. */
class XENON_CORE_COVALENCE_FORCE;
typedef XE_ELEMENT_POINTER<XENON_CORE_COVALENCE_FORCE> XE_ELM_COVALENCE_FORCE;
/*! Definicja elementu: Force. */
class XENON_CORE_FORCE;
typedef XE_ELEMENT_POINTER<XENON_CORE_FORCE> XE_ELM_FORCE;
/*! Definicja elementu: Surface. */
class XENON_CORE_SURFACE;
typedef XE_ELEMENT_POINTER<XENON_CORE_SURFACE> XE_ELM_SURFACE;
/*! Definicja elementu: Particle. */
class XENON_CORE_PARTICLE;
typedef XE_ELEMENT_POINTER<XENON_CORE_PARTICLE> XE_ELM_PARTICLE;
/*! Definicja elementu: Global Force. */
class XENON_CORE_GLOBAL_FORCE;
typedef XE_ELEMENT_POINTER<XENON_CORE_GLOBAL_FORCE> XE_ELM_GLOBAL_FORCE;
/*! Definicja elementu: Black Hole. */
class XENON_CORE_BLACK_HOLE;
typedef XE_ELEMENT_POINTER<XENON_CORE_BLACK_HOLE> XE_ELM_BLACK_HOLE;
/*! Definicja elementu: Wave Force. */
class XENON_CORE_WAVE_FORCE;
typedef XE_ELEMENT_POINTER<XENON_CORE_WAVE_FORCE> XE_ELM_WAVE_FORCE;
/*! Definicja elementu: DNA. */
class XENON_CORE_DNA;
typedef XE_ELEMENT_POINTER<XENON_CORE_DNA> XE_ELM_DNA;
/*! Definicja elementu: Socket. */
class XENON_CORE_SOCKET;
typedef XE_ELEMENT_POINTER<XENON_CORE_SOCKET> XE_ELM_SOCKET;

#endif /* XE_NOT_USE_LIBRARY */

// Uniform translation operators
class XE_ELM_UNIFORM
{
private:
XE_BYTE bytes[sizeof(XE_ELEMENT_POINTER<XE_DWORD>)];
public:
XE_ELM_UNIFORM();
XE_ELM_UNIFORM(void* elmptr);
void ConvertFrom(void* elmptr);
void ConvertTo(void* elmptr);
bool IsEmpty();
};

#ifdef XE_CANIMP

XE_ELM_UNIFORM::XE_ELM_UNIFORM()
{
*(XE_ELEMENT_POINTER<XE_DWORD>*)&bytes=XE_ELEMENT_POINTER<XE_DWORD>();
}

XE_ELM_UNIFORM::XE_ELM_UNIFORM(void* elmptr)
{
*(XE_ELEMENT_POINTER<XE_DWORD>*)&bytes=XE_ELEMENT_POINTER<XE_DWORD>();
ConvertFrom(elmptr);
}

void XE_ELM_UNIFORM::ConvertFrom(void *elmptr)
{
if(elmptr)*(XE_ELEMENT_POINTER<XE_DWORD>*)&bytes=*(XE_ELEMENT_POINTER<XE_DWORD>*)elmptr;
}

void XE_ELM_UNIFORM::ConvertTo(void *elmptr)
{
if(elmptr)*(XE_ELEMENT_POINTER<XE_DWORD>*)elmptr=*(XE_ELEMENT_POINTER<XE_DWORD>*)&bytes;
}

bool XE_ELM_UNIFORM::IsEmpty()
{
return(((XE_ELEMENT_POINTER<XE_DWORD>*)&bytes)->IsEmpty());
}

#endif /* XE_CANIMP */

#ifndef XE_NOT_USE_LIBRARY

#ifdef XE_COMPILE_CORE_VIRTUAL_FILE

/*--- XenoN Virtual File ---*/
class XVF_FILE
{
#ifdef XE_COMPILE_EXPORT
public:
#else
private:
#endif /* XE_COMPILE_EXPORT */
char Name[256];
unsigned int Size;
void* Data;
#ifdef XE_COMPILE_EXPORT
XVF_FILE();
XVF_FILE(char* name,unsigned int size,void* data);
void Free();
bool Empty();
void Encrypt(char* pass);
void Decrypt(char* pass);
#endif /* XE_COMPILE_EXPORT */
};

struct XVF_INFO
{
unsigned int FileOffset;
unsigned int FileSize;
char FileName[256];
};

#endif /* XE_COMPILE_CORE_VIRTUAL_FILE */

/*--- XE_FILE ---*/
//! Klasa bufora pliku.
class XE_FILE
{
//public:
//#ifdef XE_COMPILE_CORE_VIRTUAL_FILE
////! Klasa pliku wirtualnego.
//class VIRTUAL
//{
//public:
////! Nazwa pliku.
//XE_STRING Name;
////! Wskaznik na plik.
//XE_FILE* File;
///*! Konstruktor domyslny. */
//VIRTUAL();
//};
////! Klasa opisu archiwum.
//class ARCHIVE
//{
//friend class XE_FILE;
//private:
//XE_STRING Path;
//XE_ARRAY<XVF_INFO> FilesInfo;
//public:
///*! Konstruktor domyslny. */
//ARCHIVE();
///*! Otwiera archiwum. */
//bool Open(char* fname);
///*! Zamyka archiwum. */
//void Close();
///*! Zwraca nazwe danego pliku. */
//char* FileName(XE_HANDLE handle);
///*! Zwraca rozmiar danego pliku. */
//unsigned int FileSize(XE_HANDLE handle);
///*! Laduje plik z archiwum do pamieci. */
//bool LoadFile(XE_FILE* file,XE_HANDLE handle);
///*! Zapisuje pliki z pamieci do archiwum. */
//static bool SaveFiles(char* fname,VIRTUAL* files,unsigned int count);
///*! Szuka plik w archiwum. */
//bool Find(char* name,XE_HANDLE& result);
///*! Zwraca ilosc plikow w archiwum. */
//unsigned int Count();
//};
//#endif /* XE_COMPILE_CORE_VIRTUAL_FILE */
private:
XE_BUFFER Buffer;
unsigned int CurrentSize;
public:
//! Ilosc bajtow rozszerzajacych bufor przy jego powiekszaniu.
unsigned int ReallocBytes;
/*! Konstruktor domyslny. */
XE_FILE();
/*! Laduje dane z pliku.
\param fname Sciezka do pliku.
\param offset Przesuniecie kursora w pliku.
\param size Ilosc bajtow do zaladowania.
\return true jesli operacja powiodla sie.
*/
bool Load(char* fname,unsigned int offset=0,unsigned int size=0);
/*! Zapisuje dane do pliku.
\param fname Sciezka do pliku.
\param offset Przesuniecie kursora w pliku.
\param size Ilosc bajtow do zaladowania.
\return true jesli operacja powiodla sie.
*/
bool Save(char* fname,unsigned int offset=0,unsigned int size=0);
/*! Dopisuje dane na koncu pliku.
\param fname Sciezka do pliku.
\return true jesli operacja powiodla sie.
*/
bool Append(char* fname);
/*! Wpisuje dane do bufora pliku.
\param buff Wskaznik na dane do wczytania.
\param size Rozmiar danych do wczytania.
\return Rozmiar wczytanych danych.
*/
unsigned int Write(void* buff,unsigned int size=1);
/*! Wpisuje cstring do bufora pliku.
\param str Wskaznik na cstring do wczytania.
\param nullterm Ustala czy wpisywany string ma byc zakonczony wartoscia NULL.
\return Rozmiar wczytanych danych.
*/
unsigned int WriteString(char* str,bool nullterm=false);
/*! Odczytuje dane z bufora pliku.
\param buff Wskaznik na bufor docelowy.
\param size Rozmiar danych do odczytania.
\return Rozmiar odczytanych danych.
*/
unsigned int Read(void* buff,unsigned int size=1);
/*! Odczytuje string z bufora pliku.
\param str Referencja na string docelowy.
\param nullterm Ustala czy odczytywany string jest zakonczony wartoscia NULL.
\return Dlugosc odczytanego stringu.
*/
unsigned int ReadString(XE_STRING& str,bool nullterm=false);
/*! Ustawia kursor w buforze pliku.
\param offset Przesuniecie (w bajtach).
\param relative Ustala czy przesuniecie ma byc wzgledne.
\param reverse Ustala czy przesuniecie ma byc wsteczne.
*/
void Seek(unsigned int offset,bool relative=false,bool reverse=false);
/*! Sprawdza czy kursor w pliku osiagna koniec pliku.
\return true jesli kursor znajduje sie na koncu pliku.
*/
bool Eof();
/*! Zwraca rozmiar pliku lub rozmiar bufora pliku, jesli nie podano sciezki do pliku.
\param fname Sciezka do pliku.
\return Rozmiar (w bajtach).
*/
unsigned int Size(char* fname=0);
/*! Zwraca pozycje kursora w buforze pliku.
\return Pozycja (w bajtach).
*/
unsigned int Position();
/*! Zwraca wskaznik na dane bufora pliku.
\return Wskaznik na dane.
*/
void* Data();
/*! Zwalnia bufor pliku z pamieci. */
void Free();
/*! Zwraca wskaznik na bufor pliku.
\return Wskaznik na bufor pliku.
*/
XE_BUFFER* GetBuffer();
/*! Aktualizuje status bufora pliku. */
void Update();
};

//#ifdef XE_COMPILE_CORE_VIRTUAL_FILE
//
//XE_FILE::ARCHIVE::ARCHIVE()
//{
//}
//
//bool XE_FILE::ARCHIVE::Open(char* fname)
//{
//if(!XeVirtualFileInfoOpen(fname,&FilesInfo))return(false);
//Path.Set(fname);
//return(true);
//}
//
//void XE_FILE::ARCHIVE::Close()
//{
//Path.Clear();
//XeVirtualFileInfoClose(&FilesInfo);
//}
//
//char* XE_FILE::ARCHIVE::FileName(XE_HANDLE handle)
//{
//if(!Count()||handle>=Count())return(NULL);
//return(FilesInfo[handle].FileName);
//}
//
//unsigned int XE_FILE::ARCHIVE::FileSize(XE_HANDLE handle)
//{
//if(!Count()||handle>=Count())return(0);
//return(FilesInfo[handle].FileSize);
//}
//
//bool XE_FILE::ARCHIVE::LoadFile(XE_FILE* file,XE_HANDLE handle)
//{
//if(!file||!Path.Length()||!Count()||handle>=Count())return(false);
//XVF_FILE virt;
//if(XeVirtualFileLoad(&virt,Path.Get(),FilesInfo[handle]))
//{
//file->Buffer.Reserve(XeVirtualFileSize(&virt));
//file->CurrentSize=file->Buffer.Size();
//bool status=XeVirtualFileGetData(&virt,file->Buffer.Get(),file->Buffer.Size());
//XeVirtualFileFree(&virt);
//return(status);
//}
//return(false);
//}
//
//bool XE_FILE::ARCHIVE::SaveFiles(char* fname,VIRTUAL* files,unsigned int count)
//{
//if(!files||!count)return(false);
//XE_ARRAY<XVF_FILE> virt(count);
//for(unsigned int i=0;i<count;i++)
//XeVirtualFileMakeFile(&virt[i],files[i].Name.Get(),files[i].File->Buffer.Get(),files[i].File->CurrentSize);
//bool status=XeVirtualFileSave(fname,virt.Get(),count);
//virt.Free();
//return(true);
//}
//
//bool XE_FILE::ARCHIVE::Find(char* name,XE_HANDLE& result)
//{
//for(unsigned int i=0;i<FilesInfo.Size();i++)
//if(!strcmp(FilesInfo[i].FileName,name))
//{
//result=i;
//return(true);
//}
//return(false);
//}
//
//unsigned int XE_FILE::ARCHIVE::Count()
//{
//return(FilesInfo.Size());
//}
//
//XE_FILE::VIRTUAL::VIRTUAL()
//{
//File=0;
//}
//
//#endif /* XE_COMPILE_CORE_VIRTUAL_FILE */

XE_FILE::XE_FILE()
{
CurrentSize=0;
ReallocBytes=0;
}

bool XE_FILE::Load(char* fname,unsigned int offset,unsigned int size)
{
FILE* file=0;
fopen_s(&file,fname,"rb");
if(!file)return(false);
if(!size)
{
fseek(file,0,SEEK_END);
size=ftell(file);
rewind(file);
}
if(offset)fseek(file,offset,SEEK_SET);
Buffer.Reserve(size);
CurrentSize=fread(Buffer.Get(),1,size,file);
fclose(file);
return(true);
}

bool XE_FILE::Save(char* fname,unsigned int offset,unsigned int size)
{
FILE* file=0;
fopen_s(&file,fname,"wb");
if(!file)return(false);
bool status=false;
if(offset)fseek(file,offset,SEEK_SET);
if(size)
status=fwrite(Buffer.Get(),min(CurrentSize,size),1,file)?true:false;
else
status=fwrite(Buffer.Get(),CurrentSize,1,file)?true:false;
fclose(file);
return(status);
}

bool XE_FILE::Append(char* fname)
{
FILE* file=0;
fopen_s(&file,fname,"ab");
if(!file)return(false);
bool status=fwrite(Buffer.Get(),CurrentSize,1,file)?true:false;
fclose(file);
return(true);
}

unsigned int XE_FILE::Write(void* buff,unsigned int size)
{
unsigned int curr=Buffer.Current();
unsigned int need=curr+size;
if(ReallocBytes)
{
unsigned int reneed=Buffer.Size();
while(reneed<need)reneed+=ReallocBytes;
Buffer.Resize(reneed);
}
else
Buffer.Resize(need);
Buffer.Seek(curr);
unsigned int w=Buffer.Write((char*)buff,size);
CurrentSize=need;
return(w);
}

unsigned int XE_FILE::WriteString(char* str,bool nullterm)
{
if(!str)return(0);
if(!nullterm)
return(Write(str,strlen(str)));
else
return(Write(str,strlen(str)+1));
}

unsigned int XE_FILE::Read(void* buff,unsigned int size)
{
if(!buff||Buffer.Current()>=CurrentSize)return(0);
return(Buffer.Read((char*)buff,min(size,CurrentSize-Buffer.Current())));
}

unsigned int XE_FILE::ReadString(XE_STRING& str,bool nullterm)
{
if(!Buffer.Size())return(0);
if(!nullterm)
{
str.Copy(&Buffer.Get()[Buffer.Current()],CurrentSize-min(Buffer.Current(),CurrentSize));
Seek(str.Length(),true);
return(str.Length());
}
else
{
str.Copy(&Buffer.Get()[Buffer.Current()],min(strlen(&Buffer.Get()[Buffer.Current()]),CurrentSize-min(Buffer.Current(),CurrentSize)));
Seek(str.Length()+1,true);
return(str.Length()+1);
}
}

void XE_FILE::Seek(unsigned int offset,bool relative,bool reverse)
{
if(!relative&&!reverse)
Buffer.Seek(offset);
else
if(!relative&&reverse)
Buffer.Seek(CurrentSize-min(offset,CurrentSize));
else
if(relative&&!reverse)
Buffer.Seek(offset,XE_ARRAY<char>::CUR);
else
if(relative&&reverse)
Buffer.Seek(offset,XE_ARRAY<char>::RET);
if(Buffer.Current()>=CurrentSize)Buffer.Seek(CurrentSize);
}

bool XE_FILE::Eof()
{
if(Buffer.Current()>=CurrentSize)return(true);
return(false);
}

unsigned int XE_FILE::Size(char* fname)
{
if(!fname)return(CurrentSize);
FILE* file=0;
fopen_s(&file,fname,"rb");
if(!file)return(0);
fseek(file,0,SEEK_END);
unsigned int size=ftell(file);
fclose(file);
return(size);
}

unsigned int XE_FILE::Position()
{
return(Buffer.Current());
}

void* XE_FILE::Data()
{
return(Buffer.Get());
}

void XE_FILE::Free()
{
Buffer.Free();
CurrentSize=0;
}

XE_BUFFER* XE_FILE::GetBuffer()
{
return(&Buffer);
}

void XE_FILE::Update()
{
CurrentSize=Buffer.Size();
}

#ifdef XE_COMPILE_CORE_INTUICIO

/*--- XenoN Core Intuicio ---*/
typedef void(*XE_INTUICIO_INTERFACE)(XE_ARRAY<char>* Arguments);
typedef XE_ARRAY<char> XE_INTUICIO_PROGRAM;
#define XE_INTUICIO_EXTERNAL(name) void name (XE_ARRAY<char>* Arguments)
#define XE_INTUICIO_KEYWORD(name) XE_DWORD name=0; Arguments->ReadBytes(&name,4);
#define XE_INTUICIO_POINTER(type,name) type* name=0; Arguments->ReadBytes(&name,4);
#define XE_INTUICIO_CONSTANT(type,name) type name; Arguments->ReadBytes(&name,sizeof(type));
#define XE_INTUICIO_RESTART Arguments->Seek(0);

#endif /* XE_COMPILE_CORE_INTUICIO */

#ifdef XE_COMPILE_PSYCHE

/*--- XenoN Core Psyche ---*/
class XENON_CORE_NEURON;
class XENON_CORE_DETECTOR;
class XENON_CORE_EFFECTOR;
class XENON_CORE_SIGNAL;

class XENON_CORE_PSYCHE
{
#ifdef XE_COMPILE_EXPORT
public:
#else
private:
#endif /* XE_COMPILE_EXPORT */
XE_ELEMENTS<XENON_CORE_NEURON> Neurons;
XE_ELEMENTS<XENON_CORE_DETECTOR> Detectors;
XE_ELEMENTS<XENON_CORE_EFFECTOR> Effectors;
XE_ELEMENTS<XENON_CORE_SIGNAL> Signals;
#ifdef XE_COMPILE_EXPORT
XENON_CORE_PSYCHE();
void Progress(float factor);
bool Load(char* fname);
bool Save(char* fname,bool savesignals=false,bool savedetsignal=false,bool saveeffsignal=false);
#endif /* XE_COMPILE_EXPORT */
};

struct XE_PSYCHE_SIGNAL_NEURON_INFO
{
XE_ELM_UNIFORM SynapsisFirst;
XE_ELM_UNIFORM SynapsisLast;
XE_DWORD Gate;
XE_HALFVECTOR Position;
float Length;
};

struct XE_PSYCHE_DRAW_DESC
{
XE_VECTOR DetectorColor;
XE_VECTOR EffectorColor;
XE_VECTOR NeuronColor;
XE_VECTOR SignalColor;
XE_VECTOR SynapsisColor;
XE_ELM_TEXTURE DetectorTexture;
XE_ELM_TEXTURE EffectorTexture;
XE_ELM_TEXTURE NeuronTexture;
double DetectorSize;
double EffectorSize;
double NeuronSize;
bool DrawDetectors;
bool DrawEffectors;
bool DrawNeurons;
bool DrawSynapsis;
bool DrawSignals;
};

#endif /* XE_COMPILE_PSYCHE */

#ifdef XE_COMPILE_PHOTON

/*--- XenoN Core Rendertarget ---*/
class XENON_CORE_RENDERTARGET
{
#ifdef XE_COMPILE_EXPORT
public:
#else
private:
#endif /* XE_COMPILE_EXPORT */
struct Image
{
int width;
int height;
int pitch;
HDC hdc;
HBITMAP hBitmap;
BITMAPINFO info;
BYTE *pPixels;
};
XE_ESTATE rendermode;
HWND hwnd;
HDC hdc;
HGLRC hrc;
HANDLE hpbuffer;
HDC hpbufferdc;
HGLRC hpbufferrc;
XE_BUFFER hpbufferpix;
Image hpbufferimage;
int hpbufferdepth;
int x;
int y;
int width;
int height;
double angle;
double znear;
double zfar;
double aspect;
#ifdef XE_COMPILE_EXPORT
XENON_CORE_RENDERTARGET();
bool Create(HANDLE handle,XE_ESTATE mode,int bits,int depthbits);
void Destroy();
bool Activate();
bool Change(XE_ESTATE mode);
bool ImageCreate(Image *image);
void ImageDestroy(Image *image);
void ImagePreMultAlpha(Image *image);
#endif /* XE_COMPILE_EXPORT */
};

/*--- XenoN Core Texture ---*/
class XENON_CORE_TEXTURE
{
#ifdef XE_COMPILE_EXPORT
public:
#else
private:
#endif /* XE_COMPILE_EXPORT */
unsigned int texture;
unsigned int width;
unsigned int height;
void* pixels;
unsigned int psize;
unsigned int type;
#ifdef XE_COMPILE_EXPORT
XENON_CORE_TEXTURE();
void Create();
bool LoadEmpty(int w,int h,bool achan);
bool LoadFromXET(char* fname,bool achan);
bool LoadFromXETmem(void* pbuff,bool achan);
bool LoadFromRGBA(char* fname,int _width,int _height);
bool LoadFromRGBAmem(void* pbuff,int _width,int _height);
bool LoadFromRender(int px,int py,int width,int height,bool asnew);
bool LoadEmptyData(int w,int h);
bool LoadData(void* data,int w,int h);
bool Update(void* data,int px,int py,int width,int height);
bool Activate();
void Destroy();
void* Lock(bool xetfile=false);
void Unlock(bool update=true);
bool IsLocked();
#endif /* XE_COMPILE_EXPORT */
};

/*--- XenoN Core FrameBuffer ---*/
class XENON_CORE_FRAMEBUFFER
{
#ifdef XE_COMPILE_EXPORT
public:
#else
private:
#endif /* XE_COMPILE_EXPORT */
XE_ELM_TEXTURE texture;
unsigned int hbuffer;
unsigned int hdepth;
int Width;
int Height;
bool usefbo;
#ifdef XE_COMPILE_EXPORT
XENON_CORE_FRAMEBUFFER();
bool Create(int width,int height,XE_ESTATE type,bool forcednofbo=false);
bool CreateFromTexture(XE_ELM_TEXTURE tex,bool forcednofbo=false);
void Destroy();
void Activate();
void Unactivate();
void ActivateTexture();
bool SwapTexture(XE_ELM_TEXTURE tex);
#endif /* XE_COMPILE_EXPORT */
};

/*--- XenoN Core VertexBuffer ---*/
class XENON_CORE_VERTEXBUFFER
{
#ifdef XE_COMPILE_EXPORT
public:
#else
private:
#endif /* XE_COMPILE_EXPORT */
int MaxSize;
int MaxSizeIndices;
unsigned int Vbuffer;
unsigned int Ibuffer;
#ifdef XE_COMPILE_EXPORT
XENON_CORE_VERTEXBUFFER();
bool Create(int maxsize,void* data,XE_ESTATE type,int maxsizeindices,void* dataindices,XE_ESTATE typeindices);
void Destroy();
void Activate(XE_ESTATE mode);
static bool SendData(XE_ESTATE mode,int size,int offset,void* data);
static void* Map(XE_ESTATE mode,XE_ESTATE type);
static void Unmap(XE_ESTATE mode);
bool HasBuffer(XE_ESTATE mode);
int Size(XE_ESTATE mode);
#endif /* XE_COMPILE_EXPORT */
};

/*--- XenoN Core Shader ---*/
class XENON_CORE_SHADER
{
#ifdef XE_COMPILE_EXPORT
public:
#else
private:
#endif /* XE_COMPILE_EXPORT */
unsigned int shader;
XE_ESTATE type;
#ifdef XE_COMPILE_EXPORT
XENON_CORE_SHADER();
bool LoadFromFile(XE_ESTATE mode,char* filename,char* prepend);
bool LoadFromMemory(XE_ESTATE mode,char* buffer,char* prepend);
bool MakeProgram(unsigned int* shaders,unsigned int count);
bool Activate();
static bool Unactivate(XE_ESTATE type);
void Destroy();
#endif /* XE_COMPILE_EXPORT */
};

/*--- XenoN Core DisplayList ---*/
class XENON_CORE_DISPLAYLIST
{
#ifdef XE_COMPILE_EXPORT
public:
#else
private:
#endif /* XE_COMPILE_EXPORT */
unsigned int dlist;
unsigned int size;
#ifdef XE_COMPILE_EXPORT
XENON_CORE_DISPLAYLIST(unsigned int count=1);
void destroy();
void begin(unsigned int _id);
void end();
void call(unsigned int _id);
#endif /* XE_COMPILE_EXPORT */
};

#endif /* XE_COMPILE_PHOTON */

#ifdef XE_COMPILE_CHAOS

/*--- XenoN Core Substance ---*/
class XENON_CORE_SUBSTANCE
{
#ifdef XE_COMPILE_EXPORT
public:
#else
private:
#endif /* XE_COMPILE_EXPORT */
double object;
XE_VECTOR position;
double range;
double mass;
double energy;
unsigned int solid; // XE_FLAG_SUBSTANCE_SOLID_RANGE; XE_FLAG_SUBSTANCE_SOLID_SURFACE;
XE_VECTOR inertion;
XE_VECTOR prev;
XE_VECTOR temp;
double density;
#ifdef XE_COMPILE_EXPORT
XENON_CORE_SUBSTANCE();
#endif /* XE_COMPILE_EXPORT */
};

/*--- XenoN Core Ray ---*/
class XENON_CORE_RAY
{
#ifdef XE_COMPILE_EXPORT
public:
#else
private:
#endif /* XE_COMPILE_EXPORT */
XE_VECTOR position;
XE_ANGLEVECTOR direction;
double angle;
double energy;
double frequency;
#ifdef XE_COMPILE_EXPORT
XENON_CORE_RAY();
#endif /* XE_COMPILE_EXPORT */
};

/*--- XenoN Core Covalence Force ---*/
class XENON_CORE_COVALENCE_FORCE
{
#ifdef XE_COMPILE_EXPORT
public:
#else
private:
#endif /* XE_COMPILE_EXPORT */
double force;
double resilience;
double range;
XE_ESTATE solid;
XE_ELM_SUBSTANCE subst1;
XE_ELM_SUBSTANCE subst2;
#ifdef XE_COMPILE_EXPORT
XENON_CORE_COVALENCE_FORCE();
#endif /* XE_COMPILE_EXPORT */
};

/*--- XenoN Core Force ---*/
class XENON_CORE_FORCE
{
#ifdef XE_COMPILE_EXPORT
public:
#else
private:
#endif /* XE_COMPILE_EXPORT */
XE_VECTOR position;
XE_ANGLEVECTOR direction;
double force;
double speed;
XE_ELM_SUBSTANCE subst;
XE_ELM_SUBSTANCE substdir;
float ricochet;
double range;
#ifdef XE_COMPILE_EXPORT
XENON_CORE_FORCE();
#endif /* XE_COMPILE_EXPORT */
};

/*--- XenoN Core Surface ---*/
class XENON_CORE_SURFACE
{
#ifdef XE_COMPILE_EXPORT
public:
#else
private:
#endif /* XE_COMPILE_EXPORT */
XE_ELM_TEXTURE texture_id;
XE_ESTATE texture_type;
double order;
XE_VECTOR tex1;
XE_VECTOR tex2;
XE_VECTOR tex3;
double density;
int orientation;
XE_ELM_SUBSTANCE subst1;
XE_ELM_SUBSTANCE subst2;
XE_ELM_SUBSTANCE subst3;
#ifdef XE_COMPILE_EXPORT
XENON_CORE_SURFACE();
#endif /* XE_COMPILE_EXPORT */
};

/*--- XenoN Core Particle ---*/
class XENON_CORE_PARTICLE
{
#ifdef XE_COMPILE_EXPORT
public:
#else
private:
#endif /* XE_COMPILE_EXPORT */
double size;
XE_ELM_TEXTURE texture_id;
XE_ESTATE texture_type;
XE_ELM_SUBSTANCE subst;
#ifdef XE_COMPILE_EXPORT
XENON_CORE_PARTICLE();
#endif /* XE_COMPILE_EXPORT */
};

/*--- XenoN Core Global Force ---*/
class XENON_CORE_GLOBAL_FORCE
{
#ifdef XE_COMPILE_EXPORT
public:
#else
private:
#endif /* XE_COMPILE_EXPORT */
XE_ANGLEVECTOR direction;
double force;
#ifdef XE_COMPILE_EXPORT
XENON_CORE_GLOBAL_FORCE();
#endif /* XE_COMPILE_EXPORT */
};

/*--- XenoN Core Black Hole ---*/
class XENON_CORE_BLACK_HOLE
{
#ifdef XE_COMPILE_EXPORT
public:
#else
private:
#endif /* XE_COMPILE_EXPORT */
XE_VECTOR position;
double force;
double range;
XE_ELM_BLACK_HOLE bhole;
#ifdef XE_COMPILE_EXPORT
XENON_CORE_BLACK_HOLE();
#endif /* XE_COMPILE_EXPORT */
};

/*--- XenoN Core Wave Force ---*/
class XENON_CORE_WAVE_FORCE
{
#ifdef XE_COMPILE_EXPORT
public:
#else
private:
#endif /* XE_COMPILE_EXPORT */
XE_VECTOR position;
double force;
double range;
double length;
double speed;
XE_ESTATE orientation;
#ifdef XE_COMPILE_EXPORT
XENON_CORE_WAVE_FORCE();
#endif /* XE_COMPILE_EXPORT */
};

#endif /* XE_COMPILE_CHAOS */

#ifdef XE_COMPILE_CORE_DNA

/*--- XenoN Core DNA ---*/
class XENON_CORE_DNA
{
#ifdef XE_COMPILE_EXPORT
public:
#else
private:
#endif /* XE_COMPILE_EXPORT */
XE_ELM_PSYCHE Brain;
XE_ARRAY<XE_ELM_SUBSTANCE> Cells;
XE_ARRAY<XE_ELM_COVALENCE_FORCE> Muscles;
XE_ARRAY<XE_ELM_SURFACE> Skin;
XE_ELM_TEXTURE BodyTexture;
#ifdef XE_COMPILE_EXPORT
XENON_CORE_DNA();
void Die(bool destroybody=true);
void Birth(unsigned int cells,unsigned int muscles,unsigned int skin);
bool Link(XE_ESTATE type,unsigned int id,void* element);
bool Unlink(XE_ESTATE type,unsigned int id);
void* Get(XE_ESTATE type,unsigned int id);
void Draw();
unsigned int Count(XE_ESTATE type);
#endif /* XE_COMPILE_EXPORT */
};

#endif /* XE_COMPILE_CORE_DNA */

#ifdef XE_COMPILE_ETHER

/*--- XenoN Core Socket ---*/
class XENON_CORE_SOCKET
{
#ifdef XE_COMPILE_EXPORT
public:
#else
private:
#endif /* XE_COMPILE_EXPORT */
SOCKET Socket;
bool UDPmode;
unsigned int UDPmaxsize;
XE_BYTE Type;
sockaddr_in Addr;
bool NagleMode;
bool AsyncMode;
#ifdef XE_COMPILE_EXPORT
XENON_CORE_SOCKET();
bool Open(bool udpmode);
void Close();
void Shutdown(bool sending,bool receiving);
bool Connect(char* server,XE_WORD port);
bool Listen(XE_WORD port,int maxconn);
bool Accept(XENON_CORE_SOCKET* listen);
unsigned int Send(XE_FILE* buffer,bool sizeheader,unsigned int offset,unsigned int bsize);
unsigned int Receive(XE_FILE* buffer,unsigned int size,bool peek);
bool Async(bool mode);
bool Nagle(bool mode);
bool IsOpened();
bool IsConnected();
XE_DWORD GetIP();
XE_WORD GetPort();
static void IpToString(XE_DWORD ip,XE_STRING& result);
static XE_DWORD StringToIp(char* ip);
static XE_DWORD AddressToIp(char* address);
#endif /* XE_COMPILE_EXPORT */
};

#endif /* XE_COMPILE_ETHER */

/*--- XeEngineHandle() ---*/
/*! Zwraca uchwyt modulu silnika.
\return Uchwyt modulu silnika.
*/
HMODULE XE_CALLTYPE XeEngineHandle()
#ifdef XE_CANIMP
{
#ifndef XE_COMPILE_STATIC
return(GetModuleHandle("XenoN Core.dll"));
#else
return(GetModuleHandle(NULL));
#endif /* XE_COMPILE_STATIC */
}
#endif
;

#ifdef XE_COMPILE_CORE_VIRTUAL_FILE

/*--- XeVirtualFileMakeFile() ---*/
/*! Tworzy plik gotowy do zapisu do archiwum.
\param result Wskaznik na plik wynikowy.
\param name Nazwa pliku.
\param data Wskaznik na dane pliku.
\param size Rozmiar danych pliku (w bajtach).
*/
XE_IMPEX void XE_CALLTYPE XeVirtualFileMakeFile(XVF_FILE* result,char* name,void* data,unsigned int size);

/*--- XeVirtualFileSize() ---*/
/*! Zwraca wielkosc pliku.
\param file Wskaznik na plik.
\return Rozmiar danych pliku (w bajtach).
*/
XE_IMPEX unsigned int XE_CALLTYPE XeVirtualFileSize(XVF_FILE* file);

/*--- XeVirtualFileGetData() ---*/
/*! Pobiera dane pliku do bufora.
\param file Wskaznik na plik.
\param destbuff Wskaznik na bufor docelowy.
\param destsize Rozmiar bufora docelowego (w bajtach).
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeVirtualFileGetData(XVF_FILE* file,void* destbuff,unsigned int destsize);

/*--- XeVirtualFileFree() ---*/
/*! Zwalnia plik z pamieci (Wazne aby plik byl plikiem zaladowanym z archiwum!).
\param file Wskaznik na plik.
*/
XE_IMPEX void XE_CALLTYPE XeVirtualFileFree(XVF_FILE* file);

/*--- XeVirtualFileInfoOpen() ---*/
/*! Otwiera i zwraca informacje o plikach.
\param fname Sciezka do pliku archiwum.
\param infotab Wskaznik na docelowa tablice informacji o plikach.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeVirtualFileInfoOpen(char* fname,XE_ARRAY<XVF_INFO>* infotab);

/*--- XeVirtualFileInfoClose() ---*/
/*! Zamyka i usuwa informacje o plikach.
\param infotab Wskaznik na tablice informacji o plikach.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeVirtualFileInfoClose(XE_ARRAY<XVF_INFO>* infotab);

/*--- XeVirtualFileLoad() ---*/
/*! Laduje plik z archiwum.
\param resfile Wskaznik na plik.
\param fname Sciezka do pliku archiwum.
\param info Referencja na informacje o pliku.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeVirtualFileLoad(XVF_FILE* resfile,char* fname,XVF_INFO& info);

/*--- XeVirtualFileLoadPart() ---*/
/*! Laduje czesc pliku z archiwum.
\param resfile Wskaznik na plik.
\param fname Sciezka do pliku archiwum.
\param info Referencja na informacje o pliku.
\param offset Przesuniecie (w bajtach).
\param size Rozmiar danych (w bajtach).
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeVirtualFileLoadPart(XVF_FILE* resfile,char* fname,XVF_INFO& info,unsigned int offset, unsigned int size);

/*--- XeVirtualFileSave() ---*/
/*! Zapisuje pliki do archiwum.
\param fname Sciezka do pliku.
\param files Wskaznik na tablice plikow.
\param count Liczba plikow.
\param astemplate Ustala czy ma zapisac jako szablon archiwum.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeVirtualFileSave(char* fname,XVF_FILE* files,unsigned int count,bool astemplate=false);

/*--- XeVirtualFileMakeTemplate() ---*/
/*! Tworzy szablon archiwum.
\param fname Sciezka do pliku.
\param files Wskaznik na tablice informacji o plikach.
\param count Liczba plikow.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeVirtualFileMakeTemplate(char* fname,XVF_INFO* files,unsigned int count);

/*--- XeVirtualFileComplete() ---*/
/*! Uzupelnia plik w archiwum.
\param fname Sciezka do pliku.
\param files Wskaznik na tablice plikow.
\param count Liczba plikow.
\param id Indeks pliku.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeVirtualFileComplete(char* fname,XVF_FILE* files,unsigned int count,unsigned int id);

/*--- XeVirtualFileReplace() ---*/
/*! Podmienia plik w archiwum.
\param fname Sciezka do pliku.
\param file Plik.
\param id Indeks pliku.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeVirtualFileReplace(char* fname,XVF_FILE file,unsigned int id);

#endif /* XE_COMPILE_CORE_VIRTUAL_FILE */

#ifdef XE_COMPILE_CORE_INTUICIO

/*--- XeIntuicioCompile ---*/
/*! Kompiluje skrypt Intuicio.
\param program Wskaznik na program Intuicio.
\param buffer Bufor tresci skryptu.
\param size Rozmiar bufora w bajtach.
\param deflist Uchwyt listy definicji externala.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeIntuicioCompile(XE_INTUICIO_PROGRAM* program,char* buffer,unsigned int size,XE_HANDLE deflist=0);

/*--- XeIntuicioFree() ---*/
/*! Zwalnia program Intuicio z pamieci.
\param program Wskaznik na program Intuicio.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeIntuicioFree(XE_INTUICIO_PROGRAM* program);

/*--- XeIntuicioCompileToFile() ---*/
/*! Kompiluje skrypt Intuicio do pliku.
\param buffer Bufor tresci skryptu.
\param size Rozmiar bufora w bajtach.
\param fname Sciezka do pliku wynikowego.
\param deflist Uchwyt listy definicji externala.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeIntuicioCompileToFile(char* buffer,unsigned int size,char* fname,XE_HANDLE deflist=0);

/*--- XeIntuicioCompileFile() ---*/
/*! Kompiluje skrypt Intuicio z pliku.
\param program Wskaznik na program Intuicio.
\param script Sciezka do pliku z trescia skryptu.
\param deflist Uchwyt listy definicji externala.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeIntuicioCompileFile(XE_INTUICIO_PROGRAM* program,char* script,XE_HANDLE deflist=0);

/*--- XeIntuicioCompileFileToFile() ---*/
/*! Kompiluje skrypt Intuicio z pliku.
\param script Sciezka do pliku z trescia skryptu.
\param fname Sciezka do pliku wynikowego.
\param deflist Uchwyt listy definicji externala.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeIntuicioCompileFileToFile(char* script,char* fname,XE_HANDLE deflist=0);

/*--- XeIntuicioExecute() ---*/
/*! Wykonuje program Intuicio.
\param program Wskaznik na program Intuicio.
\param params Wskaznik na bufor parametrow.
\param parsize Rozmiar bufora parametrow w bajtach.
\param deflist Uchwyt listy definicji externala.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeIntuicioExecute(XE_INTUICIO_PROGRAM* program,void* params=0,unsigned int parsize=0,XE_HANDLE deflist=0);

/*--- XeIntuicioExecuteFile() ---*/
/*! Wykonuje program Intuicio.
\param fname Sciezka do pliku programu.
\param params Wskaznik na bufor parametrow.
\param parsize Rozmiar bufora parametrow w bajtach.
\param deflist Uchwyt listy definicji externala.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeIntuicioExecuteFile(char* fname,void* params=0,unsigned int parsize=0,XE_HANDLE deflist=0);

/*--- XeIntuicioLinkDefinition() ---*/
/*! Linkuje skrypt definicji externali.
\param name Nazwa biblioteki externali.
\param data Wskaznik na tresc skryptu.
\param size Rozmiar skryptu w bajtach.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeIntuicioLinkDefinition(char* name,void* data,unsigned int size);

/*--- XeIntuicioLinkDefinitionFile() ---*/
/*! Linkuje skrypt definicji externali z pliku.
\param name Nazwa biblioteki externali.
\param fname Sciezka do pliku definicji externali.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeIntuicioLinkDefinitionFile(char* name,char* fname);

/*--- XeIntuicioUnlinkDefinitions() ---*/
/*! Odlinkowuje skrypty definicji externali. */
XE_IMPEX void XE_CALLTYPE XeIntuicioUnlinkDefinitions();

/*--- XeIntuicioAddDefinitionProc() ---*/
/*! Dodaje procedure definicji externala.
\param ext Nazwa externala.
\param def Nazwa definicji.
\param proc Procedura externala.
\return Jesli operacja powiodla sie zwraca wskaznik na procedure externala, jesli nie to zwraca wartosc NULL.
*/
XE_IMPEX XE_INTUICIO_INTERFACE* XE_CALLTYPE XeIntuicioAddDefinitionProc(char* ext,char* def,XE_INTUICIO_INTERFACE proc);

/*--- XeIntuicioClearDefinitionsProc() ---*/
/*! Czysci liste procedur definicji externali. */
XE_IMPEX void XE_CALLTYPE XeIntuicioClearDefinitionsProc();

/*--- XeIntuicioBuildDefinitionList() ---*/
/*! Generuje liste definicji externala z zalinkowanych skryptow i zwraca jej uchwyt.
\return Uchwyt listy definicji externala.
*/
XE_IMPEX XE_HANDLE XE_CALLTYPE XeIntuicioBuildDefinitionList();

/*--- XeIntuicioFreeDefinitionList() ---*/
/*! Zwalnia liste definicji externala z pamieci.
\param deflist Uchwyt listy definicji externala.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeIntuicioFreeDefinitionList(XE_HANDLE deflist);

#endif /* XE_COMPILE_CORE_INTUICIO */

/*--- XeStoreStates() ---*/
/*! Wrzuca zmienne stanu na stos.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeStoreStates();

/*--- XeRestoreStates() ---*/
/*! Zdejmuje zmienne stanu ze stosu.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeRestoreStates();

/*--- XeGetState() ---*/
/*! Pobiera wartosc zmiennej stanu silnika.
\param index Zmienna stanu.
\return Wskaznik nieokreslony, ktory nalezy rzutowac na wlasciwy danej zmiennej typ. Przyklad: XE_ESTATE texturing=*(XE_ESTATE*)XeGetState(XE_RENDER_TEXTURING_2D);
*/
XE_IMPEX void* XE_CALLTYPE XeGetState(XE_ESTATE index);

/*--- XeSetState() ---*/
/*! Ustala wartosc zmiennej stanu silnika.
\param index Zmienna stanu.
\param ... Kolejne wartosci.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeSetState(XE_ESTATE index,...);

#ifdef XE_COMPILE_PHOTON

/*--- XeIsSupported() ---*/
/*! Sprawdza czy rozszerzenie jest obslugiwane.
\param extension Rozszerzenie.
\return true jesli rozszerzenie jest dostepne.
*/
XE_IMPEX bool XE_CALLTYPE XeIsSupported(XE_ESTATE extension);

#endif /* XE_COMPILE_PHOTON */

#ifdef XE_COMPILE_CORE_MATH

/*--- XeLengthdir() ---*/
/*! Konwertuje kierunek i dlugosc na wektor.
\param a Kierunek i dlugosc.
\param result Wskaznik na wektor wynikowy.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeLengthdir(XE_ANGLEVECTOR a,XE_VECTOR* result);

/*--- XeLengthdirLen() ---*/
/*! Konwertuje kierunek i dlugosc na wektor.
\param a Kierunek.
\param l Dlugosc.
\return Wektor wynikowy.
*/
XE_VECTOR XE_CALLTYPE XeLengthdirLen(XE_ANGLEVECTOR a,double l)
#ifdef XE_CANIMP
{
XE_ANGLEVECTOR va;
va=a;
va.Length=l;
XE_VECTOR vr;
XeLengthdir(va,&vr);
return(vr);
}
#endif
;

/*--- XeDirection() ---*/
/*! Oblicza kierunek i odleglosc pomiedzy dwoma punktami.
\param b Pozycja poczatkowa.
\param e Pozycja koncowa.
\param result Wskaznik na kierunek i dlugosc.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeDirection(XE_VECTOR b,XE_VECTOR e,XE_ANGLEVECTOR* result);

/*--- XeDirectionir() ---*/
/*! Oblicza kierunek i odleglosc pomiedzy dwoma punktami.
\param b Pozycja poczatkowa.
\param e Pozycja koncowa.
\return Kierunek i dlugosc.
*/
XE_ANGLEVECTOR XE_CALLTYPE XeDirectionDir(XE_VECTOR b,XE_VECTOR e)
#ifdef XE_CANIMP
{
XE_ANGLEVECTOR vr;
XeDirection(b,e,&vr);
return(vr);
}
#endif
;

#ifdef XE_COMPILE_PHOTON

/*--- XeCameraOrtho() ---*/
/*! Uaktywnia projekcje prostopadla.
\param px Pozycja X na scenie.
\param py Pozycja Y na scenie.
\param width Szerokosc na scenie.
\param height Wysokosc na scenie.
\param angle Kat obrotu widoku.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeCameraOrtho(double px,double py,double width,double height,double angle);

/*--- XeCameraOrthoCustomBuild() ---*/
/*! Buduje macierz projekcji prostopadlej uzytkownika.
\param matrix Wskaznik na docelowa macierz.
\param px Pozycja X na scenie.
\param py Pozycja Y na scenie.
\param width Szerokosc na scenie.
\param height Wysokosc na scenie.
\param angle Kat obrotu widoku.
\param znear Najblizsza granica widocznosci.
\param zfar Najdalsza granica widocznosci.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeCameraOrthoCustomBuild(XE_MATRIX* matrix,double px,double py,double width,double height,double angle,double znear=-1.0,double zfar=1.0);

/*--- XeCameraOrthoCustomBuildHalf() ---*/
/*! Buduje macierz float projekcji prostopadlej uzytkownika.
\param matrix Wskaznik na docelowa macierz float.
\param px Pozycja X na scenie.
\param py Pozycja Y na scenie.
\param width Szerokosc na scenie.
\param height Wysokosc na scenie.
\param angle Kat obrotu widoku.
\param znear Najblizsza granica widocznosci.
\param zfar Najdalsza granica widocznosci.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeCameraOrthoCustomBuildHalf(XE_HALFMATRIX* matrix,float px,float py,float width,float height,float angle,float znear=-1.0f,float zfar=1.0f);

/*--- XeCameraOrthoBuild() ---*/
/*! Buduje macierz projekcji prostopadlej kamery.
\param matrix Wskaznik na docelowa macierz.
\param px Pozycja X na scenie.
\param py Pozycja Y na scenie.
\param width Szerokosc na scenie.
\param height Wysokosc na scenie.
\param angle Kat obrotu widoku.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeCameraOrthoBuild(XE_MATRIX* matrix,double px,double py,double width,double height,double angle=0.0);

/*--- XeCameraOrthoBuildHalf() ---*/
/*! Buduje macierz float projekcji prostopadlej kamery.
\param matrix Wskaznik na docelowa macierz float.
\param px Pozycja X na scenie.
\param py Pozycja Y na scenie.
\param width Szerokosc na scenie.
\param height Wysokosc na scenie.
\param angle Kat obrotu widoku.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeCameraOrthoBuildHalf(XE_HALFMATRIX* matrix,float px,float py,float width,float height,float angle=0.0f);

/*--- XeCameraPerspective() ---*/
/*! Uaktywnia projekcje perspektywistyczna.
\param a Kierunek oka kamery.
\param p Pozycja kamery.
\param projection Typ projekcji: false jesli oko patrzy z pozycji kamery; true jesli oko patrzy na pozycje kamery;
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeCameraPerspective(XE_ANGLEVECTOR a,XE_VECTOR p,bool projection=false);

/*--- XeCameraPerspectiveCustomBuild() ---*/
/*! Buduje macierz projekcji perspektywistycznej uzytkownika.
\param matrix Wskaznik na docelowa macierz.
\param a Kierunek oka kamery.
\param p Pozycja kamery.
\param projection Typ projekcji: false jesli oko patrzy z pozycji kamery; true jesli oko patrzy na pozycje kamery;
\param angle Kat stozka pola widzenia.
\param aspect Aspekt obrazu.
\param znear Najblizsza granica widocznosci.
\param zfar Najdalsza granica widocznosci.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeCameraPerspectiveCustomBuild(XE_MATRIX* matrix,XE_ANGLEVECTOR a,XE_VECTOR p,bool projection=false,double angle=45.0,double aspect=1.0,double znear=1.0,double zfar=10000.0);

/*--- XeCameraPerspectiveCustomBuildHalf() ---*/
/*! Buduje macierz float projekcji perspektywistycznej uzytkownika.
\param matrix Wskaznik na docelowa macierz.
\param a Kierunek oka kamery.
\param p Pozycja kamery.
\param projection Typ projekcji: false jesli oko patrzy z pozycji kamery; true jesli oko patrzy na pozycje kamery;
\param angle Kat stozka pola widzenia.
\param aspect Aspekt obrazu.
\param znear Najblizsza granica widocznosci.
\param zfar Najdalsza granica widocznosci.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeCameraPerspectiveCustomBuildHalf(XE_HALFMATRIX* matrix,XE_ANGLEVECTOR a,XE_HALFVECTOR p,bool projection=false,float angle=45.0f,float aspect=1.0f,float znear=1.0f,float zfar=10000.0f);

/*--- XeCameraPerspectiveBuild() ---*/
/*! Buduje macierz projekcji perspektywistycznej uzytkownika.
\param matrix Wskaznik na docelowa macierz.
\param a Kierunek oka kamery.
\param p Pozycja kamery.
\param projection Typ projekcji: false jesli oko patrzy z pozycji kamery; true jesli oko patrzy na pozycje kamery;
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeCameraPerspectiveBuild(XE_MATRIX* matrix,XE_ANGLEVECTOR a,XE_VECTOR p,bool projection=false);

/*--- XeCameraPerspectiveBuildHalf() ---*/
/*! Buduje macierz float projekcji perspektywistycznej uzytkownika.
\param matrix Wskaznik na docelowa macierz.
\param a Kierunek oka kamery.
\param p Pozycja kamery.
\param projection Typ projekcji: false jesli oko patrzy z pozycji kamery; true jesli oko patrzy na pozycje kamery;
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeCameraPerspectiveBuildHalf(XE_HALFMATRIX* matrix,XE_ANGLEVECTOR a,XE_HALFVECTOR p,bool projection=false);

/*--- XeCameraUpdate() ---*/
/*! Aktualizuje ostatnio ustawiona projekcje perspektywistyczna.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeCameraUpdate();

/*--- XeCameraFrom() ---*/
/*! Zwraca pozycje z ktorej patrzy kamera.
\return Pozycja kamery.
*/
XE_IMPEX bool XE_CALLTYPE XeCameraFrom(XE_VECTOR* v);

/*--- XeCameraTo() ---*/
/*! Zwraca pozycje na ktora patrzy kamera.
\return Pozycja oka kamery.
*/
XE_IMPEX bool XE_CALLTYPE XeCameraTo(XE_VECTOR* v);

/*--- XeCameraUp() ---*/
/*! Zwraca wartosc znormalizowanego wektora gory kamery.
\return Wektor gory kamery.
*/
XE_IMPEX bool XE_CALLTYPE XeCameraUp(XE_VECTOR* v);

#endif /* XE_COMPILE_PHOTON */

#endif /* XE_COMPILE_CORE_MATH */

/*--- XeInitiation() ---*/
/*! Inicjacja silnika.
\return true jesli operacja poziodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeInitiation();

#ifdef XE_COMPILE_PHOTON

/*--- XeExtensions() ---*/
/*! Inicjalizacja rozszerzen.
\param forced Ustala czy rozszerzenia maja byc zainicjowane mimo iz juz wczesniej byly zainicjowane.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeExtensions(bool forced=false);

#endif /* XE_COMPILE_PHOTON */

#ifdef XE_COMPILE_PHOTON

namespace Photon
{

/*--- XeRenderExecute() ---*/
/*! Wysyla do karty rozkaz wykonania wywolanych operacji rederingu.
\param mode Typ rozkazu. Dostepne wartosci:<br>
XE_FLUSH: Nakazuje wykonanie operacji najszybciej na ile pozwala na to karta.<br>
XE_FINISH: Nakazuje wykonanie operacji czekajac dopuki nie zostana wszystkie wykonane.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeRenderExecute(XE_ESTATE mode);

/*--- XeRenderScene() ---*/
/*! Renderuje scene.
\param flag Flagi operacji. Dostepne wartosci:<br>
XE_FLAG_SWAP: Zamienia bufory ramki.<br>
XE_FLAG_COLORBUFF: Czysci bufor koloru.<br>
XE_FLAG_DEPTHBUFF: Czysci bufor glebi.<br>
XE_FLAG_ACCUMBUFF: Czysci bufor akumulacyjny.<br>
XE_FLAG_STENCILBUFF: Czysci bufor szablonowy.<br>
XE_FLAG_TRANSFORMIDENT: Resetuje transformacje (laduje macierz jednostkowa).
\return true jesli operacja powiodla sie
*/
XE_IMPEX bool XE_CALLTYPE XeRenderScene(unsigned int flag);

/*--- XeRenderGetData() ---*/
/*! Pobiera dane pixeli z bufora ramki.
\param x Pozycja X (zaczynajac od lewego dolnego rogu).
\param y Pozycja Y (zaczynajac od lewego dolnego rogu).
\param w Szerokosc.
\param h Wysokosc.
\param format Typ bufora. Dostepne wartosci:<br>
XE_RENDER_FORMAT_RGBA: Bufor koloru.<br>
XE_RENDER_FORMAT_STENCIL: Bufor szablonowy.<br>
XE_RENDER_FORMAT_DEPTH: Bufor glebi.
\param buffer Wskaznik na bufor do ktorego zapisze dane.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeRenderGetData(int x,int y,int w,int h,XE_ESTATE format,void* buffer);

/*--- XeRenderDrawData() ---*/
/*! Wpisuje dane pixeli do bufora ramki.
\param x Pozycja X (zaczynajac od lewego dolnego rogu).
\param y Pozycja Y (zaczynajac od lewego dolnego rogu).
\param w Szerokosc.
\param h Wysokosc.
\param format Typ bufora. Dostepne wartosci:<br>
XE_RENDER_FORMAT_RGBA: Bufor koloru.<br>
XE_RENDER_FORMAT_STENCIL: Bufor szablonowy.<br>
XE_RENDER_FORMAT_DEPTH: Bufor glebi.
\param buffer Wskaznik na bufor z ktorego wczyta dane.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeRenderDrawData(int x,int y,int w,int h,XE_ESTATE format,void* buffer);

/*--- XeRenderVsync() ---*/
/*! Wlacza badz wylacza pionowa synchronizacje obrazu.
\param mode XE_TRUE jesli ma wlaczyc, XE_FALSE jesli ma wylaczyc synchronizacje.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeRenderVsync(XE_ESTATE mode);

/*--- XePrimitiveBegin() ---*/
/*! Rozpoczyna proces rysowania prymitywow.
\param type Typ figur. Dostepne wartosci:<br>
XE_POINTS: Punkty.<br>
XE_LINES: Linie.<br>
XE_LINELOOP: Petla linii.<br>
XE_LINESTRIP: Tasma linii.<br>
XE_TRIANGLES: Trojkaty.<br>
XE_TRIANGLESTRIP: Tasma trojkatow.<br>
XE_TRIANGLEFAN: Wachlaz trojkatow.<br>
XE_QUADS: Czworokaty.<br>
XE_QUADSTRIP: Tasma czworokatow.<br>
XE_POLYGON: Wielokat.
*/
XE_IMPEX void XE_CALLTYPE XePrimitiveBegin(XE_ESTATE type);

/*--- XePrimitiveEnd() ---*/
/*! Konczy proces rysowania prymitywow. */
XE_IMPEX void XE_CALLTYPE XePrimitiveEnd();

/*--- XePutVertex() ---*/
/*! Dodaje wierzcholek do sceny.
\param v Pozycja.
*/
XE_IMPEX void XE_CALLTYPE XePutVertex(XE_VECTOR v);

/*--- XePutNormal() ---*/
/*! Ustala wektor normalny.
\param v Normalna.
*/
XE_IMPEX void XE_CALLTYPE XePutNormal(XE_VECTOR v);

/*--- XePutFogvertex() ---*/
/*! Ustala gestosc mgly.
\param d Gestosc (zakres: <0;1>).
*/
XE_IMPEX void XE_CALLTYPE XePutFogvertex(double d);

/*--- XePutTexvertex() ---*/
/*! Ustala koordynaty tekstury.
\param v Koordynaty tekstury.
*/
XE_IMPEX void XE_CALLTYPE XePutTexvertex(XE_VECTOR v);

/*--- XePutMultitexvertex() ---*/
/*! Ustala koordynaty multitekstury.
\param un Numer jednostki teksturujacej.
\param v Koordynaty multitekstury.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XePutMultitexvertex(unsigned int un,XE_VECTOR v);

/*--- XeRenderTargetCreate() ---*/
/*! Tworzy kontekst renderowania.
\param elm Wskaznik na element RenderTarget. Zwraca do niego utworzony element jesli operacja powiodla sie.
\param handle Uchwyt okna do ktorego podepniemy kontekst.
\param mode Tryb rysowania. Dostepne wartosci:<br>
XE_DRAW_TO_WINDOW: Rysowanie do okna.<br>
XE_DRAW_TO_LAYER: Rysowanie do warstwy (polprzezroczystego okna).
\param bits Ilosc bitow glebi koloru (8,16,24,32).
\param depthbits Ilosc bitow bufora glebi (8,16,24,32).
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeRenderTargetCreate(XE_ELM_RENDERTARGET* elm,HANDLE handle,XE_ESTATE mode,int bits,int depthbits);

/*--- XeRenderTargetDestroy() ---*/
/*! Niszczy kontekst renderowania.
\param index Element RenderTarget.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeRenderTargetDestroy(XE_ELM_RENDERTARGET index);

/*--- XeRenderTargetChange() ---*/
/*! Zmienia tryb kontekstu renderowania.
\param index Element RenderTarget.
\param mode Tryb rysowania. Dostepne wartosci:<br>
XE_DRAW_TO_WINDOW: Rysowanie do okna.<br>
XE_DRAW_TO_LAYER: Rysowanie do warstwy (polprzezroczystego okna).
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeRenderTargetChange(XE_ELM_RENDERTARGET index,XE_ESTATE mode);

/*--- XeRenderTargetActivate) ---*/
/*! Aktywuje kontekst renderowania.
\param index Element RenderTarget.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeRenderTargetActivate(XE_ELM_RENDERTARGET index);

/*--- XeRenderTargetRelease() ---*/
/*! Zwalnia aktywny kontekst renderowania. */
XE_IMPEX void XE_CALLTYPE XeRenderTargetRelease();

/*--- XeRenderTargetShare() ---*/
/*! Wspoldzieli zasoby graficzne z danym kontekstem.
\param index Element RenderTarget.
\param target Element RenderTarget z ktorym ma wspoldzielic zasoby.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeRenderTargetShare(XE_ELM_RENDERTARGET index,XE_ELM_RENDERTARGET target);

/*--- XeRenderTargetArea() ---*/
/*! Ustawia wlasciwosci pola viewportu.
\param index Element RenderTarget.
\param x Pozycja X w oknie.
\param y Pozycja Y w oknie.
\param width Szerokosc w oknie.
\param height Wysokosc w oknie.
\param angle Kat pola widzenia w perspektywie.
\param znear Najblizsza wartosc Z-bufora.
\param zfar Najdalsza wartosc Z-bufora.
\param aspect Aspekt obrazu.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeRenderTargetArea(XE_ELM_RENDERTARGET index,int x,int y,int width,int height,double angle=45,double znear=-1,double zfar=1,double aspect=0);

/*--- XeRenderTargetGet() ---*/
/*! Pobiera wartosc zmiennej stanu kontekstu.
\param index Element RenderTarget.
\param type Zmienna stanu. Dostepne wartosci:<br>
XE_RENDERTARGET_WINDOW_HANDLE: Uchwyt okna.<br>
XE_RENDERTARGET_DISPLAY_CONTEXT: Kontekst wyswietlania.<br>
XE_RENDERTARGET_RENDER_CONTEXT: Kontenkst renderingu.<br>
XE_RENDERTARGET_X: Pozycja X.<br>
XE_RENDERTARGET_Y: Pozycja Y.<br>
XE_RENDERTARGET_WIDTH: Szerokosc.<br>
XE_RENDERTARGET_HEIGHT: Wysokosc.<br>
XE_RENDERTARGET_ANGLE: Kat pola widzenia w perspektywie.<br>
XE_RENDERTARGET_ZNEAR: Najblizsza wartosc Z-bufora.<br>
XE_RENDERTARGET_ZFAR: Najdalsza wartosc Z-bufora.<br>
XE_RENDERTARGET_ASPECT: Aspekt obrazu.
XE_RENDERTARGET_DRAW_MODE: Tryb rysowania na ekranie. Dostepne wartosci:<br>
XE_DRAW_TO_WINDOW: Rysowanie do okna.<br>
XE_DRAW_TO_LAYER: Rysowanie do warstwy (polprzezroczystego okna).
\return Wskaznik nieokreslony na zmienna stanu kontekstu. Nalezy rzutowac na konkretny typ danej zmiennej stanu.
*/
XE_IMPEX void* XE_CALLTYPE XeRenderTargetGet(XE_ELM_RENDERTARGET index,XE_ESTATE type);

/*--- XeFrameBufferCreate() ---*/
/*! Tworzy bufor ramki.
\param elm Wskaznik na element bufora ramki.
\param width Szerokosc.
\param height Wysokosc.
\param type Typ tekstury bufora ramki. Dostepne wartosci:<br>
XE_2D: Tekstura 2D.<br>
XE_DATA: Tekstura float.
\param forcednofbo Ustala czy bufor ramki ma bazowac wylacznie na teksturze, nawet gdy obsluguje FBO.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeFrameBufferCreate(XE_ELM_FRAMEBUFFER* elm,int width,int height,XE_ESTATE type=XE_2D,bool forcednofbo=false);

/*--- XeFrameBufferCreateFromTexture() ---*/
/*! Tworzy bufor ramki.
\param elm Wskaznik na element bufora ramki.
\param tex Element tekstury.
\param forcednofbo Ustala czy bufor ramki ma bazowac wylacznie na teksturze, nawet gdy obsluguje FBO.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeFrameBufferCreateFromTexture(XE_ELM_FRAMEBUFFER* elm,XE_ELM_TEXTURE tex,bool forcednofbo=false);

/*--- XeFrameBufferDestroy() ---*/
/*! Niszczy bufor ramki.
\param index Element bufora ramki.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeFrameBufferDestroy(XE_ELM_FRAMEBUFFER index);

/*--- XeFrameBufferActivate() ---*/
/*! Aktywuje bufor ramki.
\param index Element bufora ramki.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeFrameBufferActivate(XE_ELM_FRAMEBUFFER index);

/*--- XeFrameBufferActivateTexture() ---*/
/*! Aktywuje bufor ramki jako teksture.
\param index Element bufora ramki.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeFrameBufferActivateTexture(XE_ELM_FRAMEBUFFER index);

/*--- XeFrameBufferUnactivate() ---*/
/*! Dezaktywuje bufor ramki.
\param index Element bufora ramki.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeFrameBufferUnactivate(XE_ELM_FRAMEBUFFER index);

/*--- XeFrameBufferMipmap() ---*/
/*! Generuje mipmapy dla tekstury bufora ramki.
\param index Element bufora ramki.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeFrameBufferMipmap(XE_ELM_FRAMEBUFFER index);

/*--- XeFrameBufferSwapTexture() ---*/
/*! Podmienia zawartosc tekstury, tylko wtedy gdy maja te same wymiary.
\param index Element bufora ramki.
\param tex Element tekstury.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeFrameBufferSwapTexture(XE_ELM_FRAMEBUFFER index,XE_ELM_TEXTURE tex);

/*--- XeFrameBufferGet() ---*/
/*! Pobiera wartosc zmiennej stanu bufora ramki.
\param index Element bufora ramki.
\param type Zmienna stanu. Dostepne wartosci:<br>
XE_PIXELBUFFER_WIDTH: Szerokosc.<br>
XE_PIXELBUFFER_HEIGHT: Wysokosc.<br>
\return Wskaznik nieokreslony na zmienna stanu bufora ramki. Nalezy rzutowac na konkretny typ danej zmiennej stanu.
*/
XE_IMPEX void* XE_CALLTYPE XeFrameBufferGet(XE_ELM_FRAMEBUFFER index,XE_ESTATE type);

/*--- XeFrameBufferUseFBO() ---*/
/*! Sprawdza czy bufor ramki uzywa rozszerzenia FBO.
\param index Element bufora ramki.
\return true jesli uzywa rozszerzenia FBO.
*/
XE_IMPEX bool XE_CALLTYPE XeFrameBufferUseFBO(XE_ELM_FRAMEBUFFER index);

/*--- XeVertexBufferInit() ---*/
/*! Inicjuje obsluge bufora wierzcholkow.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeVertexBufferInit();

/*--- XeVertexBufferCreate() ---*/
/*! Tworzy bufor wierzcholkow.
\param elm Wskaznik na element bufora wierzcholkow.
\param maxsize Maksymalny rozmiar bufora wierzcholkow w bajtach.
\param data Wskaznik na dane bufora wierzcholkow.
\param type Typ obslugi bufora wierzcholkow. Dostepne wartosci:<br>
XE_VERTEXBUFFER_DYNAMIC_COPY: Dane beda aktualizowane. Przeplyw danych GPU->GPU.<br>
XE_VERTEXBUFFER_DYNAMIC_DRAW: Dane beda aktualizowane. Przeplyw danych APP->GPU.<br>
XE_VERTEXBUFFER_DYNAMIC_READ: Dane beda aktualizowane. Przeplyw danych GPU->APP.<br>
XE_VERTEXBUFFER_STATIC_COPY: Dane nie beda aktualizowane. Przeplyw danych GPU->GPU.<br>
XE_VERTEXBUFFER_STATIC_DRAW: Dane nie beda aktualizowane. Przeplyw danych APP->GPU.<br>
XE_VERTEXBUFFER_STATIC_READ: Dane nie beda aktualizowane. Przeplyw danych GPU->APP.<br>
XE_VERTEXBUFFER_STREAM_COPY: Dane beda aktualizowane co klatke. Przeplyw danych GPU->GPU.<br>
XE_VERTEXBUFFER_STREAM_DRAW: Dane beda aktualizowane co klatke. Przeplyw danych APP->GPU.<br>
XE_VERTEXBUFFER_STREAM_READ: Dane beda aktualizowane co klatke. Przeplyw danych GPU->APP.
\param maxsizeindices Maksymalny rozmiar bufora indeksow w bajtach.
\param dataindices Wskaznik na dane bufora indeksow.
\param typeindices Typ obslugi bufora indeksow. Dostepne wartosci:<br>
XE_VERTEXBUFFER_DYNAMIC_COPY: Dane beda aktualizowane. Przeplyw danych GPU->GPU.<br>
XE_VERTEXBUFFER_DYNAMIC_DRAW: Dane beda aktualizowane. Przeplyw danych APP->GPU.<br>
XE_VERTEXBUFFER_DYNAMIC_READ: Dane beda aktualizowane. Przeplyw danych GPU->APP.<br>
XE_VERTEXBUFFER_STATIC_COPY: Dane nie beda aktualizowane. Przeplyw danych GPU->GPU.<br>
XE_VERTEXBUFFER_STATIC_DRAW: Dane nie beda aktualizowane. Przeplyw danych APP->GPU.<br>
XE_VERTEXBUFFER_STATIC_READ: Dane nie beda aktualizowane. Przeplyw danych GPU->APP.<br>
XE_VERTEXBUFFER_STREAM_COPY: Dane beda aktualizowane co klatke. Przeplyw danych GPU->GPU.<br>
XE_VERTEXBUFFER_STREAM_DRAW: Dane beda aktualizowane co klatke. Przeplyw danych APP->GPU.<br>
XE_VERTEXBUFFER_STREAM_READ: Dane beda aktualizowane co klatke. Przeplyw danych GPU->APP.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeVertexBufferCreate(XE_ELM_VERTEXBUFFER* elm,int maxsize,void* data,XE_ESTATE type,int maxsizeindices,void* dataindices,XE_ESTATE typeindices);

/*--- XeVertexBufferDestroy() ---*/
/*! Niszczy bufor wierzcholkow.
\param index Element bufora wierzcholkow.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeVertexBufferDestroy(XE_ELM_VERTEXBUFFER index);

/*--- XeVertexBufferActivate() ---*/
/*! Aktywuje bufor wierzcholkow.
\param index Element bufora wierzcholkow.
\param mode Tryb bufora wierzcholkow. Dostepne wartosci:<br>
XE_VERTEXBUFFER_ARRAY: Bufor wierzcholkow.<br>
XE_VERTEXBUFFER_ELEMENT: Bufor indeksow.
*/
XE_IMPEX void XE_CALLTYPE XeVertexBufferActivate(XE_ELM_VERTEXBUFFER index,XE_ESTATE mode);

/*--- XeVertexBufferUnactivate() ---*/
/*! Dezaktywuje bufor wierzcholkow.
\param mode Tryb bufora wierzcholkow. Dostepne wartosci:<br>
XE_VERTEXBUFFER_ARRAY: Bufor wierzcholkow.<br>
XE_VERTEXBUFFER_ELEMENT: Bufor indeksow.
*/
XE_IMPEX void XE_CALLTYPE XeVertexBufferUnactivate(XE_ESTATE mode);

/*--- XeVertexBufferSendData() ---*/
/*! Wysyla i podmienia dane w buforze wierzcholkow.
\param mode Tryb bufora wierzcholkow. Dostepne wartosci:<br>
XE_VERTEXBUFFER_ARRAY: Bufor wierzcholkow.<br>
XE_VERTEXBUFFER_ELEMENT: Bufor indeksow.
\param size Rozmiar danych w bajtach.
\param offset Przesuniecie danych w bajtach.
\param data Wskaznik na dane do przeslania.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeVertexBufferSendData(XE_ESTATE mode,int size,int offset,void* data);

/*--- XeVertexBufferMap() ---*/
/*! Mapuje bufor wierzcholkow (pobiera wskaznik na niego i blokuje mozliwosc renderowania go).
\param mode Tryb bufora wierzcholkow. Dostepne wartosci:<br>
XE_VERTEXBUFFER_ARRAY: Bufor wierzcholkow.<br>
XE_VERTEXBUFFER_ELEMENT: Bufor indeksow.
\param type Typ dostepu do danych. Dostepne wartosci:<br>
XE_VERTEXBUFFER_READ: Tylko do odczytu.<br>
XE_VERTEXBUFFER_WRITE: Tylko do zapisu.<br>
XE_VERTEXBUFFER_READWRITE: Do zapisu i odczytu.
\return Wskaznik na bufor jesli operacja powiodla sie, lub wartosc NULL jesli nie.
*/
XE_IMPEX void* XE_CALLTYPE XeVertexBufferMap(XE_ESTATE mode,XE_ESTATE type);

/*--- XeVertexBufferUnmap() ---*/
/*! Odmapowuje bufor wierzcholkow (odblokowuje mozliwosc renderowania go).
\param mode Tryb bufora wierzcholkow. Dostepne wartosci:<br>
XE_VERTEXBUFFER_ARRAY: Bufor wierzcholkow.<br>
XE_VERTEXBUFFER_ELEMENT: Bufor indeksow.
*/
XE_IMPEX void XE_CALLTYPE XeVertexBufferUnmap(XE_ESTATE mode);

/*--- XeVertexBufferHasBuffer() ---*/
/*! Sprawdza czy bufor posiada konkretny typ bufora wierzcholkow.
\param index Element bufora wierzcholkow.
\param mode Tryb bufora wierzcholkow. Dostepne wartosci:<br>
XE_VERTEXBUFFER_ARRAY: Bufor wierzcholkow.<br>
XE_VERTEXBUFFER_ELEMENT: Bufor indeksow.
\return true jesli posiada dany typ bufora wierzcholkow.
*/
XE_IMPEX bool XE_CALLTYPE XeVertexBufferHasBuffer(XE_ELM_VERTEXBUFFER index,XE_ESTATE mode);

/*--- XeVertexBufferSize() ---*/
/*! Zwraca rozmiar danego bufora wierzcholkow w bajtach.
\param index Element bufora wierzcholkow.
\param mode Tryb bufora wierzcholkow. Dostepne wartosci:<br>
XE_VERTEXBUFFER_ARRAY: Bufor wierzcholkow.<br>
XE_VERTEXBUFFER_ELEMENT: Bufor indeksow.
\return Rozmiar bufora.
*/
XE_IMPEX int XE_CALLTYPE XeVertexBufferSize(XE_ELM_VERTEXBUFFER index,XE_ESTATE mode);

/*--- XeShaderInit() ---*/
/*! Inicjuje obsluge shaderow.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeShaderInit();

/*--- XeShaderModel() ---*/
/*! Zwraca Shader Model obslugiwany przez silnik i karte graficzna.
\return obsługiwany Shader Model.
*/
XE_IMPEX int XE_CALLTYPE XeShaderModel();

/*--- XeShaderCreate() ---*/
/*! Tworzy shader.
\param elm Wskaznik na element shadera.
*/
XE_IMPEX void XE_CALLTYPE XeShaderCreate(XE_ELM_SHADER* elm);

/*--- XeShaderDestroy() ---*/
/*! Niszczy shader.
\param index Element shadera.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeShaderDestroy(XE_ELM_SHADER index);

/*--- XeShaderProfile() ---*/
/*! Zwraca profil shadera.
\param index Element shadera.
\return Profil shadera.
*/
XE_IMPEX XE_ESTATE XE_CALLTYPE XeShaderProfile(XE_ELM_SHADER index);

/*--- XeShaderLoad() ---*/
/*! Laduje shader z pliku.
\param index Element shadera.
\param mode Profil shadera.
\param filename Nazwa pliku.
\param prepend Dodatkowe dane na poczatku bufora (wartosc NULL jesli brak danych).
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeShaderLoad(XE_ELM_SHADER index,XE_ESTATE mode,char* filename,char* prepend);

/*--- XeShaderLoadFromMemory() ---*/
/*! Laduje shader z pamieci.
\param index Element shadera.
\param mode Profil shadera.
\param buff Bufor z zawartoscia shadera.
\param prepend Dodatkowe dane na poczatku bufora (wartosc NULL jesli brak danych).
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeShaderLoadFromMemory(XE_ELM_SHADER index,XE_ESTATE mode,char* buff,char* prepend);

/*--- XeShaderMakeProgram() ---*/
/*! Tworzy program cieniowania laczac poszczegolne shadery.
\param index Element shadera.
\param shaders Tablica shaderow do polaczenia.
\param count Liczba shaderow do polaczenia.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeShaderMakeProgram(XE_ELM_SHADER index,XE_ELM_SHADER* shaders,unsigned int count);

/*--- XeShaderLastLog() ---*/
/*! Zwraca wskaznik na cstring z trescia ostatniego logu akcji shadera.
\return Wskaznik na cstring logu.
*/
XE_IMPEX char* XE_CALLTYPE XeShaderLastLog();

/*--- XeShaderActivate() ---*/
/*! Aktywuje shader.
\param index Element shadera.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeShaderActivate(XE_ELM_SHADER index);

/*--- XeShaderUnactivate() ---*/
/*! Deaktywuje shader danego typu.
\param type Profil shadera.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeShaderUnactivate(XE_ESTATE type);

/*--- XeShaderLocalParam() ---*/
/*! Ustala wartosc zmiennej lokalnej.
\param mode Profil shadera.
\param index Indeks zmiennej lokalnej.
\param v Wektor wartosci.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeShaderLocalParam(XE_ESTATE mode,int index,XE_HALFVECTOR v);

/*--- XeShaderEnvParam() ---*/
/*! Ustala wartosc zmiennej srodowiskowej.
\param mode Profil shadera.
\param index Indeks zmiennej srodowiskowej.
\param v Wektor wartosci.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeShaderEnvParam(XE_ESTATE mode,int index,XE_HALFVECTOR v);

/*--- XeShaderUniformLocation() ---*/
/*! Zwraca indeks zmiennej (profil: XE_SHADER_PROGRAM_SL).
\param index Element shadera.
\param name Nazwa zmiennej.
\return Indeks zmiennej.
*/
XE_IMPEX int XE_CALLTYPE XeShaderUniformLocation(XE_ELM_SHADER index,char* name);

/*--- XeShaderUniform() ---*/
/*! Ustala wartosc zmiennej (profil: XE_SHADER_PROGRAM_SL).
\param index Indeks zmiennej.
\param v Wektor wartosci.
\param subs Ilosc komponentow wysylanych do zmiennej.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeShaderUniform(int index,XE_HALFVECTOR v,int subs=4);

/*--- XeShaderUniformInteger() ---*/
/*! Ustala wartosc zmiennej calkowitej (profil: XE_SHADER_PROGRAM_SL).
\param index Indeks zmiennej.
\param s Tablica 4 elementow wartosci.
\param subs Ilosc komponentow wysylanych do zmiennej.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeShaderUniformInteger(int index,int s[4],int subs=4);

/*--- XeShaderUniformMatrix() ---*/
/*! Ustala wartosc zmiennej - macierz (profil: XE_SHADER_PROGRAM_SL).
\param index Indeks zmiennej.
\param m Tablica 4 wektorow wartosci.
\param transpose Ustala czy macierz ma byc transponowana.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeShaderUniformMatrix(int index,XE_HALFVECTOR m[4],bool transpose=false);

/*--- XeShaderAttributeLocation() ---*/
/*! Zwraca indeks atrybutu (profil: XE_SHADER_PROGRAM_SL).
\param index Element shadera.
\param name Nazwa atrybutu.
\return Indeks atrybutu.
*/
XE_IMPEX int XE_CALLTYPE XeShaderAttributeLocation(XE_ELM_SHADER index,char* name);

/*--- XeShaderAttribute() ---*/
/*! Ustala wartosc atrybutu (profil: XE_SHADER_PROGRAM_SL).
\param index Indeks atrybutu.
\param v Wektor wartosci.
\param subs Ilosc komponentow wysylanych do atrybutu.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeShaderAttribute(int index,XE_HALFVECTOR v,int subs=4);

/*--- XeTextureCreate() ---*/
/*! Tworzy teksture i automatycznie ja aktywuje.
\param elm Wskaznik na element tekstury. Zwraca do niego utworzony element jesli operacja powiodla sie.
*/
XE_IMPEX void XE_CALLTYPE XeTextureCreate(XE_ELM_TEXTURE* elm);

/*--- XeTextureDestroy() ---*/
/*! Niszczy teksture.
\param index Element tekstury.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeTextureDestroy(XE_ELM_TEXTURE index);

/*--- XeTextureLoad() ---*/
/*! Laduje teksture z pliku.
\param index Element tekstury.
\param filename Cstring z nazwa pliku.
\param achannel Jesli true to laduje z kanalem przezroczystosci.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeTextureLoad(XE_ELM_TEXTURE index,char* filename,bool achannel);

/*--- XeTextureLoadEmpty() ---*/
/*! Laduje pusta teksture 2D.
\param index Element tekstury.
\param width Szerokosc tekstury.
\param height Wysokosc tekstury.
\param achannel Jesli true to laduje z kanalem przezroczystosci.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeTextureLoadEmpty(XE_ELM_TEXTURE index,int width,int height,bool achannel);

/*--- XeTextureLoadFromMemory() ---*/
/*! Laduje teksture 2D z pamieci.
\param index Element tekstury.
\param buff Wskaznik na bufor w ktorym sa dane tekstury.
\param achannel Jesli true to laduje z kanalem przezroczystosci.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeTextureLoadFromMemory(XE_ELM_TEXTURE index,void* buff,bool achannel);

/*--- XeTextureLoadRGBA() ---*/
/*! Laduje teksture 2D RAW RGBA z pliku.
\param index Element tekstury.
\param filename Cstring z nazwa pliku.
\param width Szerokosc tekstury RAW RGBA.
\param height Wysokosc tekstury RAW RGBA.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeTextureLoadRGBA(XE_ELM_TEXTURE index,char* filename,int width,int height);

/*--- XeTextureLoadRGBAFromMemory() ---*/
/*! Laduje teksture 2D RAW RGBA z pamieci.
\param index Element tekstury.
\param buff Wskaznik na bufor w ktorym sa dane tekstury RAW RGBA.
\param width Szerokosc tekstury RAW RGBA.
\param height Wysokosc tekstury RAW RGBA.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeTextureLoadRGBAFromMemory(XE_ELM_TEXTURE index,void* buff,int width,int height);

/*--- XeTextureFromRender() ---*/
/*! Laduje teksture 2D z bufora kolorow aktywnego kontekstu renderujacego.
\param index  Element tekstury.
\param x Pozycja X w buforze.
\param y Pozycja Y w buforze.
\param width Szerokosc w buforze.
\param height Wysokosc w buforze.
\param asnew Tworzy nowa teksture jesli wartosc jest true, jesli zas false to zastepuje istniejace dane.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeTextureFromRender(XE_ELM_TEXTURE index,int x,int y,int width,int height,bool asnew);

/*--- XeTextureLoadEmptyData() ---*/
/*! Laduje pusta teksture float.
\param index Element tekstury.
\param width Szerokosc tekstury.
\param height Wysokosc tekstury.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeTextureLoadEmptyData(XE_ELM_TEXTURE index,int width,int height);

/*--- XeTextureLoadData() ---*/
/*! Laduje teksture float RAW RGBA z pamieci.
\param index Element tekstury.
\param buff Wskaznik na bufor w ktorym sa dane tekstury RAW RGBA.
\param width Szerokosc tekstury RAW RGBA.
\param height Wysokosc tekstury RAW RGBA.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeTextureLoadData(XE_ELM_TEXTURE index,void* buff,int width,int height);

/*--- XeTextureUpdate() ---*/
/*! Aktualizuje teksture pobierajac piksele z danego bufora.
\param index Element tekstury.
\param data Wskaznik na bufor danych pikseli.
\param x Pozycja X w buforze.
\param y Pozycja Y w buforze.
\param width Szerokosc w buforze.
\param height Wysokosc w buforze.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeTextureUpdate(XE_ELM_TEXTURE index,void* data,int x,int y,int width,int height);

/*--- XeTextureActivate() ---*/
/*! Aktywuje teksture.
\param index Element tekstury.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeTextureActivate(XE_ELM_TEXTURE index);

/*--- XeTextureUnactivate() ---*/
/*! Dezaktywuje teksture.
\param type Typ tekstury. Dostepne wartosci:<br>
XE_2D: Tekstura 2D.<br>
XE_DATA: Tekstura float.
*/
XE_IMPEX void XE_CALLTYPE XeTextureUnactivate(XE_ESTATE type=XE_2D);

/*--- XeTextureLock() ---*/
/*! Blokuje piksele tekstury i umieszcza je w podrecznym buforze.
\param index Element tekstury.
\param lockasxet Ustala czy blokowane piksele maja byc kompilowane do formatu XET.
\return Wskaznik na bufor pikseli lub wartosc NULL jesli operacja nie powiodla sie.
*/
XE_IMPEX void* XE_CALLTYPE XeTextureLock(XE_ELM_TEXTURE index,bool lockasxet=false);

/*--- XeTextureUnlock() ---*/
/*! Odblokowuje piksele tekstury.
\param index Element tekstury.
\param update Ustala czy ma zaktualizowac piksele tekstury z bufora.
*/
XE_IMPEX void XE_CALLTYPE XeTextureUnlock(XE_ELM_TEXTURE index,bool update);

/*--- XeTextureIsLocked() ---*/
/*! Sprawdza czy piksele tekstury sa zablokowane.
\param index Element tekstury.
\return true jesli piksele sa zablokowane.
*/
XE_IMPEX bool XE_CALLTYPE XeTextureIsLocked(XE_ELM_TEXTURE index);

/*--- XeTextureParamater() ---*/
/*! Ustala parametr aktywnej tekstury.
\param type Typ parametru. Dostepne wartosci:<br>
XE_TEXTURE_PARAM_PRIORITY: Priorytet tekstury (zakres: <0;1>).<br>
XE_TEXTURE_PARAM_MIN_FILTER: Filtr pomniejszajacy.<br>
XE_TEXTURE_PARAM_MAG_FILTER: Filtr powiekszajacy.<br>
XE_TEXTURE_PARAM_WRAP_S: Poziome powtarzanie tekstury.<br>
XE_TEXTURE_PARAM_WRAP_T: Pionowe powtarzanie tekstury.
\param value Wartosc parametru.
*/
XE_IMPEX void XE_CALLTYPE XeTextureParameter(XE_ESTATE type,float value);

/*--- XeTextureMode() ---*/
/*! Ustala tryb mieszania kolorow tekstur z kolorami wierzcholkow.
\param mode Tryb mieszania. Dostepne wartosci:<br>
XE_TEXTURE_MODE_ADD: Dodaje kolor tekstury do koloru wierzcholka.<br>
XE_TEXTURE_MODE_MODULATE: Moduluje kolor tekstury przez kolor wierzcholka.<br>
XE_TEXTURE_MODE_DECAL: Zastepuje kolor wierzcholka kolorem tekstury.<br>
XE_TEXTURE_MODE_BLEND: Miesza kolor wierzcholka z kolorem tekstury.<br>
XE_TEXTURE_MODE_REPLACE: Zastepuje kolor wierzcholka kolorem tekstury.<br>
XE_TEXTURE_MODE_COMBINE: Opcje mieszania dla multitekstur .[Obecnie wartosc nie uzywana]
*/
XE_IMPEX void XE_CALLTYPE XeTextureMode(XE_ESTATE mode);

/*--- XeTextureMulti() ---*/
/*! Wlacza tryb multiteksturingu.
Pozwala to na mieszanie tekstur poprzez uzywanie wiecej niz jednej jednostki teksturujacej.
Dodatkowo umieszcza w zmiennej stanu XE_TEXTURE_UNIT_COUNT wartosc maxymalnej liczby jednostek teksturowania jaka dysponuje karta.
\return true jesli operacja powiodla sie; false jesli karta nie obsluguje wieloteksturowania.
*/
XE_IMPEX bool XE_CALLTYPE XeTextureMulti();

/*--- XeTextureUnit() ---*/
/*! Aktywuje jednostke teksturowania.
\param unit Numer jednostki teksturowania.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeTextureUnit(int unit);

/*--- XeTextureGet() ---*/
/*! Pobiera wartosc zmiennej stanu tekstury.
\param index Element tekstury.
\param type Zmienna stanu. Dostepne wartosci:<br>
XE_TEXTURE_WIDTH: Szerokosc tekstury.<br>
XE_TEXTURE_HEIGHT: Wysokosc tekstury.<br>
XE_TEXTURE_PIXEL_BUFFER: Wskaznik na bufor zablokowanych pikseli.<br>
XE_TEXTURE_PIXEL_SIZE: Rozmiar bufora zablokowanych pikseli.
\return Wskaznik nieokreslony na zmienna stanu tekstury. Nalezy rzutowac na konkretny typ danej zmiennej stanu.
*/
XE_IMPEX void* XE_CALLTYPE XeTextureGet(XE_ELM_TEXTURE index,XE_ESTATE type);

/*--- XeTextureGetReal() ---*/
/*! Pobiera wartosc zmiennej stanu tekstury bezposrednio z GPU.
\param index Element tekstury.
\param type Zmienna stanu. Dostepne wartosci:<br>
XE_TEXTURE_WIDTH: Szerokosc tekstury.<br>
XE_TEXTURE_HEIGHT: Wysokosc tekstury.<br>
\return Wartość zmiennej stanu.
*/
XE_IMPEX int XE_CALLTYPE XeTextureGetReal(XE_ELM_TEXTURE index,XE_ESTATE type);

/*--- XeTextureType() ---*/
/*! Pobiera typ tekstury.
\param index Element tekstury.
\return Typ tekstury. Dostepne wartosci:<br>
XE_2D: Tekstura 2D.<br>
XE_DATA: Tekstura float.
*/
XE_IMPEX XE_ESTATE XE_CALLTYPE XeTextureType(XE_ELM_TEXTURE index);

/*--- XeDisplaylistCreate() ---*/
/*! Tworzy liste wyswietlania.
Czyli liste polecen graficznych rysujacych scene. Polecenia sa kompilowane i zapamietywane, a podczas wywolywania listy polecenia sa przetwarzane szybko.
\param elm Wskaznik na element listy wyswietlania. Zwraca do niego utworzony element jesli operacja powiodla sie.
\param count Liczba sublist. Domyslnie rowna 1.
*/
XE_IMPEX void XE_CALLTYPE XeDisplaylistCreate(XE_ELM_DISPLAYLIST* elm,unsigned int count=1);

/*--- XeDisplaylistDestroy() ---*/
/*! Niszczy liste wyswietlania.
\param index Element listy wyswietlania.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeDisplaylistDestroy(XE_ELM_DISPLAYLIST index);

/*--- XeDisplaylistBegin() ---*/
/*! Rozpoczyna zapamietywanie polecen rysowania.
\param index Element listy wyswietlania.
\param id Indeks sublisty.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeDisplaylistBegin(XE_ELM_DISPLAYLIST index,unsigned int id=0);

/*--- XeDisplaylistEnd() ---*/
/*! Konczy zapamietywanie polecen rysowania. */
XE_IMPEX void XE_CALLTYPE XeDisplaylistEnd();

/*--- XeDisplaylistCall() ---*/
/*! Wywoluje zapamietane polecenia z listy wyswietlania.
\param index Element listy wyswietlania.
\param id Indeks sublisty.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeDisplaylistCall(XE_ELM_DISPLAYLIST index,unsigned int id=0);

#ifdef XE_COMPILE_CORE_MATH

/*--- XeRenderQueueActivate() ---*/
/*! Aktywuje kolejke renderowania (wlacza buforowanie danych grupujac je tak by wykonywaly mniejsza ilosc polecen rysowania). */
XE_IMPEX void XE_CALLTYPE XeRenderQueueActivate();

/*--- XeRenderQueueUnactivate() ---*/
/*! Dezaktywuje kolejke renderowania.*/
XE_IMPEX void XE_CALLTYPE XeRenderQueueUnactivate();

/*--- XeRenderQueueFlush() ---*/
/*! Nakazuje wyrenderowac buforowane dane.
\param leavebuffers Ustala czy ma zachowac dane w buforach po ich wyrenderowaniu.
*/
XE_IMPEX void XE_CALLTYPE XeRenderQueueFlush(bool leavebuffers=false);

/*--- XeRenderQueueClear() ---*/
/*! Czysci kolejke renderowania (zalecane po uzyciu XeRenderQueueFlush()). */
XE_IMPEX void XE_CALLTYPE XeRenderQueueClear();

/*--- XeRenderQueueTransformVertices() ---*/
/*! Wlacza/wylacza transformacje wierzcholkow.
\param mode Tryb. true: wlaczony; false: wylaczony.
*/
XE_IMPEX void XE_CALLTYPE XeRenderQueueTransformVertices(bool mode);

/*--- XeRenderQueueTransformCoords() ---*/
/*! Wlacza/wylacza transformacje koordynatow.
\param mode Tryb. true: wlaczony; false: wylaczony.
*/
XE_IMPEX void XE_CALLTYPE XeRenderQueueTransformCoords(bool mode);

/*--- XeRenderQueueReallocCount() ---*/
/*! Ustala liczbe elementow o jaka zwieksza sie pojemnosc maksymalna bufora po jej przekroczeniu.
\param count Liczba elementow.
*/
XE_IMPEX void XE_CALLTYPE XeRenderQueueReallocCount(unsigned int count);

/*--- XeRenderQueueGetModelviewMat() ---*/
/*! Pobiera bierzaca macierz widoku. */
XE_IMPEX void XE_CALLTYPE XeRenderQueueGetModelviewMat();

/*--- XeRenderQueueGetTextureMat() ---*/
/*! Pobiera bierzaca macierz tekstur. */
XE_IMPEX void XE_CALLTYPE XeRenderQueueGetTextureMat();

#endif /* XE_COMPILE_CORE_MATH */

} // namespace Photon

#endif /* XE_COMPILE_PHOTON */

#ifdef XE_COMPILE_CHAOS

namespace Chaos
{

#ifdef XE_COMPILE_CORE_MATH

/*--- XeShapeIsConvex2D() ---*/
/*! Sprawdza czy bryla jest wypukla.
\param vertices Wskaznik na tablice wierzcholkow.
\param count Ilosc wierzcholkow w tablicy.
\return true jesli bryla jest wypukla.
*/
XE_IMPEX bool XE_CALLTYPE XeShapeIsConvex2D(XE_VECTOR* vertices,unsigned int count);

/*--- XeShapeCollision2D() ---*/
/*! Sprawdza czy punkt koliduje z bryla wypukla.
\param point Punkt.
\param vertices Wskaznik na tablice wierzcholkow.
\param count Ilosc wierzcholkow w tablicy.
\param negfaces Zmienia kierunek scian na przeciwne.
\param onplane Ustala czy uznac punkt na plaszczyznie sciany jako kolizje.
\return true jesli nastapila kolizja.
*/
XE_IMPEX bool XE_CALLTYPE XeShapeCollision2D(XE_VECTOR point,XE_VECTOR* vertices,unsigned int count,bool negfaces=false,bool onplane=true);

/*--- XePolygonsCollision2D() ---*/
/*! Sprawdza czy dwa wielokaty wypukle koliduja ze soba.
\param shape1 Wskaznik na tablice wierzcholkow pierwszego wielokata.
\param count1 Ilosc wierzcholkow pierwszego wielokata.
\param shape2 Wskaznik na tablice wierzcholkow drugiego wielokata.
\param count2 Ilosc wierzcholkow drugiego wielokata.
\return true jesli nastapila kolizja.
*/
XE_IMPEX bool XE_CALLTYPE XePolygonsCollision2D(XE_VECTOR* shape1,unsigned int count1,XE_VECTOR* shape2,unsigned int count2);

#endif /* XE_COMPILE_CORE_MATH */

/*--- XePhysicsSetFactor() ---*/
/*! Ustala wspolczynnik fizyczny.
\param factor Wspolczynnik.
*/
XE_IMPEX void XE_CALLTYPE XePhysicsSetFactor(double factor);

/*--- XePhysicsGetFactor() ---*/
/*! Zwraca wspolczynnik fizyczny.
\return Wspolczynnik.
*/
XE_IMPEX double XE_CALLTYPE XePhysicsGetFactor();

#ifdef XE_COMPILE_CORE_MATH

/*--- XePhysicsUpdate() ---*/
/*! Aktualizuje fizyke. */
XE_IMPEX void XE_CALLTYPE XePhysicsUpdate();

#endif /* XE_COMPILE_CORE_MATH */

/*--- XeSubstanceCreate() ---*/
/*! Tworzy substancje.
\param elm Wskaznik na element substancji. Zwraca do niego utworzony element jesli operacja powiodla sie.
*/
XE_IMPEX void XE_CALLTYPE XeSubstanceCreate(XE_ELM_SUBSTANCE* elm);

/*--- XeSubstanceSetObject() ---*/
/*! Ustala obiekt.
\param index Element substancji.
\param object Obiekt.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeSubstanceSetObject(XE_ELM_SUBSTANCE index,double object);

/*--- XeSubstanceSetPosition() ---*/
/*! Ustala pozycje.
\param index Element substancji.
\param p Pozycja.
\param notresetprev Sprawdza czy ma nie ustawiac (resetowac) stan poprzedniej pozycji na obecna (domyslnie: false).
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeSubstanceSetPosition(XE_ELM_SUBSTANCE index,XE_VECTOR p,bool notresetprev=false);

/*--- XeSubstanceSetRange() ---*/
/*! Ustala promien.
\param index Element substancji.
\param range Promien.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeSubstanceSetRange(XE_ELM_SUBSTANCE index,double range);

/*--- XeSubstanceSetMass() ---*/
/*! Ustala mase.
\param index Element substancji.
\param mass Masa.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeSubstanceSetMass(XE_ELM_SUBSTANCE index,double mass);

/*--- XeSubstanceSetEnergy() ---*/
/*! Ustala energie.
\param index Element substancji.
\param energy Energia.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeSubstanceSetEnergy(XE_ELM_SUBSTANCE index,double energy);

/*--- XeSubstanceSetSolid() ---*/
/*! Ustala czy obiekt jest trwaly.
\param index Element substancji.
\param solid Trwalosc. Dostepne wartosci:<br>
XE_FLAG_SUBSTANCE_SOLID_RANGE: Umozliwia kolidowanie promieniowe.<br>
XE_FLAG_SUBSTANCE_SOLID_SURFACE: Umozliwia kolidowanie powierzchniowe.<br>
Wartosci sa maskami bitowymi wiec mozna je laczyc operatorem "|".
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeSubstanceSetSolid(XE_ELM_SUBSTANCE index,unsigned int solid);

/*--- XeSubstanceAddInertion() ---*/
/*! Dodaje bezwladnosc.
\param index Element substancji.
\param i Wektor bezwladnosci.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeSubstanceAddInertion(XE_ELM_SUBSTANCE index,XE_VECTOR i);

/*--- XeSubstanceResetInertion() ---*/
/*! Zeruje bezwladnosc.
\param index Element substancji.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeSubstanceResetInertion(XE_ELM_SUBSTANCE index);

/*--- XeSubstanceGet() ---*/
/*! Pobiera wartosc zmiennej stanu substancji.
\param index Element substancji.
\param type Zmienna stanu substancji. Dostepne wartosci:<br>
XE_SUBSTANCE_ID: Indeks substancji.<br>
XE_SUBSTANCE_OBJECT: Obiekt.<br>
XE_SUBSTANCE_POSITION: Pozycja.<br>
XE_SUBSTANCE_RANGE: Promien.<br>
XE_SUBSTANCE_MASS: Masa.<br>
XE_SUBSTANCE_ENERGY: Energia.<br>
XE_SUBSTANCE_SOLID: Trwalosc.<br>
XE_SUBSTANCE_INERTION: Wektor bezwladnosci.
XE_SUBSTANCE_LAST_POSITION: Pozycja z ostatniej aktualizacji.
\return Wskaznik nieokreslony na zmienna stanu substancji. Nalezy rzutowac na konkretny typ danej zmiennej stanu. 
*/
XE_IMPEX void* XE_CALLTYPE XeSubstanceGet(XE_ELM_SUBSTANCE index,XE_ESTATE type);

/*--- XeSubstanceDestroy() ---*/
/*! Niszczy substancje.
\param index Element substancji.
*/
XE_IMPEX bool XE_CALLTYPE XeSubstanceDestroy(XE_ELM_SUBSTANCE index);

/*--- XeRayCreate() ---*/
/*! Tworzy wiazke (promien).
\param elm Wskaznik na element wiazki. Zwraca do niego utworzony obiekt jesli operacja powiodla sie.
*/
XE_IMPEX void XE_CALLTYPE XeRayCreate(XE_ELM_RAY* elm);

/*--- XeRaySetPosition() ---*/
/*! Ustala pozycje.
\param index Element wiazki.
\param p Pozycja.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeRaySetPosition(XE_ELM_RAY index,XE_VECTOR p);

/*--- XeRaySetDirection() ---*/
/*! Ustala kierunek.
\param index Element wiazki.
\param dir Kierunek.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeRaySetDirection(XE_ELM_RAY index,XE_ANGLEVECTOR dir);

/*--- XeRaySetAngle() ---*/
/*! Ustala kat zasiegu (rozproszenia).
\param index Element wiazki.
\param angle Kat angle.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeRaySetAngle(XE_ELM_RAY index,double angle);

/*--- XeRaySetEnergy() ---*/
/*! Ustala energie.
\param index Element wiazki.
\param energy Energia.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeRaySetEnergy(XE_ELM_RAY index,double energy);

/*--- XeRaySetFrequency() ---*/
/*! Ustala czestotliwosc.
\param index Element wiazki.
\param frequency Czestotliwosc.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeRaySetFrequency(XE_ELM_RAY index,double frequency);

/*--- XeRayGet() ---*/
/*! Pobiera wartosc zmiennej stanu wiazki.
\param index Element wiazki.
\param type Zmienna stanu wiazki. Dostepne wartosci:<br>
XE_RAY_ID: Indeks wiazki.<br>
XE_RAY_POSITION: Pozycja.<br>
XE_RAY_DIRECTION: Kierunek.<br>
XE_RAY_ANGLE: Zasieg (rozproszenie).<br>
XE_RAY_ENERGY: Energia.<br>
XE_RAY_FREQUENCY: Czestotliwosc.
\return Wskaznik nieokreslony na zmienna stanu wiazki. Nalezy rzutowac na konkretny typ danej zmiennej stanu.
*/
XE_IMPEX void* XE_CALLTYPE XeRayGet(XE_ELM_RAY index,XE_ESTATE type);

/*--- XeRayDestroy() ---*/
/*! Niszczy wiazke.
\param index Element wiazki.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeRayDestroy(XE_ELM_RAY index);

/*--- XeCovalenceForceCreate() ---*/
/*! Tworzy sile kowalencyjna (miedzyczasteczkowa).
\param elm Wskaznik na element sily kowalencyjnej. Zwraca do niego utworzony element jesli operacja powiodla sie.
*/
XE_IMPEX void XE_CALLTYPE XeCovalenceForceCreate(XE_ELM_COVALENCE_FORCE* elm);

/*--- XeCovalenceForceSetSubstance() ---*/
/*! Ustala substancje.
\param index Element sily kowalencyjnej.
\param type Typ substancji. Dostepne wartosci:<br>
XE_COVALENCE_FORCE_SUBSTANCE_1: Pierwsza substancja.<br>
XE_COVALENCE_FORCE_SUBSTANCE_2: Druga substancja.
\param substance Element substancji.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeCovalenceForceSetSubstance(XE_ELM_COVALENCE_FORCE index,XE_ESTATE type,XE_ELM_SUBSTANCE substance);

/*--- XeCovalenceForceSetForce() ---*/
/*! Ustala sile.
\param index Element sily kowalencyjnej.
\param force Sila.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeCovalenceForceSetForce(XE_ELM_COVALENCE_FORCE index,double force);

/*--- XeCovalenceForceSetForceFromDistance() ---*/
/*! Ustala sile biorac ja z odleglosci pomiedzy substancjami.
\param index Element sily kowalencyjnej.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeCovalenceForceSetForceFromDistance(XE_ELM_COVALENCE_FORCE index);

/*--- XeCovalenceSetForceResilience() ---*/
/*! Ustala sprezystosc.
\param index Element sily kowalencyjnej.
\param resilience Sprezystosc.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeCovalenceForceSetResilience(XE_ELM_COVALENCE_FORCE index,double resilience);

/*--- XeCovalenceForceSetRange() ---*/
/*! Ustala zasieg sily, po ktorego przekroczeniu nastepuje rozerwanie sily.
\param index Element sily kowalencyjnej.
\param range Zasieg.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeCovalenceForceSetRange(XE_ELM_COVALENCE_FORCE index,double range);

/*--- XeCovalenceForceSetSolid() ---*/
/*! Ustala zasieg sily, po ktorego przekroczeniu nastepuje rozerwanie sily.
\param index Element sily kowalencyjnej.
\param solid Solidnosc; zachowanie sily po przekroczeniu zasiegu. Dostepne wartosci:<br>
XE_COVALENCE_FORCE_SOLID_ALWAYS: Utrzymuje sile zawsze.<br>
XE_COVALENCE_FORCE_SOLID_TEMPORARY: Tymczasowo nie utrzymuje sily, dopoki dystans pomiedzy substancjami nie bedzie w zasiegu sily.<br>
XE_COVALENCE_FORCE_SOLID_NEVER: Po przekroczeniu zasiegu zrywa polaczenie.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeCovalenceForceSetSolid(XE_ELM_COVALENCE_FORCE index,XE_ESTATE solid);

/*--- XeCovalenceForceGet() ---*/
/*! Pobiera wartosc zmiennej stanu sily kowalencyjnej.
\param index Element sily kowalencyjnej.
\param type Zmienna stanu sily kowalencyjnej. Dostepne wartosci:<br>
XE_COVALENCE_FORCE_ID: Indeks sily kowalencyjnej.<br>
XE_COVALENCE_FORCE_SUBSTANCE_1: Pierwsza substancja.<br>
XE_COVALENCE_FORCE_SUBSTANCE_2: Druga substancja.<br>
XE_COVALENCE_FORCE_FORCE: Sila.<br>
XE_COVALENCE_FORCE_RESILIENCE: Sprezystosc.<br>
XE_COVALENCE_FORCE_RANGE: Zasieg. Jesli odleglosc substancji bedzie wieksza niz zasieg i sila nie jest solidna to nastapi rozerwanie na stale badz na chwile.<br>
XE_COVALENCE_FORCE_SOLID: Solidnosc, utrzymywanie polaczenia po przekroczeniu zasiegu: zawsze, tymczasowo lub nigdy.
\return Wskaznik nieokreslony na zmienna stanu sily kowalencyjnej. Nalezy rzutowac na konkretny typ danej zmiennej stanu.
*/
XE_IMPEX void* XE_CALLTYPE XeCovalenceForceGet(XE_ELM_COVALENCE_FORCE index,XE_ESTATE type);

/*--- XeCovalenceForceDestroy() ---*/
/*! Niszczy sile kowalencyjna.
\param index Element sily kowalencyjnej.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeCovalenceForceDestroy(XE_ELM_COVALENCE_FORCE index);

/*--- XeForceCreate() ---*/
/*! Tworzy sile.
\param elm Wskaznik na element sily. Zwraca do niego utworzony element jesli operacja powiodla sie.
*/
XE_IMPEX void XE_CALLTYPE XeForceCreate(XE_ELM_FORCE* elm);

/*--- XeForceSetPosition() ---*/
/*! Ustala pozycje.
\param index Element sily.
\param p Pozycja.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeForceSetPosition(XE_ELM_FORCE index,XE_VECTOR p);

/*--- XeForceSetDirection() ---*/
/*! Ustala kierunek.
\param index Element sily.
\param dir Kierunek.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeForceSetDirection(XE_ELM_FORCE index,XE_ANGLEVECTOR dir);

/*--- XeForceSetDirectionToAnchor() ---*/
/*! Ustala kierunek do zakotwiczonej substancji kierunku.
\param index Element sily.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeForceSetDirectionToAnchor(XE_ELM_FORCE index);

/*--- XeForceSetForce() ---*/
/*! Ustala sile.
\param index Element sily.
\param force Sila.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeForceSetForce(XE_ELM_FORCE index,double force);

/*--- XeForceSetSpeed() ---*/
/*! Ustala predkosc.
\param index Element sily.
\param speed Predkosc.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeForceSetSpeed(XE_ELM_FORCE index,double speed);

/*--- XeForceSetAnchor() ---*/
/*! Ustala kotwice (substancje z ktora jest polaczony).
\param index Element sily.
\param anchor Substancja.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeForceSetAnchor(XE_ELM_FORCE index,XE_ELM_SUBSTANCE anchor);

/*--- XeForceSetAnchorDirection() ---*/
/*! Ustala kotwice kierunku (substancje z ktora jest polaczony, ktora wskazuje kierunek sily).
\param index Element sily.
\param anchordir Substancja.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeForceSetAnchorDirection(XE_ELM_FORCE index,XE_ELM_SUBSTANCE anchordir);

/*--- XeForceSetRicochet() ---*/
/*! Ustala wspolczynnik sily odrzutu oddzialujaca na inne substancje gdy sila jest polaczana z jakas substancja.
\param index Element sily.
\param ricochet Wspolczynnik sily dla reszty substancji.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeForceSetRicochet(XE_ELM_FORCE index,float ricochet);

/*--- XeForceSetRange() ---*/
/*! Ustala zasieg sily. Jesli rowny 0 to zasieg jest globalny.
\param index Element sily.
\param range Zasieg sily.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeForceSetRange(XE_ELM_FORCE index,double range);

/*--- XeForceGet() ---*/
/*! Pobiera wartosc zmiennej stanu sily.
\param index Element sily.
\param type Zmienna stanu. Dostepne wartosci:<br>
XE_FORCE_ID: Indeks sily.<br>
XE_FORCE_POSITION: Pozycja.<br>
XE_FORCE_DIRECTION: Kierunek.<br>
XE_FORCE_FORCE: Sila.<br>
XE_FORCE_SPEED: Predkosc.<br>
XE_FORCE_ANCHOR: Kotwica (substancja z ktora jest polaczony).<br>
XE_FORCE_ANCHOR_DIRECTION: Kotwica kierunku (substancja z ktora jest polaczony, ktora wskazuje kierunek sily).<br>
XE_FORCE_RICOCHET: Wspolczynnik sily oddzialujacej na inne sily gdy dana sila ma ustalona kotwice.
\return Wskaznik nieokreslony na zmienna stanu sily. Nalezy rzutowac na konkretny typ danej zmiennej stanu.
*/
XE_IMPEX void* XE_CALLTYPE XeForceGet(XE_ELM_FORCE index,XE_ESTATE type);

/*--- XeForceDestroy() ---*/
/*! Niszczy sile.
\param index Element sily.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeForceDestroy(XE_ELM_FORCE index);

/*--- XeSurfaceCreate() ---*/
/*! Tworzy powierzchnie.
\param elm Wskaznik na element powierzchni. Zwraca do niego utworzony element jesli operacja powiodla sie.
*/
XE_IMPEX void XE_CALLTYPE XeSurfaceCreate(XE_ELM_SURFACE* elm);

/*--- XeSurfaceSetSubstance() ---*/
/*! Ustala substancje.
\param index Element powierzchni.
\param type Typ substancji. Dostepne wartosci:<br>
XE_SURFACE_SUBSTANCE_1: Pierwsza substancja.<br>
XE_SURFACE_SUBSTANCE_2: Druga substancja.<br>
XE_SURFACE_SUBSTANCE_3: Trzecia substancja.
\param substance Element substancji.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeSurfaceSetSubstance(XE_ELM_SURFACE index,XE_ESTATE type,XE_ELM_SUBSTANCE substance);

/*--- XeSurfaceSetTexture() ---*/
/*! Ustala teksture.
\param index Element powierzchni.
\param texture Element tekstury.
\param type Typ tekstury. Musi byc: XE_2D.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeSurfaceSetTexture(XE_ELM_SURFACE index,XE_ELM_TEXTURE texture,XE_ESTATE type);

/*--- XeSurfaceSetCoord() ---*/
/*! Ustala koordynaty tekstury.
\param index Element powierzchni.
\param type Typ substancji tekstury. Dostepne wartosci:<br>
XE_SURFACE_COORD_SUBSTANCE_1: Pierwsza substancja tekstury.<br>
XE_SURFACE_COORD_SUBSTANCE_2: Druga substancja tekstury.<br>
XE_SURFACE_COORD_SUBSTANCE_3: Trzecia substancja tekstury.
\param c Koordynaty.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeSurfaceSetCoord(XE_ELM_SURFACE index,XE_ESTATE type,XE_VECTOR c);

/*--- XeSurfaceSetDensity() ---*/
/*! Ustala gestosc (grubosc) powierzchni (wazne przy wykrywaniu kolizji).
\param index Element powierzchni.
\param density Gestosc.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeSurfaceSetDensity(XE_ELM_SURFACE index,double density);

/*--- XeSurfaceSetOrientation() ---*/
/*! Ustala orientacje.
\param index Element powierzchni.
\param orientation Orientacja (wartosci: -1;0;1).
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeSurfaceSetOrientation(XE_ELM_SURFACE index,int orientation);

/*--- XeSurfaceSetOrder() ---*/
/*! Ustala kolejnosc dla kolejki rysowania.
\param index Element powierzchni.
\param order Kolejnosc.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeSurfaceSetOrder(XE_ELM_SURFACE index,double order);

/*--- XeSurfaceGet() ---*/
/*! Pobiera wartosc zmiennej stanu powierzchni.
\param index Element powierzchni.
\param type Zmienna stanu. Dostepne wartosci:<br>
XE_SURFACE_ID: Indeks powierzchni.<br>
XE_SURFACE_SUBSTANCE_1: Pierwsza substancja.<br>
XE_SURFACE_SUBSTANCE_2: Druga substancja.<br>
XE_SURFACE_SUBSTANCE_3: Trzecia substancja.<br>
XE_SURFACE_TEXTURE: Element tekstury.<br>
XE_SURFACE_TEXTURE_TYPE: Typ tekstury.<br>
XE_SURFACE_ORDER: Kolejnosc dla kolejki rysowania.<br>
XE_SURFACE_COORD_SUBSTANCE_1: Koordynaty pierwszej tekstury.<br>
XE_SURFACE_COORD_SUBSTANCE_2: Koordynaty drugiej tekstury.<br>
XE_SURFACE_COORD_SUBSTANCE_3: Koordynaty trzeciej tekstury.<br>
XE_SURFACE_DIFFUSION: Rozproszenie swiatla.<br>
XE_SURFACE_MIRROR: Poziom odbicia swiatla.<br>
XE_SURFACE_ORIENTATION: Orientacja.
\return Wskaznik nieokreslony na zmienna stanu powierzchni. Nalezy rzutowac na konkretny typ danej zmiennej stanu.
*/
XE_IMPEX void* XE_CALLTYPE XeSurfaceGet(XE_ELM_SURFACE index,XE_ESTATE type);

/*--- XeSurfaceDestroy() ---*/
/*! Niszczy powierzchnie.
\param index Element powierzchni.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeSurfaceDestroy(XE_ELM_SURFACE index);

/*--- XeSurfaceSort() ---*/
/*! Sortuje powierzchnie.
\param invert Sortowanie odwrotne.
*/
XE_IMPEX void XE_CALLTYPE XeSurfaceSort(bool invert);

/*--- XeParticleCreate() ---*/
/*! Tworzy czasteczke.
\param elm Wskaznik na element czasteczki. Zwraca do niego utworzony element jesli operacja powiodla sie.
*/
XE_IMPEX void XE_CALLTYPE XeParticleCreate(XE_ELM_PARTICLE* elm);

/*--- XeParticleSetSubstance() ---*/
/*! Ustala substancje.
\param index Element czasteczki.
\param substance Element substancji.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeParticleSetSubstance(XE_ELM_PARTICLE index,XE_ELM_SUBSTANCE substance);

/*--- XeParticleSetSize() ---*/
/*! Ustala rozmiar.
\param index Element czasteczki.
\param size Rozmiar.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeParticleSetSize(XE_ELM_PARTICLE index,double size);

/*--- XeParticleSetTexture() ---*/
/*! Ustala teksture.
\param index Element czasteczki.
\param texture Element tekstury.
\param type Typ tekstury. Musi byc: XE_2D.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeParticleSetTexture(XE_ELM_PARTICLE index,XE_ELM_TEXTURE texture,XE_ESTATE type);

/*--- XeParticleGet() ---*/
/*! Pobiera wartosc zmiennej stanu czasteczki.
\param index Element czasteczki.
\param type Zmienna stanu czasteczki. Dostepne wartosci:<br>
XE_PARTICLE_ID: Indeks czasteczki.<br>
XE_PARTICLE_WIDTH: Szerokosc.<br>
XE_PARTICLE_HEIGHT: Wysokosc.<br>
XE_PARTICLE_TEXTURE: Element tekstury.<br>
XE_PARTICLE_TEXTURE_TYPE: Typ tekstury.<br>
XE_PARTICLE_SUBSTANCE: Element substancji.
\return Wskaznik nieokreslony na zmienna stanu czasteczki. Nalezy rzutowac na konkretny typ danej zmiennej stanu.
*/
XE_IMPEX void* XE_CALLTYPE XeParticleGet(XE_ELM_PARTICLE index,XE_ESTATE type);

/*--- XeParticleDestroy() ---*/
/*! Niszczy czasteczke.
\param index Element czasteczki.
*/
XE_IMPEX bool XE_CALLTYPE XeParticleDestroy(XE_ELM_PARTICLE index);

/*--- XeGlobalForceCreate() ---*/
/*! Tworzy sile globalna.
\param elm Wskaznik na element sily globalnej. Zwraca do niego utworzony element jesli operacja powiodla sie.
*/
XE_IMPEX void XE_CALLTYPE XeGlobalForceCreate(XE_ELM_GLOBAL_FORCE* elm);

/*--- XeGlobalForceSetDirection() ---*/
/*! Ustala kierunek.
\param index Element sily globalnej.
\param dir Kierunek.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeGlobalForceSetDirection(XE_ELM_GLOBAL_FORCE index,XE_ANGLEVECTOR dir);

/*--- XeGlobalForceSetForce() ---*/
/*! Ustala sile.
\param index Element sily globalnej.
\param force Sila.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeGlobalForceSetForce(XE_ELM_GLOBAL_FORCE index,double force);

/*--- XeGlobalForceGet() ---*/
/*! Pobiera wartosc zmiennej stanu sily globalnej.
\param index Element sily globalnej.
\param type Zmienna stanu sily globalnej. Dostepne wartosci:<br>
XE_GLOBAL_FORCE_ID: Indeks sily globalnej.<br>
XE_GLOBAL_FORCE_DIRECTION: Kierunek.<br>
XE_GLOBAL_FORCE_FORCE: Sila.
\return Wskaznik nieokreslony na zmienna stanu sily globalnej. Nalezy rzutowac na konkretny typ danej zmiennej stanu.
*/
XE_IMPEX void* XE_CALLTYPE XeGlobalForceGet(XE_ELM_GLOBAL_FORCE index,XE_ESTATE type);

/*--- XeGlobalForceDestroy() ---*/
/*! Niszczy sile globalna.
\param index Element sily globalnej.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeGlobalForceDestroy(XE_ELM_GLOBAL_FORCE index);

/*--- XeBlackHoleCreate() ---*/
/*! Tworzy czarna dziure.
\param elm Wskaznik na element czarnej dziury. Zwraca do niego utworzony element jesli operacja powiodla sie.
*/
XE_IMPEX void XE_CALLTYPE XeBlackHoleCreate(XE_ELM_BLACK_HOLE* elm);

/*--- XeBlackHoleSetPosition() ---*/
/*! Ustala pozycje.
\param index Element czarnej dziury.
\param p Pozycja.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeBlackHoleSetPosition(XE_ELM_BLACK_HOLE index,XE_VECTOR p);

/*--- XeBlackHoleSetForce() ---*/
/*! Ustala sile.
\param index Element czarnej dziury.
\param force Sila.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeBlackHoleSetForce(XE_ELM_BLACK_HOLE index,double force);

/*--- XeBlackHoleSetRange() ---*/
/*! Ustala zasieg.
\param index Element czarnej dziury.
\param range Zasieg.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeBlackHoleSetRange(XE_ELM_BLACK_HOLE index,double range);

/*--- XeBlackHoleSetAnchor() ---*/
/*! Ustala kotwice (czarna dziure z ktora jest polaczona).
\param index Element czarnej dziury.
\param anchor Element czarnej dziury.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeBlackHoleSetAnchor(XE_ELM_BLACK_HOLE index,XE_ELM_BLACK_HOLE anchor);

/*--- XeBlackHoleGet() ---*/
/*! Pobiera wartosc zmiennej stanu czarnej dziury.
\param index Element czarnej dziury.
\param type Zmienna stanu czarnej dziury. Dostepne wartosci:<br>
XE_BLACK_HOLE_ID: Indeks czarnej dziury.<br>
XE_BLACK_HOLE_POSITION: Pozycja.<br>
XE_BLACK_HOLE_FORCE: Sila.<br>
XE_BLACK_HOLE_RANGE: Zasieg.<br>
XE_BLACK_HOLE_ANCHOR: Element czarnej dziury.
\return Wskaznik nieokreslony na zmienna stanu czarnej dziury. Nalezy rzutowac na konkretny typ danej zmiennej stanu.
*/
XE_IMPEX void* XE_CALLTYPE XeBlackHoleGet(XE_ELM_BLACK_HOLE index,XE_ESTATE type);

/*--- XeBlackHoleDestroy() ---*/
/*! Niszczy czarna dziure.
\param index Element czarnej dziury.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeBlackHoleDestroy(XE_ELM_BLACK_HOLE index);

/*--- XeWaveForceCreate() ---*/
/*! Tworzy sile falowa.
\param elm Wskaznik na element sily falowej. Zwraca do niego utworzony element jesli operacja powiodla sie.
*/
XE_IMPEX void XE_CALLTYPE XeWaveForceCreate(XE_ELM_WAVE_FORCE* elm);

/*--- XeWaveForceSetPosition() ---*/
/*! Ustala pozycje.
\param index Element sily falowej.
\param p Pozycja.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeWaveForceSetPosition(XE_ELM_WAVE_FORCE index,XE_VECTOR p);

/*--- XeWaveForceSetForce() ---*/
/*! Ustala sile.
\param index Element sily falowej.
\param force Sila.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeWaveForceSetForce(XE_ELM_WAVE_FORCE index,double force);

/*--- XeWaveForceSetRange() ---*/
/*! Ustala zasieg.
\param index Element sily falowej.
\param range Zasieg.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeWaveForceSetRange(XE_ELM_WAVE_FORCE index,double range);

/*--- XeWaveForceSetLength() ---*/
/*! Ustala dlugosc.
\param index Element sily falowej.
\param length Dlugosc.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeWaveForceSetLength(XE_ELM_WAVE_FORCE index,double length);

/*--- XeWaveForceSetSpeed() ---*/
/*! Ustala predkosc.
\param index Element sily falowej.
\param speed Predkosc.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeWaveForceSetSpeed(XE_ELM_WAVE_FORCE index,double speed);

/*--- XeWaveForceSetOrientation() ---*/
/*! Ustala sile.
\param index Element sily falowej.
\param orientation Orientacja. Dostepne wartosci:<br>
XE_WAVE_FORCE_ORIENTATION_IN: Sila dosrodkowa.<br>
XE_WAVE_FORCE_ORIENTATION_OUT: Sila odsrodkowa.<br>
XE_WAVE_FORCE_ORIENTATION_INNER: Sila dosrodkowa dla grzbietu fali.<br>
XE_WAVE_FORCE_ORIENTATION_OUTER: Sila odsrodkowa dla grzbietu fali.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeWaveForceSetOrientation(XE_ELM_WAVE_FORCE index,XE_ESTATE orientation);

/*--- XeWaveForceGet() ---*/
/*! Pobiera wartosc zmiennej stanu sily falowej.
\param index Element sily falowej.
\param type Zmienna stanu sily falowej. Dostepne wartosci:<br>
XE_WAVE_FORCE_ID: Indeks sily falowej.<br>
XE_WAVE_FORCE_POSITION: Pozycja.<br>
XE_WAVE_FORCE_FORCE: Sila.<br>
XE_WAVE_FORCE_RANGE: Zasieg.<br>
XE_WAVE_FORCE_LENGTH: Dlugosc.<br>
XE_WAVE_FORCE_SPEED: Predkosc.<br>
XE_WAVE_FORCE_ORIENTATION: Orientacja.
\return Wskaznik nieokreslony na zmienna stanu sily falowej. Nalezy rzutowac na konkretny typ danej zmiennej stanu.
*/
XE_IMPEX void* XE_CALLTYPE XeWaveForceGet(XE_ELM_WAVE_FORCE index,XE_ESTATE type);

/*--- XeWaveForceDestroy() ---*/
/*! Niszczy sile falowa.
\param index Element sily falowej.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeWaveForceDestroy(XE_ELM_WAVE_FORCE index);

} // namespace Chaos

#endif /* XE_COMPILE_CHAOS */

#ifdef XE_COMPILE_CORE_MATH

/*--- XeInAngleField() ---*/
/*! Sprawdza czy punkt znajduje sie w polu widzenia kamery perspektywistycznej o danym kacie patrzenia.
\param p Pozycja punktu.
\param angle Kat pola widzenia.
\return true jesli punkt znajduje sie w polu widzenia.
*/
XE_IMPEX bool XE_CALLTYPE XeInAngleField(XE_VECTOR p,double angle);

/*--- XeAngleDifference() ---*/
/*! oblicza roznice pomiedzy dwoma katami.
\param angle1 Pierwszy kat.
\param angle2 Drugi kat.
\return Roznica katow.
*/
XE_IMPEX double XE_CALLTYPE XeAngleDifference(double angle1,double angle2);

#endif /* XE_COMPILE_CORE_MATH */

#ifdef XE_COMPILE_PHOTON

namespace Photon
{

/*--- XeMatrixMode() ---*/
/*! Ustala aktywny tryb macierzy.
\param mode Tryb macierzy. Dostepne wartosci:<br>
XE_MATRIX_PROJECTION: Macierz projekcji.<br>
XE_MATRIX_MODELVIEW: Macierz widoku (domyslna).<br>
XE_MATRIX_TEXTURE: Macierz tekstury.
*/
XE_IMPEX void XE_CALLTYPE XeMatrixMode(XE_ESTATE mode);

/*--- XeMatrixIdentity() ---*/
/*! Laduje macierz jednostkowa do aktywnej macierzy (zerujemy transformacje). */
XE_IMPEX void XE_CALLTYPE XeMatrixIdentity();

/*--- XeMatrixPush() ---*/
/*! Odklada bierzaca macierz na stos. */
XE_IMPEX void XE_CALLTYPE XeMatrixPush();

/*--- XeMatrixPop() ---*/
/*! Pobiera macierz ze stosu i oznaczamy jako bierzaca. */
XE_IMPEX void XE_CALLTYPE XeMatrixPop();

/*--- XeMatrixRotate() ---*/
/*! Obraca uklad wspolrzednych.
\param angle kat obrotu.
\param x Ilosc pelnych obrotow angle dla osi X.
\param y Ilosc pelnych obrotow angle dla osi Y.
\param z Ilosc pelnych obrotow angle dla osi Z.
*/
XE_IMPEX void XE_CALLTYPE XeMatrixRotate(double angle,double x,double y,double z);

/*--- XeMatrixScale() ---*/
/*! Skaluje uklad wspolrzednych.
\param x Skala osi X.
\param y Skala osi Y.
\param z Skala osi Z.
*/
XE_IMPEX void XE_CALLTYPE XeMatrixScale(double x,double y,double z);

/*--- XeMatrixTranslate() ---*/
/*! Przesuwamy uklad wspolrzednych.
\param x Wartosc przesuniecia osi X.
\param y Wartosc przesuniecia osi Y.
\param z Wartosc przesuniecia osi Z.
*/
XE_IMPEX void XE_CALLTYPE XeMatrixTranslate(double x,double y,double z);

/*--- XeMatrixLoad() ---*/
/*! Laduje macierz.
\param mat Wskaznik na macierz.
*/
XE_IMPEX void XE_CALLTYPE XeMatrixLoad(double* mat);

/*--- XeMatrixLoadHalf() ---*/
/*! Laduje macierz float.
\param mat Wskaznik na macierz float.
*/
XE_IMPEX void XE_CALLTYPE XeMatrixLoadHalf(float* mat);

/*--- XeMatrixSave() ---*/
/*! Zapisuje macierz do tablicy.
\param type Typ macierzy.
\param mat Wskaznik na macierz.
*/
XE_IMPEX void XE_CALLTYPE XeMatrixSave(XE_ESTATE type,double* mat);

/*--- XeMatrixSaveHalf() ---*/
/*! Zapisuje macierz float do tablicy.
\param type Typ macierzy.
\param mat Wskaznik na macierz float.
*/
XE_IMPEX void XE_CALLTYPE XeMatrixSaveHalf(XE_ESTATE type,float* mat);

} // namespace Photon

#endif /* XE_COMPILE_PHOTON */

/*--- XeClosure() ---*/
/*! Konczy prace silnika. */
XE_IMPEX void XE_CALLTYPE XeClosure();

#ifdef XE_COMPILE_MATH

/*--- XeCryptEncode() ---*/
/*! Szyfruje bufor.
\param buff_src Wskaznik na bufor zrodlowy.
\param buff_dst Wskaznik na bufor docelowy.
\param buff_size Rozmiar (w bajtach) bufora docelowego.
\param key_string Cstring klucza.
\param key_size Dlugosc cstringa klucza.
*/
XE_IMPEX void XE_CALLTYPE XeCryptEncode(void* buff_src,void* buff_dst,unsigned int buff_size,char* key_string,unsigned int key_size);

/*--- XeCryptDecode() ---*/
/*! Deszyfruje bufor.
\param buff_src Wskaznik na bufor zrodlowy.
\param buff_dst Wskaznik na bufor docelowy.
\param buff_size Rozmiar (w bajtach) bufora docelowego.
\param key_string Cstring klucza.
\param key_size Dlugosc cstringa klucza.
*/
XE_IMPEX void XE_CALLTYPE XeCryptDecode(void* buff_src,void* buff_dst,unsigned int buff_size,char* key_string,unsigned int key_size);

/*--- XeCurve() ---*/
/*! Oblicza wartosc krzywej na danej pozycji.
\param pos Pozycja na krzywej (zakres: <0;1>).
\param count Liczba punktow kontrolnych.
\param points Wskaznik na tablice punktow kontrolnych.
\param format Format punktow (ilosc punktow w strukturze).
\param offset Odstep od pierwszego punktu w strukturze.
\return Wartosc krzywej.
*/
XE_IMPEX double XE_CALLTYPE XeCurve(double pos,unsigned int count,double* points,unsigned int format,unsigned int offset);

/*--- XeDataPack() ---*/
/*! Pakuje (kompresuje) dane do nowego buforu i zwraca wskaznik na niego.
\param data Wskaznik na bufor danych surowych.
\param sizeraw Wielkosc danych surowych.
\param sizecomp Referencja na wielkosc danych spakowanych.
\param password Cstring z haslem.
\param passsize Dlugosc hasla.
\return Wskaznik na utworzony bufor.
*/
XE_IMPEX void* XE_CALLTYPE XeDataPack(void* data,unsigned int sizeraw,unsigned int& sizecomp,char* password,unsigned int passsize);

/*--- XeDataUnpack() ---*/
/*! Pakuje (kompresuje) dane do nowego buforu i zwraca wskaznik na niego.
\param data Wskaznik na bufor danych surowych.
\param sizeraw Referencja na wielkosc danych surowych.
\param sizecomp Wielkosc danych spakowanych.
\param password Cstring z haslem.
\param passsize Dlugosc hasla.
\return Wskaznik na utworzony bufor.
*/
XE_IMPEX void* XE_CALLTYPE XeDataUnpack(void* data,unsigned int& sizeraw,unsigned int sizecomp,char* password,unsigned int passsize);

#endif /* XE_COMPILE_MATH */

/*--- XeElementsCount() ---*/
/*! Zwraca ilosc elementow danego typu.
\param element Typ elementu.
\return Ilosc elementow.
*/
XE_IMPEX unsigned int XE_CALLTYPE XeElementsCount(XE_ESTATE element);

/*--- XeElementsFirst() ---*/
/*! Pobiera pierwszy element danego typu.
\param type Typ elementu.
\param elm Wskaznik na element.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeElementsFirst(XE_ESTATE type,void* elm);

/*--- XeElementsLast() ---*/
/*! Pobiera ostatni element danego typu.
\param type Typ elementu.
\param elm Wskaznik na element.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeElementsLast(XE_ESTATE type,void* elm);

#ifdef XE_COMPILE_PHOTON

namespace Photon
{

/*--- XeDrawData() ---*/
/*! Rysuje prymitywy pobierajac ich dane z wielu buforow. Upakowanie danych: coord4;color4;normal3;vertex4;attrib?.
\param mode Typ prymitywow. Dostepne wartosci:<br>
XE_POINTS: Punkty.<br>
XE_LINES: Linie.<br>
XE_LINELOOP: Petla linii.<br>
XE_LINESTRIP: Tasma linii.<br>
XE_TRIANGLES: Trojkaty.<br>
XE_TRIANGLESTRIP: Tasma trojkatow.<br>
XE_TRIANGLEFAN: Wachlaz trojkatow.<br>
XE_QUADS: Czworokaty.<br>
XE_QUADSTRIP: Tasma czworokatow.<br>
XE_POLYGON: Wielokat.
\param first Indeks pierwszego elementu.
\param count Liczba elementow do narysowania.
\param vertex Wskaznik na bufor wierzcholkow (4).
\param color Wskaznik na bufor koloru (4).
\param coord Wskaznik na bufor koordynatow tekstury (4).
\param normal Wskaznik na bufor normalnych (3).
\param indices Wskaznik na bufor z indeksami wierzcholkow.
\param attribs Wskaznik na bufor atrybutow.
\param valtype Typ danych (XE_SINGLE: float; XE_DOUBLE: double).
\param vertoff Przesuniecie danych wierzcholkow.
\param coloff Przesuniecie danych koloru.
\param crdoff Przesuniecie danych koordynatow.
\param normoff Przesuniecie danych normalnych.
\param attoff Przesuniecie danych atrybutow.
\param attloc Indeks zmiennej atrybutow shadera.
\param attcom Liczba komponentow (1-4).
\param attnorm Ustala czy atrybuty maja byc znormalizowane (przyciete do wartosci: <-1;1>).
\param vertexasoffset Ustala czy wskaznik wierzcholkow ma byc uznawany za przesuniecie danych.
\param colorasoffset Ustala czy wskaznik kolorow ma byc uznawany za przesuniecie danych.
\param coordasoffset Ustala czy wskaznik koordynatow ma byc uznawany za przesuniecie danych.
\param normalasoffset Ustala czy wskaznik normalnych ma byc uznawany za przesuniecie danych.
\param indicesasoffset Ustala czy wskaznik indeksow ma byc uznawany za przesuniecie danych.
*/
XE_IMPEX void XE_CALLTYPE XeDrawData(XE_ESTATE mode,int first,int count,void* vertex,void* color,void* coord,void* normal,unsigned int* indices=0,void* attribs=0,XE_ESTATE valtype=XE_SINGLE,int vertoff=0,int coloff=0,int crdoff=0,int normoff=0,int attoff=0,int attloc=0,int attcom=4,bool attnorm=false,bool vertexasoffset=false,bool colorasoffset=false,bool coordasoffset=false,bool normalasoffset=false,bool indicesasoffset=false);

/*--- XeDrawDataStruct() ---*/
/*! Rysuje prymitywy pobierajac ich dane z jednego bufora. Upakowanie danych: coord4;color4;normal3;vertex4;attrib?.
\param mode Typ prymitywow. Dostepne wartosci:<br>
XE_POINTS: Punkty.<br>
XE_LINES: Linie.<br>
XE_LINELOOP: Petla linii.<br>
XE_LINESTRIP: Tasma linii.<br>
XE_TRIANGLES: Trojkaty.<br>
XE_TRIANGLESTRIP: Tasma trojkatow.<br>
XE_TRIANGLEFAN: Wachlaz trojkatow.<br>
XE_QUADS: Czworokaty.<br>
XE_QUADSTRIP: Tasma czworokatow.<br>
XE_POLYGON: Wielokat.
\param first Indeks pierwszego elementu.
\param count Liczba elementow do narysowania.
\param data Wskaznik na bufor danych wierzcholkow.
\param indices Wskaznik na bufor z indeksami wierzcholkow.
\param attribs Wskaznik na bufor atrybutow.
\param off Przesuniecie danych wierzcholkow.
\param attoff Przesuniecie danych atrybutow.
\param attloc Indeks zmiennej atrybutow shadera.
\param attcom Liczba komponentow (1-4).
\param attnorm Ustala czy atrybuty maja byc znormalizowane (przyciete do wartosci: <-1;1>).
*/
XE_IMPEX void XE_CALLTYPE XeDrawDataStruct(XE_ESTATE mode,int first,int count,XE_PERVERTEX* data,unsigned int* indices=0,void* attribs=0,int off=0,int attoff=0,int attloc=0,int attcom=0,bool attnorm=false);

#ifdef XE_COMPILE_CHAOS

/*--- XeDrawSurfaces() ---*/
/*! Rysuje powierzchnie, wedle podanych elementow granicznych.
\param first Pierwszy element. Jesli element jest pusty, to zaczyna od pierwszego elementu powierzchni.
\param last Ostatni element. Jesli element jest pusty, to konczy na ostatnim elemencie powierzchni.
\param usebuff Ustala czy przy rysowaniu ma buforowac dane, aby zaoszczedzic na czasie renderowania.
\param texture Tekstura ktora pokrywa wszystkie powierzchnie, jesli element nie jest pusty.
*/
XE_IMPEX void XE_CALLTYPE XeDrawSurfaces(XE_ELM_SURFACE first=XE_ELM_SURFACE(),XE_ELM_SURFACE last=XE_ELM_SURFACE(),bool usebuff=true,XE_ELM_TEXTURE texture=XE_ELM_TEXTURE());

/*--- XeDrawSurfacesArray() ---*/
/*! Rysuje tablice powierzchni.
\param surfaces Wskaznik na tablice powierzchni.
\param count Liczba elementow tablicy.
\param usebuff Ustala czy przy rysowaniu ma buforowac dane, aby zaoszczedzic na czasie renderowania.
\param texture Tekstura ktora pokrywa wszystkie powierzchnie, jesli element nie jest pusty.
*/
XE_IMPEX void XE_CALLTYPE XeDrawSurfacesArray(XE_ELM_SURFACE* surfaces,unsigned int count,bool usebuff=true,XE_ELM_TEXTURE texture=XE_ELM_TEXTURE());

/*--- XeDrawParticles() ---*/
/*! Rysuje czasteczki, wedle podanych elementow granicznych.
\param first Pierwszy element. Jesli element jest pusty, to zaczyna od pierwszego elementu czasteczek.
\param last Ostatni element. Jesli element jest pusty, to konczy na ostatnim elemencie czasteczek.
\param usebuff Ustala czy przy rysowaniu ma buforowac dane, aby zaoszczedzic na czasie renderowania.
\param smooth Ustala czy czasteczki maja byc smukle.
*/
XE_IMPEX void XE_CALLTYPE XeDrawParticles(XE_ELM_PARTICLE first=XE_ELM_PARTICLE(),XE_ELM_PARTICLE last=XE_ELM_PARTICLE(),bool usebuff=true,bool smooth=false);

/*--- XeDrawParticlesArray() ---*/
/*! Rysuje tablice czasteczek.
\param particles Wskaznik na tablice czasteczek.
\param count Liczba elementow tablicy.
\param usebuff Ustala czy przy rysowaniu ma buforowac dane, aby zaoszczedzic na czasie renderowania.
\param smooth Ustala czy czasteczki maja byc smukle.
*/
XE_IMPEX void XE_CALLTYPE XeDrawParticlesArray(XE_ELM_PARTICLE* particles,unsigned int count,bool usebuff=true,bool smooth=false);

#endif /* XE_COMPILE_CHAOS */

/*--- XeGenerateFont() ---*/
/*! Generuje czcionke systemowa do podanej tablicy 256 istniejacych list wyswietlania.
\param elm Wskaznik na element listy wyswietlania do ktorego zwroci dane.
\param fontdata Dane czcionki. Pusty wskaznik spowoduje utworzenie standardowa czcionke 16x16 znakow.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeGenerateFont(XE_ELM_DISPLAYLIST* elm,XE_FONTMAP* fontdata);

/*--- XeDrawFont() ---*/
/*! Rysuje tekst wybrana czcionka.
\param fontlists Element listy wyswietlania z danymi czcionki.
\param text Tekst do wyrenderowania.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeDrawFont(XE_ELM_DISPLAYLIST fontlists,char* text);

} // namespace Photon

#endif /* XE_COMPILE_PHOTON */

#ifdef XE_COMPILE_TIMEBRIDGE

namespace TimeBridge
{

/*--- XeRandomize() ---*/
/*! Losuje wartosc liczbowa z przedzialu <0;1>.
\param time Czas.
\param seed Ziarno.
\return Wylosowana wartosc liczbowa.
*/
XE_IMPEX double XE_CALLTYPE XeRandomize(double time, double seed=0);

#ifdef XE_COMPILE_TIMEBRIDGE_ALLOW_TO_SUBSTANCES_TRAVELING

/*--- XeAccelerateSubstance() ---*/
/*! Nadaje substancji przyspieszenie czasowe.
\param elm Element substancji.
\param timerel Przyspieszenie czasowe.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeAccelerateSubstance(XE_ELM_SUBSTANCE elm,double timerel);

#endif /* XE_COMPILE_TIMEBRIDGE_ALLOW_TO_SUBSTANCES_TRAVELING */

} // namespace TimeBridge

#endif /* XE_COMPILE_TIMEBRIDGE */

#ifdef XE_COMPILE_PSYCHE

namespace Psyche
{

/*--- XePsycheCreate() ---*/
/*! Tworzy umysl.
\param elm Wskaznik na element umyslu.
*/
XE_IMPEX void XE_CALLTYPE XePsycheCreate(XE_ELM_PSYCHE* elm);

/*--- XePsycheProgress() ---*/
/*! Wykonuje postep umyslu.
\param index Element umyslu.
\param factor Wspolczynnik predkosci sygnalow.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XePsycheProgress(XE_ELM_PSYCHE index,float factor);

/*--- XePsycheSave() ---*/
/*! Zapisuje umysl do pliku.
\param index Element umyslu.
\param fname Sciezka do pliku.
\param savesignals Ustala czy maja byc zapisane sygnaly.
\param savedetsignals Ustala czy maja byc zapisane sygnaly przechowywane w detektorach.
\param saveeffsignals Ustala czy maja byc zapisane sygnaly przechowywane w efektorach.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XePsycheSave(XE_ELM_PSYCHE index,char* fname,bool savesignals=false,bool savedetsignals=false,bool saveeffsignals=false);

/*--- XePsycheLoad() ---*/
/*! Laduje umysl z pliku.
\param index Element umyslu.
\param fname Sciezka do pliku.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XePsycheLoad(XE_ELM_PSYCHE index,char* fname);

/*--- XePsycheAdd() ---*/
/*! Dodaje komorke do umyslu.
\param index Element umyslu.
\param unit Wskaznik na uniwersalny element komorki.
\param type Typ komorki. Dostepne wartosci:<br>
XE_PSYCHE_NEURON: Neuron.<br>
XE_PSYCHE_DETECTOR: Detektor.<br>
XE_PSYCHE_EFFECTOR: Efektor.
\param value Wartosc poczatkowa (dla neuronu to wartosc bramy, dla detektora lub efektora to wartosc sygnalu).
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XePsycheAdd(XE_ELM_PSYCHE index,XE_ELM_UNIFORM* unit,XE_ESTATE type,XE_DWORD value=0);

/*--- XePsycheRemove() ---*/
/*! Usuwa komorke z umyslu.
\param index Element umyslu.
\param unit Uniwersalny element komorki.
\param type Typ komorki. Dostepne wartosci:<br>
XE_PSYCHE_NEURON: Neuron.<br>
XE_PSYCHE_DETECTOR: Detektor.<br>
XE_PSYCHE_EFFECTOR: Efektor.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XePsycheRemove(XE_ELM_PSYCHE index,XE_ELM_UNIFORM unit,XE_ESTATE type);

/*--- XePsycheLink() ---*/
/*! Laczy komorke umyslu z inna komorka (tworzy synapse).
\param unit Uniwersalny element komorki ktorej dodamy synapse.
\param type Typ komorki do ktorej podlaczymy synapse. Dostepne wartosci:<br>
XE_PSYCHE_NEURON: Neuron.<br>
XE_PSYCHE_EFFECTOR: Efektor.
\param cell Uniwersalny element komorki do ktorego podlaczymy synapse.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XePsycheLink(XE_ELM_UNIFORM unit,XE_ESTATE type,XE_ELM_UNIFORM cell);

/*--- XePsycheUnlink() ---*/
/*! Odlacza komorke umyslu od innej komorki (usuwa synapse).
\param unit Uniwersalny element komorki ktorej usuniemy synapse.
\param type Typ komorki od ktorej odlaczymy synapse. Dostepne wartosci:<br>
XE_PSYCHE_NEURON: Neuron.<br>
XE_PSYCHE_EFFECTOR: Efektor.
\param cell Uniwersalny element komorki od ktorej odlaczymy synapse.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XePsycheUnlink(XE_ELM_UNIFORM unit,XE_ESTATE type,XE_ELM_UNIFORM cell);

/*--- XePsycheSet() ---*/
/*! Ustala wartosc stanu komorki umyslu.
\param unit Uniwersalny element komorki.
\param type Typ zmiennej stanu. Dostepne wartosci:<br>
XE_PSYCHE_NEURON_LENGTH: Dlugosc neuronu [float].<br>
XE_PSYCHE_NEURON_GATE: Brama neuronu [XE_DWORD].<br>
XE_PSYCHE_NEURON_POSITION: Pozycja neuronu [XE_HALFVECTOR].<br>
XE_PSYCHE_DETECTOR_SIGNAL: Sygnal detektora [XE_DWORD].<br>
XE_PSYCHE_DETECTOR_POSITION: Pozycja detektora [XE_HALFVECTOR].<br>
XE_PSYCHE_EFFECTOR_SIGNAL: Sygnal efektora [XE_DWORD].<br>
XE_PSYCHE_EFFECTOR_POSITION: Pozycja efektora [XE_HALFVECTOR].<br>
XE_PSYCHE_SIGNAL_SIGNAL: Wartosc sygnalu [XE_DWORD].
\param ... Wartosci do przeslania.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XePsycheSet(XE_ELM_UNIFORM unit,XE_ESTATE type,...);

/*--- XePsycheGet() ---*/
/*! Pobiera wartosc stanu komorki umyslu.
\param unit Uniwersalny element komorki.
\param type Typ zmiennej stanu. Dostepne wartosci:<br>
XE_PSYCHE_NEURON_LENGTH: Dlugosc neuronu [float].<br>
XE_PSYCHE_NEURON_GATE: Brama neuronu [XE_DWORD].<br>
XE_PSYCHE_NEURON_POSITION: Pozycja neuronu [XE_HALFVECTOR].<br>
XE_PSYCHE_DETECTOR_SIGNAL: Sygnal detektora [XE_DWORD].<br>
XE_PSYCHE_DETECTOR_POSITION: Pozycja detektora [XE_HALFVECTOR].<br>
XE_PSYCHE_EFFECTOR_SIGNAL: Sygnal efektora [XE_DWORD].<br>
XE_PSYCHE_EFFECTOR_POSITION: Pozycja efektora [XE_HALFVECTOR].<br>
XE_PSYCHE_SIGNAL_SIGNAL: Wartosc sygnalu [XE_DWORD].<br>
XE_PSYCHE_SIGNAL_POSITION: Pozycja sygnalu <0;?> [float].<br>
XE_PSYCHE_SYNAPSIS_GATE: Brama komorki [XE_DWORD].<br>
XE_PSYCHE_SYNAPSIS_POSITION: Pozycja komorki [XE_HALFVECTOR].
\return Wskaznik nieokreslony na zmienna stanu komorki. Nalezy rzutowac na konkretny typ danej zmiennej stanu.
*/
XE_IMPEX void* XE_CALLTYPE XePsycheGet(XE_ELM_UNIFORM unit,XE_ESTATE type);

/*--- XePsycheSignalNeuronInfo() ---*/
/*! Pobiera opis neuronu w ktorym przebywa dany sygnal.
\param unit Uniwersalny element komorki sygnalu.
\param info Wskaznik na obiekt struktury opisu neuronu.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XePsycheSignalNeuronInfo(XE_ELM_UNIFORM unit,XE_PSYCHE_SIGNAL_NEURON_INFO* info);

/*--- XePsycheElement() ---*/
/*! Dokonuje danej operacji na uniwerslanym elemencie komorki.
\param index Element umyslu.
\param unit Wskaznik na uniwersalny element komorki.
\param type Typ komorki i akcji. Dostepne wartosci:<br>
XE_PSYCHE_NEURON_FIRST: Pobiera pierwszy neuron.<br>
XE_PSYCHE_NEURON_LAST: Pobiera ostatni neuron.<br>
XE_PSYCHE_NEURON_NEXT: Inkrementuje neuron.<br>
XE_PSYCHE_NEURON_PREV: Dekrementuje neuron.<br>
XE_PSYCHE_DETECTOR_FIRST: Pobiera pierwszy detektor.<br>
XE_PSYCHE_DETECTOR_LAST: Pobiera ostatni detektor.<br>
XE_PSYCHE_DETECTOR_NEXT: Inkrementuje detektor.<br>
XE_PSYCHE_DETECTOR_PREV: Dekrementuje detektor.<br>
XE_PSYCHE_EFFECTOR_FIRST: Pobiera pierwszy efektor.<br>
XE_PSYCHE_EFFECTOR_LAST: Pobiera ostatni efektor.<br>
XE_PSYCHE_EFFECTOR_NEXT: Inkrementuje efektor.<br>
XE_PSYCHE_EFFECTOR_PREV: Dekrementuje efektor.<br>
XE_PSYCHE_SIGNAL_FIRST: Pobiera pierwszy sygnal.<br>
XE_PSYCHE_SIGNAL_LAST: Pobiera ostatni sygnal.<br>
XE_PSYCHE_SIGNAL_NEXT: Inkrementuje sygnal.<br>
XE_PSYCHE_SIGNAL_PREV: Dekrementuje sygnal.<br>
*/
XE_IMPEX bool XE_CALLTYPE XePsycheElement(XE_ELM_PSYCHE index,XE_ELM_UNIFORM* unit,XE_ESTATE type);

/*--- XePsycheSynapsis() ---*/
/*! Pobiera uniwersalny element komorki z danej synapsy.
\param unit Uniwersalny element komorki.
\param type Typ komorki z ktorej pobieramy element. Dostepne wartosci:<br>
XE_PSYCHE_NEURON_FIRST: Pierwszy neuron.<br>
XE_PSYCHE_NEURON_LAST: Ostatni neuron.<br>
XE_PSYCHE_DETECTOR_FIRST: Pierwszy detektor.<br>
XE_PSYCHE_DETECTOR_LAST: Ostatni detektor.<br>
XE_PSYCHE_SYNAPSIS_FIRST: Pierwszy neuron.<br>
XE_PSYCHE_SYNAPSIS_LAST: Ostatni neuron.<br>
XE_PSYCHE_SYNAPSIS_NEXT: Nastepna synapsa.<br>
XE_PSYCHE_SYNAPSIS_PREV: Poprzednia synapsa.
\param cell Wskaznik na uniwersalny element komorki do ktorego zapiszemy wynik.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XePsycheSynapsis(XE_ELM_UNIFORM unit,XE_ESTATE type,XE_ELM_UNIFORM* cell);

/*--- XePsycheSynapsisCount() ---*/
/*! Zwraca ilosc synaps komorki umyslu danego typu.
\param unit Uniwersalny element komorki.
\param type Typ komorki z ktorej pobieramy element. Dostepne wartosci:<br>
XE_PSYCHE_NEURON: Neuron.<br>
XE_PSYCHE_DETECTOR: Detektor.<br>
XE_PSYCHE_SYNAPSIS: Synapsa.
\return Ilosc synaps.
*/
XE_IMPEX unsigned int XE_CALLTYPE XePsycheSynapsisCount(XE_ELM_UNIFORM unit,XE_ESTATE type);

/*--- XePsycheCount() ---*/
/*! Zwraca ilosc komorek umyslu danego typu.
\param index Element umyslu.
\param type Typ komorki. Dostepne wartosci:<br>
XE_PSYCHE_NEURON: Neuron.<br>
XE_PSYCHE_DETECTOR: Detektor.<br>
XE_PSYCHE_EFFECTOR: Efektor.<br>
XE_PSYCHE_SIGNAL: Sygnal.<br>
XE_PSYCHE: Wszystkie.
\return Ilosc komorek.
*/
XE_IMPEX unsigned int XE_CALLTYPE XePsycheCount(XE_ELM_PSYCHE index,XE_ESTATE type);

/*--- XePsycheSend() ---*/
/*! Wysyla sygnal do detektora.
\param unit Uniwersalny element komorki.
\param signal Wartosc sygnalu.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XePsycheSend(XE_ELM_UNIFORM unit,XE_DWORD signal);

/*--- XePsycheReceive() ---*/
/*! Odbiera sygnal od efektora.
\param unit Uniwersalny element komorki.
\return Wartosc sygnalu.
*/
XE_IMPEX XE_DWORD XE_CALLTYPE XePsycheReceive(XE_ELM_UNIFORM unit);

/*--- XePsycheDraw() ---*/
/*! Rysuje umysl uwzgledniajac opis rysowania.
\param index Element umyslu.
\param desc Wskaznik na opis rysowania umyslu.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XePsycheDraw(XE_ELM_PSYCHE index,XE_PSYCHE_DRAW_DESC* desc);

/*--- XePsycheKillSignal() ---*/
/*! Zabija dany sygnal.
\param index Element umyslu.
\param unit Uniwersalny element komorki sygnalu.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XePsycheKillSignal(XE_ELM_PSYCHE index,XE_ELM_UNIFORM unit);

/*--- XePsycheKillSignals() ---*/
/*! Zabija wszystkie sygnaly.
\param index Element umyslu.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XePsycheKillSignals(XE_ELM_PSYCHE index);

/*--- XePsycheClear() ---*/
/*! Niszczy wszystkie komorki danego typu.
\param index Element umyslu.
\param type Typ komorki. Dostepne wartosci:<br>
XE_PSYCHE_NEURON: Neuron.<br>
XE_PSYCHE_DETECTOR: Detektor.<br>
XE_PSYCHE_EFFECTOR: Efektor.<br>
XE_PSYCHE_SIGNAL: Sygnal.<br>
XE_PSYCHE: Wszystkie.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XePsycheClear(XE_ELM_PSYCHE index,XE_ESTATE type);

/*--- XePsycheDestroy() ---*/
/*! Niszczy umysl.
\param index Element umyslu.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XePsycheDestroy(XE_ELM_PSYCHE index);

} /* namespace Psyche */

#endif /* XE_COMPILE_PSYCHE */

#ifdef XE_COMPILE_CORE_DNA

/*--- XeDnaCreate() ---*/
/*! Tworzy cialo DNA.
\param elm Wskaznik na element DNA. Zwraca do niego utworzony element jesli operacja powiodla sie.
*/
XE_IMPEX void XE_CALLTYPE XeDnaCreate(XE_ELM_DNA* elm);

/*--- XeDnaBirth() ---*/
/*! Narodziny (przygotowanie) DNA.
\param index Element DNA.
\param cells Ilosc komorek (substancji).
\param muscles Ilosc miesni (sil kowalencyjnych).
\param skin Ilosc pokrycia skory (powierzchni).
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeDnaBirth(XE_ELM_DNA index,unsigned int cells,unsigned int muscles,unsigned int skin);

/*--- XeDnaDie() ---*/
/*! Smierc (wyczyszczenie) DNA.
\param index Element DNA.
\param destroybody Ustala czy poza odlaczeniem ma tez niszczyc elementy.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeDnaDie(XE_ELM_DNA index,bool destroybody=true);

/*--- XeDnaDraw() ---*/
/*! Rysuje cialo DNA.
\param index Element DNA.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeDnaDraw(XE_ELM_DNA index);

/*--- XeDnaLink() ---*/
/*! Laczy element z cialem DNA.
\param index Element DNA.
\param type Typ elementu. Dostepne wartosci:<br>
XE_DNA_BRAIN: Element umyslu.<br>
XE_DNA_CELLS: Element substancji.<br>
XE_DNA_MUSCLES: Element sily kowalencyjnej.<br>
XE_DNA_SKIN: Element powierzchni.<br>
XE_DNA_BODY_TEXTURE: Element tekstury.
\param id Indeks elementu w ciele DNA.
\param element Wskaznik na element.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeDnaLink(XE_ELM_DNA index,XE_ESTATE type,unsigned int id,void* element);

/*--- XeDnaUnlink() ---*/
/*! Odlacza element od ciala DNA.
\param index Element DNA.
\param type Typ elementu. Dostepne wartosci:<br>
XE_DNA_BRAIN: Element umyslu.<br>
XE_DNA_CELLS: Element substancji.<br>
XE_DNA_MUSCLES: Element sily kowalencyjnej.<br>
XE_DNA_SKIN: Element powierzchni.<br>
XE_DNA_BODY_TEXTURE: Element tekstury.
\param id Indeks elementu w ciele DNA.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeDnaUnlink(XE_ELM_DNA index,XE_ESTATE type,unsigned int id);

/*--- XeDnaGet() ---*/
/*! Zwraca wskaznik na element ciala DNA.
\param index Element DNA.
\param type Typ elementu. Dostepne wartosci:<br>
XE_DNA_BRAIN: Element umyslu.<br>
XE_DNA_CELLS: Element substancji.<br>
XE_DNA_MUSCLES: Element sily kowalencyjnej.<br>
XE_DNA_SKIN: Element powierzchni.<br>
XE_DNA_BODY_TEXTURE: Element tekstury.
\param id Indeks elementu w ciele DNA.
\return Wskaznik na dany element.
*/
XE_IMPEX void* XE_CALLTYPE XeDnaGet(XE_ELM_DNA index,XE_ESTATE type,unsigned int id=0);

/*--- XeDnaCount() ---*/
/*! Zwraca liczbe elementow danego typu ciala DNA.
\param index Element DNA.
\param type Typ elementu. Dostepne wartosci:<br>
XE_DNA_BRAIN: Element umyslu.<br>
XE_DNA_CELLS: Element substancji.<br>
XE_DNA_MUSCLES: Element sily kowalencyjnej.<br>
XE_DNA_SKIN: Element powierzchni.<br>
XE_DNA_BODY_TEXTURE: Element tekstury.
\return Ilosc elementow.
*/
XE_IMPEX unsigned int XE_CALLTYPE XeDnaCount(XE_ELM_DNA index,XE_ESTATE type);

/*--- XeDnaDestroy() ---*/
/*! Niszczy cialo DNA.
\param index Element DNA.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeDnaDestroy(XE_ELM_DNA index);

#endif /* XE_COMPILE_CORE_DNA */

#ifdef XE_COMPILE_ETHER

namespace Ether
{

/*--- XeSocketInitiation() ---*/
/*! Inicjuje tryb sieciowy.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeSocketInitiation();

/*--- XeSocketClosure() ---*/
/*! Zamyka tryb sieciowy. */
XE_IMPEX void XE_CALLTYPE XeSocketClosure();

/*--- XeSocketCreate() ---*/
/*! Tworzy gniazdo sieciowe.
\param elm Wskaznik na element gniazda.
*/
XE_IMPEX void XE_CALLTYPE XeSocketCreate(XE_ELM_SOCKET* elm);

/*--- XeSocketOpen() ---*/
/*! Otwiera gniazdo sieciowe.
\param index Element gniazda.
\param udpmode Ustala czy gniazdo ma pracowac w trybie UDP.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeSocketOpen(XE_ELM_SOCKET index,bool udpmode=false);

/*--- XeSocketClose() ---*/
/*! Zamyka gniazdo sieciowe.
\param index Element gniazda.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeSocketClose(XE_ELM_SOCKET index);

/*--- XeSocketShutdown() ---*/
/*! Wylacza wysylanie/odbieranie na danym gniezdzie sieciowym. Operacja bezpowrotna!
\param index Element gniazda.
\param sending Ustala czy ma byc wylaczona mozliwosc wysylania danych.
\param receiving Ustala czy ma byc wylaczona mozliwosc odbierania danych danych.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeSocketShutdown(XE_ELM_SOCKET index,bool sending,bool receiving);

/*--- XeSocketConnect() ---*/
/*! Laczy sie z hostem.
\param index Element gniazda.
\param address Adres hosta.
\param port Numer portu.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeSocketConnect(XE_ELM_SOCKET index,char* address,XE_WORD port);

/*--- XeSocketListen() ---*/
/*! Wlacza nasluchiwanie na danym porcie.
\param index Element gniazda.
\param port Numer portu.
\param maxconn Maksymalna ilosc aktywnych polaczen.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeSocketListen(XE_ELM_SOCKET index,XE_WORD port,int maxconn);

/*--- XeSocketAccept() ---*/
/*! Akceptuje nadchodzace polaczenie i przypisuje sobie jego gniazdo.
\param index Element gniazda.
\param listen Element gniazda nasluchujacego.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeSocketAccept(XE_ELM_SOCKET index,XE_ELM_SOCKET listen);

/*--- XeSocketSend() ---*/
/*! Wysyla dane do danego gniazda.
\param index Element gniazda.
\param buffer Wskaznik na bufor (plik) danych.
\param sizeheader Ustala czy ma wyslac pakiet z naglowkiem zawierajacym rozmiar pakietu.
\param offset Pozycja od ktorej zaczyna wysylac dane bufora.
\param size Rozmiar wysylanych danych (0 jesli ma wyslac caly bufor).
\return Rozmiar wyslanych danych jesli operacja powiodla sie, lub 0 jesli nie.
*/
XE_IMPEX unsigned int XE_CALLTYPE XeSocketSend(XE_ELM_SOCKET index,XE_FILE* buffer,bool sizeheader=true,unsigned int offset=0,unsigned int size=0);

/*--- XeSocketReceive() ---*/
/*! Odbiera dane z danego gniazda.
\param index Element gniazda. Jesli jest pusty, to czysci bufor do ktorego wczesniej zostaly odebrane dane z gniazda.
\param buffer Wskaznik na bufor (plik) danych.
\param size Odbiera dane o podanym rozmiarze jesli wiekszy niz 0, inaczej wykorzystuje rozmiar w naglowku pakietu.
\param peekmsg Ustala czy dane po pobraniu maja byc pozostawione w kolejce wiadomosci gniazda.
\return Rozmiar odebranych danych jesli operacja powiodla sie, lub 0 jesli nie.
*/
XE_IMPEX unsigned int XE_CALLTYPE XeSocketReceive(XE_ELM_SOCKET index,XE_FILE* buffer,unsigned int size=0,bool peekmsg=false);

/*--- XeSocketAsync() ---*/
/*! Asynchroniczny (nieblokowany) tryb gniazda (domyslnie wylaczony).
\param index Element gniazda.
\param mode Ustala tryb synchronizacji.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeSocketAsync(XE_ELM_SOCKET index,bool mode);

/*--- XeSocketNagle() ---*/
/*! Aktywacja/dezaktywacja algorytmu Nagle (ogranicza liczbe malych pakietow wysylanych przez hosta, przez buforowanie danych, do momentu, az osiagna one naturalny rozmiar i pakiet zostanie wyslany; domyslnie wlaczony).
\param index Element gniazda.
\param mode Ustala tryb algorytmu Nagle.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeSocketNagle(XE_ELM_SOCKET index,bool mode);

/*--- XeSocketIsOpened() ---*/
/*! Sprawdza czy gniazdo jest otwarte.
\param index Element gniazda.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeSocketIsOpened(XE_ELM_SOCKET index);

/*--- XeSocketIsConnected() ---*/
/*! Sprawdza czy gniazdo jest polaczone.
\param index Element gniazda.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeSocketIsConnected(XE_ELM_SOCKET index);

/*--- XeSocketIsUDP() ---*/
/*! Sprawdza czy gniazdo pracuje w trybie UDP.
\param index Element gniazda.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeSocketIsUDP(XE_ELM_SOCKET index);

/*--- XeSocketSetUDPbuffSize() ---*/
/*! Ustala maksymalny rozmiar wysylanych/odbieranych danych w trybie UDP.
\param index Element gniazda.
\param size Rozmiar (w bajtach).
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeSocketSetUDPbuffSize(XE_ELM_SOCKET index,unsigned int size);

/*--- XeSocketGetUDPbuffSize() ---*/
/*! Zwraca maksymalny rozmiar danych wysylanych/odbieranych w trybie UDP.
\param index Element gniazda.
\return Rozmiar danych (w bajtach).
*/
XE_IMPEX unsigned int XE_CALLTYPE XeSocketGetUDPbuffSize(XE_ELM_SOCKET index);

/*--- XeSocketGetIp() ---*/
/*! Zwraca IP gniazda.
\param index Element gniazda.
\return IP gniazda jesli operacja powiodla sie, lub 0 jesli nie.
*/
XE_IMPEX XE_DWORD XE_CALLTYPE XeSocketGetIp(XE_ELM_SOCKET index);

/*--- XeSocketGetPort() ---*/
/*! Zwraca port gniazda.
\param index Element gniazda.
\return Port gniazda jesli operacja powiodla sie, lub 0 jesli nie.
*/
XE_IMPEX XE_WORD XE_CALLTYPE XeSocketGetPort(XE_ELM_SOCKET index);

/*--- XeSocketDestroy() ---*/
/*! Niszczy gniazdo sieciowe.
\param index Element gniazda.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeSocketDestroy(XE_ELM_SOCKET index);

/*--- XeNetworkIsConnected() ---*/
/*! Sprawdza czy polaczenie z internetem jest aktywne.
\return true jesli operacja powiodla sie.
*/
XE_IMPEX bool XE_CALLTYPE XeNetworkIsConnected();

/*--- XeNetworkHostName() ---*/
/*! Pobiera nazwe lokalnego hosta.
\param result Referencja na wynikowy string z nazwa.
*/
XE_IMPEX void XE_CALLTYPE XeNetworkHostName(XE_STRING& result);

/*--- XeNetworkIpToString() ---*/
/*! Konwertuje adres IP do stringa.
\param ip Adres IP.
\param result Referencja na wynikowy string adresu.
*/
XE_IMPEX void XE_CALLTYPE XeNetworkIpToString(XE_DWORD ip,XE_STRING& result);

/*--- XeNetworkStringToIp() ---*/
/*! Konwertuje string do adresu IP.
\param ip String z adresem IP.
\return Adres IP.
*/
XE_IMPEX XE_DWORD XE_CALLTYPE XeNetworkStringToIp(char* ip);

/*--- XeNetworkAddressToIp() ---*/
/*! Konwertuje string z adresem hosta do adresu IP.
\param address String z adresem hosta.
\return Adres IP.
*/
XE_IMPEX XE_DWORD XE_CALLTYPE XeNetworkAddressToIp(char* address);

} /* przestrzen nazw modulu Ether */

#endif /* XE_COMPILE_ETHER */

#endif /* XE_NOT_USE_LIBRARY */

} // namespace XeCore

#endif /* _XENON_CORE_H_ */
